/* mzttowav.c : convert mzt format to wav format for MZ series */
/*   by AKIKAWA, Hisashi 2025.7.21 version                     */
/* This software is released under the MIT license.            */
/*  (https://opensource.org/license/mit)                       */

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

size_t dataout(uint8_t data, FILE *fp);
size_t pwmbit(uint8_t bit, FILE *fp);
size_t pwm(double length, FILE *fp);
size_t blank(double length, FILE *fp);
void wav_head(uint32_t size, FILE *fp);
void fput2(uint16_t data, FILE *fp);
void fput4(uint32_t data, FILE *fp);

/* wav file parameter */
uint32_t sampling_rate;
uint8_t quantization_bit;
uint8_t channels;

/* tape parameter */
double width_short;
double time = 0.;
uint8_t stop_bit;

int main(int argc, char *argv[])
{
  uint32_t i, j;
  uint32_t opt;
  int c;
  int with_info = 1;
  uint16_t sum;
  size_t size = 0;
  FILE *fp_in, *fp_out;

  /* default parameters */
  sampling_rate = 24000; /* [Hz] */
  width_short = 250e-6; /* [sec] */
  quantization_bit = 8;
  channels = 1;
  stop_bit = 0;

  /* options */
  for (opt = 1; argc > opt && argv[opt][0] == '-'; opt++) {
    if (strcmp(argv[opt], "-r") == 0) {
      if (opt + 1 >= argc) break;
      sampling_rate = atoll(argv[++opt]);
      if (sampling_rate < 1) {
	printf("illegal sampling rate\n");
	exit(1);
      }
    } else if (strcmp(argv[opt], "-q") == 0) {
      if (opt + 1 >= argc) break;
      quantization_bit = atoi(argv[++opt]);
      if (quantization_bit != 8 && quantization_bit != 16) {
	printf("sampling bit must be 8 or 16\n");
	exit(1);
      }
    } else if (strcmp(argv[opt], "-c") == 0) {
      if (opt + 1 >= argc) break;
      channels = atoi(argv[++opt]);
      if (channels != 1 && channels != 2) {
	printf("the number of channels must be 1 or 2\n");
	exit(1);
      }
    } else if (strcmp(argv[opt], "-w") == 0) {
      if (opt + 1 >= argc) break;
      width_short = atoi(argv[++opt]) * 1e-6;
    } else if (strcmp(argv[opt], "-s") == 0) {
      if (opt + 1 >= argc) break;
      stop_bit = atoi(argv[++opt]);
      if (stop_bit < 0) {
	printf("illegal stop bit\n");
	exit(1);
      }
    } else if (strcmp(argv[opt], "-d") == 0) {
      with_info = 0;
    }
  }
  if (sampling_rate < 1. / width_short) {
    printf("illegal sampling rate\n");
    exit(1);
  }

  if (argc != opt + 2) {
    printf("usage: mzttowav [options] infile.mzt outfile.wav\n");
    printf("options:\n");
    printf(" -r sampling-rate[Hz]\n");
    printf(" -q quantization-bits\n");
    printf(" -c channels\n");
    printf(" -w short-pulse-width[usec]\n");
    printf(" -s stop-bits\n");
    printf(" -d\n");
    printf("default: -r %d -q %d -c %d -w %3.0f -s %d\n",
	   sampling_rate,
	   quantization_bit,
	   channels,
	   width_short * 1e6,
	   stop_bit
	   );
    exit(1);
  }

  /* input file */
  if (strcmp(argv[argc - 2], "-") == 0) {
    fp_in = stdin;
  } else {
    fp_in = fopen(argv[argc - 2], "rb");  /* b for Windows */
    if (fp_in == NULL) {
      printf("cannot open %s\n", argv[argc - 2]);
      exit(1);
    }
  }

  /* output file */
  if (strcmp(argv[argc - 1], "-") == 0) {
    fp_out = stdout;
  } else {
    fp_out = fopen(argv[argc - 1], "wb");  /* b for Windows */
    if (fp_out == NULL) {
      printf("cannot open %s\n", argv[argc - 1]);
      exit(1);
    }
  }
  
  /* make wav data */
  for (i = 0; i < 44; i++) {
    fputc(0, fp_out);
  }

  /* information block */
  //  size += blank(1., fp_out);
  if (with_info) {
    for (i = 0; i < 22000; i++) {
      size += pwmbit(0, fp_out);
    }
    for (i = 0; i < 40; i++) {
      size += pwmbit(1, fp_out);
    }
    for (i = 0; i < 40; i++) {
      size += pwmbit(0, fp_out);
    }
    size += pwmbit(1, fp_out);
    sum = 0;
    for (i = 0; i < 128; i++) {
      c = fgetc(fp_in);
      if (c == EOF) {
	printf("too short information block\n");
	exit(1);
      }
      size += dataout((uint8_t)c, fp_out);
      for (j = 0; j < 8; j++) {
	sum += ((uint8_t)c >> j) & 1;
      }
    }
    size += dataout(sum >> 8, fp_out);
    size += dataout(sum & 0xff, fp_out);
    size += pwmbit(1, fp_out);
  }

  /* data block */
  for (i = 0; i < 11000; i++) {
    size += pwmbit(0, fp_out);
  }
  for (i = 0; i < 20; i++) {
    size += pwmbit(1, fp_out);
  }
  for (i = 0; i < 20; i++) {
    size += pwmbit(0, fp_out);
  }
  size += pwmbit(1, fp_out);
  sum = 0;
  while ((c = fgetc(fp_in)) != EOF) {
    size += dataout(c, fp_out);
    for (j = 0; j < 8; j++) {
      sum += (c >> j) & 1;
    }
  }
  size += dataout(sum >> 8, fp_out);
  size += dataout(sum & 0xff, fp_out);
  size += pwmbit(1, fp_out);
  //  size += blank(0.5, fp_out);

  /* wav header */
  wav_head(size, fp_out);

  fclose(fp_in);
  fclose(fp_out);
  return 0;
}

size_t dataout(uint8_t data, FILE *fp)
{
  uint32_t i, size;

  /* start bit = 1 */
  size = pwmbit(1, fp);

  /* data */
  for (i = 0; i < 8; i++) {
    size += pwmbit(data & (0x80 >> i), fp);
  }

  /* stop bit = 0 if necessary */
  for (i = 0; i < stop_bit; i++) {
    size += pwmbit(0, fp);
  }

  return size;
}

size_t pwmbit(uint8_t bit, FILE *fp) {
  if (bit) {
    return pwm(width_short * 2, fp);
  } else {
    return pwm(width_short, fp);
  }
}

size_t pwm(double width, FILE *fp)
{
  /* width[usec] */
  uint32_t i, j;
  uint32_t size;
  double start = (int)(time * sampling_rate) / (double)sampling_rate;
  uint8_t *data;

  data = malloc ((sampling_rate * width * 2 + 2) * channels * quantization_bit / 8);
  if (data == NULL) {
    printf("cannot allocate memory\n");
    exit(1);
  }

  for (i = 0; time < start + width; i++, time += 1. / sampling_rate) {
    for (j = 0; j < channels; j++) {
      if (quantization_bit == 8) {
	data[i * channels + j] = 0xff;
      } else {
	data[(i * channels + j) * 2] = 0xff;
	data[(i * channels + j) * 2 + 1] = 0x7f;
      }
    }
  }
  for (; time < start + width * 2; i++, time += 1. / sampling_rate) {
    for (j = 0; j < channels; j++) {
      if (quantization_bit == 8) {
	data[i * channels + j] = 0x00;
      } else {
	data[(i * channels + j) * 2] = 0x00;
	data[(i * channels + j) * 2 + 1] = 0x80;
      }
    }
  }
  size = i * channels * quantization_bit / 8;
  fwrite(data, 1, size, fp);
  free(data);
  return size;
}

size_t blank(double period, FILE *fp)
{
  uint32_t i, j;
  for (i = 0; i < period * sampling_rate; i++) {
    for (j = 0; j < channels; j++) {
      if (quantization_bit == 8) {
	fputc(0x80, fp);
      } else {
	fput2(0, fp);
      }
    }
  }
  time += (int) (period * sampling_rate) / sampling_rate;
  return i * channels * quantization_bit / 8;
}

void wav_head(uint32_t size, FILE *fp)
{
  rewind(fp);

  /* RIFF identifier */
  fputs("RIFF", fp);

  /* file size */
  fput4(size + 36, fp);

  /* WAVE identifier */
  fputs("WAVE", fp);

  /* fmt chunk start */
  fputs("fmt ", fp);

  /* fmt chunk size */
  fput4(16, fp);

  /* format ID */
  fput2(1, fp);

  /* monoaural or streo */
  fput2(channels, fp);

  /* sampling rate */
  fput4(sampling_rate, fp);

  /* data rate */
  fput4(sampling_rate * channels * quantization_bit / 8, fp);

  /* block size */
  fput2(channels * quantization_bit / 8, fp);

  /* sampling bit */
  fput2(quantization_bit, fp);

  /* data chunk start */
  fputs("data", fp);

  /* data size */
  fput4(size, fp);
}

void fput2(uint16_t data, FILE *fp)
{
  fputc(data & 0xff, fp);
  fputc((data >> 8) & 0xff, fp);
}

void fput4(uint32_t data, FILE *fp)
{
  fputc(data & 0xff, fp);
  fputc((data >> 8) & 0xff, fp);
  fputc((data >> 16) & 0xff, fp);
  fputc((data >> 24) & 0xff, fp);
}
