/* hu2mzt.c : convert Hu-BASIC text file to MZT format for MZ-700 */
/*   by AKIKAWA, Hisashi 2025.10.7 version                        */
/* This software is released under the MIT license.               */
/*  (https://opensource.org/license/mit)                          */

#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <wctype.h>
#include <wchar.h>
#include <locale.h>
#include <math.h>

#define LINESIZE 1024

#define CODE_PRINT 0x8f
#define CODE_LPRINT 0x90
#define CODE_DATA 0x94
#define CODE_REM 0x97
#define CODE_ELSE 0xc2
#define CODE_TO 0xe0
#define CODE_SUB 0xe4
#define CODE_MINUS 0xf8

int option(int argc, char *argv[]);
int getcode(wchar_t line[], unsigned char code[]);
int getnum(wchar_t line[], unsigned char code[]);
bool dakuon(wchar_t src, wchar_t codew[]);
wchar_t altchr(wchar_t src);
unsigned char charset2(wchar_t src);
unsigned char mz700chr(wchar_t src);

int main(int argc, char *argv[])
{
  int i, j;
  int adr, size;
  long int num;
  int len;
  unsigned char c;
  unsigned char code[10];
  wchar_t codew[2];
  bool in_rem, in_data, in_dq;
  bool is_linenum = false;
  wchar_t *line, *end;
  unsigned char output[0x10000];
  FILE *fp;
  unsigned char with_linenum[] =
    {0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
     0x88, 0x89, 0x8a, 0x8b, 0x8c,
     0xc2, 0xe2};

  /* arguments */
  setlocale(LC_ALL, "");
  if (option(argc, argv) != argc - 3) {
    printf("usage: hu2mzt [option] title input-file output-file\n");
    printf("option:\n");
    printf(" -w utf-8\n");
    printf(" -e euc-jp\n");
    printf(" -s shift jis\n");
    printf("default: system locale\n");
    exit(1);
  }

  /* input file */
  if (strcmp(argv[argc - 2], "-") == 0) {
    fp = stdin;
  } else {
    fp = fopen(argv[argc - 2], "rb");  /* b for Windows */
    if (fp == NULL){
      printf("cannot open %s\n", argv[argc - 2]);
      exit(1);
    }
  }

  /* conversion */
  adr = 0x0080;
  line = malloc(LINESIZE * sizeof(wchar_t));
  memset(output, 0, sizeof(output));

  while (fgetws(line, LINESIZE, fp)) {
    if (line[0] == L'\n') continue;
    if (wcslen(line) > LINESIZE / 2) {
      printf("too long line:\n%ls", line);
      exit(1);
    }

    for (i = 0; i < wcslen(line); i++) {
      if (dakuon(line[i], codew)) {
	wmemmove(&line[i + 2], &line[i + 1], wcslen(line) - i);
	line[i] = codew[0];
	line[i + 1] = codew[1];
      }
      line[i] = altchr(line[i]);
    }

    size = 2;
    in_rem = false;
    in_data = false;
    in_dq = false;

    num = wcstol(line, &end, 10);
    if (num >> 16) {
      printf("illegal line number:\n%ls", line);
      exit(1);
    }
    output[adr + size++] = num & 0xff;
    output[adr + size++] = num >> 8;

    is_linenum = false;

    i = end - line;
    if (line[i] == L' ') {
      i++;
    }
    for (; i < wcslen(line); i++) {
      if (line[i] == L'\n') break;
      if (line[i] == 0xa5) {	/* yen mark */
	line[i] = 0x5c;
      }
      if (adr + size > 0xfff0) {
	printf("too large size\n");
	exit(1);
      }

      /* character code */
      if (line[i] == L'{') {
	bool count = false;
	int k;
	for (j = i + 1; j < wcslen(line); j++) {
	  if (line[j] == L'\n') break;
	  if (line[j] == L'}') {
	    j++;
	    break;
	  }
	  if (iswxdigit(line[j])) {
	    swscanf(&line[j], L"%1x", &k);
	    num = (num << 4) + k;
	    if (count) {
	      output[adr + size++] = num;
	      num = 0;
	    }
	    count = !count;
	  }
	}
	i = j - 1;
	continue;
      }

      /* double quotation and colon */
      if (line[i] == L'"') {
	in_dq = !in_dq;
      } else if (!in_dq && line[i] == L':') {
	in_data = false;
      }

      /* in REM, DATA, "" */
      if (in_rem || in_data || in_dq) {
	if (charset2(line[i])) {
	  output[adr + size++] = 0x10;
	  while (charset2(line[i])) {
	    output[adr + size++] = charset2(line[i++]);
	  }
	  output[adr + size++] = 0x11;
	  i--;
	} else {
	  output[adr + size++] = mz700chr(line[i]);
	}
	continue;
      }

      /* number */
      if (wcschr(L"0123456789.&", line[i])) {
	len = getnum(&line[i], code);
	if (iswdigit(line[i + len])) {
	  printf("cannot convert:\n%ls", line);
	  exit(1);
	}

	if (len > 0) {
	  i += len - 1;
	  if (code[0] == 0x18) {
	    len = 9;
	  } else if (code[0] == 0x15) {
	    len = 6;
	    if (is_linenum) {
	      if (code[1] > 0x90) {
		printf("cannot convert line number:\n%ls", line);
		exit(1);
	      }
	      len = 3;
	      code[0] = 0x0b;
	      code[2] |= 0x80;
	      for (j = 0; j < 0x90 - code[1]; j++) {
		code[3] = ((code[2] & 1) << 7) | (code[3] >> 1);
		code[2] = code[2] >> 1;
	      }
	      code[1] = code[3];
	    }
	  } else if (code[0] > 0x0a) {
	    len = 3;
	    if (is_linenum) {
	      code[0] = 0x0b;
	    }
	  } else if (code[0] > 0) {
	    len = 1;
	    if (is_linenum) {
	      len = 3;
	      code[1] = code[0] - 1;
	      code[2] = 0x00;
	      code[0] = 0x0b;
	    }
	  } else {
	    len = 0;
	  }
	}
	if (len > 0) {
	  for (j = 0; j < len; j++) {
	    output[adr + size++] = code[j];
	  }
	  continue;
	}
      }

      /* comment */
      if (line[i] == L'\'') {
	output[adr + size++] = mz700chr(L':');
	output[adr + size++] = mz700chr(L'\'');
	in_rem = true;
	continue;
      }

      /* reserved word */
      len = getcode(&line[i], code);

      if (len > 0) {
	i += len - 1;
	if (code[0] == CODE_DATA) {
	  in_data = true;
	} else if (code[0] == CODE_REM) {
	  in_rem = true;
	} else if (code[0] == CODE_ELSE) {
	  output[adr + size++] = mz700chr(L':');
	}
	for (j = 0; j < sizeof(with_linenum) / sizeof(with_linenum[0]); j++) {
	  if (code[0] == with_linenum[j]) {
	    is_linenum = true;
	    break;
	  }
	}
	if (j == sizeof(with_linenum) / sizeof(with_linenum[0]) &&
	    code[0] != CODE_TO &&
	    code[0] != CODE_SUB &&
	    code[0] != CODE_MINUS) {
	  is_linenum = false;
	}
	if (code[0] == '?') {
	  code[0] = CODE_PRINT;
	}
	output[adr + size++] = code[0];
	if (code[1]) {
	  output[adr + size++] = code[1];
	}
	continue;
      }

      /* variable */
      if (iswupper(line[i]) || iswlower(line[i])) {
	is_linenum = false;
	if (towupper(line[i]) == L'L' && towupper(line[i + 1]) == L'?') {
	  i++;
	  output[adr + size++] = CODE_LPRINT;
	} else {
	  while (iswupper(line[i]) || iswlower(line[i]) || iswdigit(line[i])) {
	    output[adr + size++] = mz700chr(towupper(line[i++]));
	  }
	  i--;
	}
	continue;
      }

      /* other characters */
      c = mz700chr(line[i]);
      if (0x20 <= c && c <= 0x5d) {
	output[adr + size++] = c;
      }
    }

    /* line end */
    output[adr + size++] = 0x00;
    output[adr] = size & 0xff;
    output[adr + 1] = size >> 8;
    adr += size;
  }

  printf("conversion end\n");

  /* program end */
  output[adr++] = 0x00;
  output[adr++] = 0x00;
  fclose(fp);

  /* MZT header */
  output[0x0000] = 0x02;	/* attribute: 02=Hu-BASIC */
  memset(&output[0x0001], 0x20, 17);
  for (i = 0; i < 16; i++) {
    if (argv[argc - 3][i] == '\0') break;
    output[0x0001 + i] = toupper(argv[argc - 3][i]);
  }
  // Hu-BASIC: without 0x0d
  //  output[i + 1] = 0x0d;
  output[0x0012] = (adr - 0x80) & 0xff;
  output[0x0013] = (adr - 0x80) >> 8;

  /* output file */
  if (strcmp(argv[argc - 1], "-") == 0) {
    fp = stdout;
  } else {
    fp = fopen(argv[argc - 1], "wb");  /* b for Windows */
    if (fp == NULL){
      printf("cannot open %s\n", argv[argc - 1]);
      exit(1);
    }
  }
  fwrite(output, 1, adr, fp);
  fclose(fp);

  return 0;
}

int option(int argc, char *argv[])
{
  int i;

  for (i = 1; i < argc; i++) {
    if (argv[i][0] != '-' || strcmp(argv[i], "-") == 0) break;
    if (strcmp(argv[i], "-w") == 0) {
      if (!setlocale(LC_ALL, "ja_JP.utf8")) {
	printf("cannot set locale\n");
	exit(1);
      }
    } else if (strcmp(argv[1], "-e") == 0) {
      if (!setlocale(LC_ALL, "ja_JP.eucJP")) {
	printf("cannot set locale\n");
	exit(1);
      }
    } else if (strcmp(argv[1], "-s") == 0) {
      if (!setlocale(LC_ALL, "ja_JP.sjis")) {
	if (!setlocale(LC_ALL, "ja_JP.PCK")) {
	  printf("cannot set locale\n");
	  exit(1);
	}
      }
    }
  }
  return i;
}

int getcode(wchar_t line[], unsigned char code[])
{
  int i, j;

  const wchar_t cmdlist[][10] = {
    L"GOTO",     L"GOSUB",    L"GO",       L"RUN",	/* 80 - 83 */
    L"RETURN",   L"RESTORE",  L"RESUME",   L"LIST",	/* 84 - 87 */
    L"LLIST",    L"DELETE",   L"RENUM",    L"AUTO",	/* 88 - 8b */
    L"EDIT",     L"FOR",      L"NEXT",     L"PRINT",	/* 8c - 8f */
    L"LPRINT",   L"INPUT",    L"LINPUT",   L"IF",	/* 90 - 93 */
    L"DATA",     L"READ",     L"DIM",      L"REM",	/* 94 - 97 */
    L"END",      L"STOP",     L"CONT",     L"CLS",	/* 98 - 9b */
    L"CLEAR",    L"ON",       L"LET",      L"NEW",	/* 9c - 9f */
    L"POKE",     L"OFF",      L"WHILE",    L"WEND",	/* a0 - a3 */
    L"REPEAT",   L"UNTIL",    L"",         L"",		/* a4 - a7 */
    L"TRACE",    L"TRON",     L"TROFF",    L"SPEED",	/* a8 - ab */
    L"",         L"",         L"DEFINT",   L"DEFSNG",	/* ac - af */
    L"DEFDBL",   L"DEFSTR",   L"DEF",      L"",		/* b0 - b3 */
    L"LOAD",     L"SAVE",     L"MERGE",    L"CHAIN",	/* b4 - b7 */
    L"CONSOLE",  L"",         L"OUT",      L"SEARCH",	/* b8 - bb */
    L"WAIT",     L"PAUSE",    L"WRITE",    L"SWAP",	/* bc - bf */
    L"ERASE",    L"ERROR",    L"ELSE",     L"CALL",	/* c0 - c3 */
    L"MON",      L"LOCATE",   L"MODE",     L"KEY",	/* c4 - c7 */
    L"PUSH",     L"POP",      L"LABEL",    L"RANDOMIZE",/* c8 - cb */
    L"OPTION",   L"LINE",     L"OPEN",     L"CLOSE",	/* cc - cf */
    L"",         L"FIELD",    L"GET",      L"PUT",	/* d0 - d3 */
    L"SET",      L"FILES",    L"LFILES",   L"DEVICE",	/* d4 - d7 */
    L"NAME",     L"KILL",     L"LSET",     L"RSET",	/* d8 - db */
    L"INIT",     L"VDIM",     L"MAXFILES", L"",		/* dc - df */
    L"TO",       L"STEP",     L"THEN",     L"USING",	/* e0 - e3 */
    L"SUB",      L"BASE",     L"TAB",      L"SPC",	/* e4 - e7 */
    L"EQV",      L"IMP",      L"XOR",      L"OR",	/* e8 - eb */
    L"AND",      L"NOT",      L"><",       L"<>",	/* ec - ef */
    L"=<",       L"<=",       L"=>",       L">=",	/* f0 - f3 */
    L"=",        L">",        L"<",        L"+",	/* f4 - f7 */
    L"-",        L"MOD",      L"\\",       L"/",	/* f8 - fb */
    L"*",        L"^"					/* fc - fd */
  };

  const wchar_t cmdlist_fe[][10] = {
    L"",         L"PSET",     L"PRESET",   L"COLOR",	/* 80 - 83 */
    L"",         L"",         L"",         L"",		/* 84 - 87 */
    L"",         L"",         L"",         L"PLAY",	/* 88 - 8b */
    L"",         L"BEEP",     L"",         L"",		/* 8c - 8f */
    L"",         L"",         L"",         L"",		/* 90 - 93 */
    L"CGEN",     L"PCOLOR",   L"SKIP",     L"RLINE",	/* 94 - 97 */
    L"MOVE",     L"RMOVE",    L"PHOME",    L"HSET",	/* 98 - 9b */
    L"GPRINT",   L"AXIS",     L"CIRCLE",   L"TEST",	/* 9c - 9f */
    L"PLOT",     L"PAGE",     L"MUSIC",    L"TEMPO",	/* a0 - a3 */
    L"CURSOR",   L"VERIFY",   L"CLR",      L"LIMIT",	/* a4 - a7 */
    L"KLIST",    L"",         L"",         L"CLICK",	/* a8 - ab */
    L"BOOT",     L"DEVI$",    L"DEVO$",			/* ac - ae */
  };

  const wchar_t cmdlist_ff[][10] = {
    L"INT",      L"ABS",      L"SIN",      L"COS",	/* 80 - 83 */
    L"TAN",      L"LOG",      L"EXP",      L"SQR",	/* 84 - 87 */
    L"RND",      L"PEEK",     L"ATN",      L"SGN",	/* 88 - 8b */
    L"FRAC",     L"FIX",      L"PAI",      L"RAD",	/* 8c - 8f */
    L"INP",      L"CDBL",     L"CSNG",     L"CINT",	/* 90 - 93 */
    L"DSKF",     L"EOF",      L"FPOS",     L"LOC",	/* 94 - 97 */
    L"LOF",      L"POS",      L"FAC",      L"SUM",	/* 98 - 9b */
    L"FRE",      L"LPOS",     L"JOY",      L"",		/* 9c - 9f */
    L"CHR$",     L"STR$",     L"HEX$",     L"OCT$",	/* a0 - a3 */
    L"BIN$",     L"MKI$",     L"MKS$",     L"MKD$",	/* a4 - a7 */
    L"SPACE$",   L"",         L"",         L"ASC",	/* a8 - ab */
    L"LEN",      L"VAL",      L"CVS",      L"CVD",	/* ac - af */
    L"CVI",      L"",         L"",         L"ERR",	/* b0 - b3 */
    L"ERL",      L"CSRLIN",   L"STRPTR",   L"DTL",	/* b4 - b7 */
    L"",         L"",         L"LEFT$",    L"RIGHT$",	/* b8 - bb */
    L"MID$",     L"INKEY$",   L"INSTR",    L"HEXCHR$",	/* bc - bf */
    L"MEM$",     L"SCRN$",    L"VARPTR",   L"STRING$",	/* c0 - c3 */
    L"TIME$",    L"",         L"",         L"FN",	/* c4 - c7 */
    L"USR",      L"",         L"",         L"ATTR$",	/* c8 - cb */
    L"",         L"CHARACTER$"				/* cc - cd */
  };

  for (i = 0; i < sizeof(cmdlist) / sizeof(cmdlist[0]); i++) {
    for (j = 0; j < wcslen(cmdlist[i]); j++) {
      if (towupper(line[j]) != cmdlist[i][j]) break;
    }
    if (j > 0 && j == wcslen(cmdlist[i])) {
      code[0] = 0x80 + i;
      code[1] = 0;
      return wcslen(cmdlist[i]);
    }
  }

  for (i = 0; i < sizeof(cmdlist_fe) / sizeof(cmdlist_fe[0]); i++) {
    for (j = 0; j < wcslen(cmdlist_fe[i]); j++) {
      if (towupper(line[j]) != cmdlist_fe[i][j]) break;
    }
    if (j > 0 && j == wcslen(cmdlist_fe[i])) {
      code[0] = 0xfe;
      code[1] = 0x80 + i;
      return wcslen(cmdlist_fe[i]);
    }	
  }
  for (i = 0; i < sizeof(cmdlist_ff) / sizeof(cmdlist_ff[0]); i++) {
    for (j = 0; j < wcslen(cmdlist_ff[i]); j++) {
      if (towupper(line[j]) != cmdlist_ff[i][j]) break;
    }
    if (j > 0 && j == wcslen(cmdlist_ff[i])) {
      code[0] = 0xff;
      code[1] = 0x80 + i;
      return wcslen(cmdlist_ff[i]);
    }	
  }
  return 0;
}

int getnum(wchar_t line[], unsigned char code[])
{
  int i;
  int figures;
  long int integer = 0;
  double frac;
  bool is_double = false;
  bool with_dot = false;
  wchar_t *end;
  const int base[] = {8, 2, 16};

  code[0] = 0;	/* for illegal format */

  /* &O, &B, &H */
  if (line[0] == '&') {
    for (i = 0; i < sizeof(base) / sizeof(base[0]); i++) {
      if (towupper(line[1]) == L"OBH"[i]) {
	integer = wcstol(line + 2, &end, base[i]);
	code[0] = 0x0d + i;
	break;
      }
    }
    if (i == sizeof(base) / sizeof(base[0])) {
      return 0;
    }
    if (integer >>16) {
      printf("over flow: %.*ls\n", (int)(end-line), line);
      exit(1);
    }
    code[1] = integer & 0xff;
    code[2] = integer >> 8;
    return end - line;
  }

  /* others */
  frac = wcstod(line, &end);
  figures = 0;
  for (i = 0; i < end - line; i++) {
    if (towupper(line[i]) == L'E') break;
    if (iswdigit(line[i])) {
      figures++;
    }
    if (line[i] == L'.') {
      with_dot = true;
    }
  }

  if (toupper(*end) == L'D') {
    is_double = true;
    integer = wcstol(end + 1, &end, 10);
    frac *= pow(10, integer);
  }

  integer = (int)frac;

  if (fabs(frac) >= pow(2, 127)) {
    printf("over flow: %.*ls\n", (int)(end - line), line);
    exit(1);
  } else if (*end == L'#') {
    is_double = true;
    end++;
  } else if (*end == L'!') {
    is_double = false;
    end++;
  } else if (figures >= 8) {
    is_double = true;
  } else if (*end == L'%' ||
	     (!is_double && frac == integer && !with_dot &&
	      (!wcspbrk(line, L"eE") || end <= wcspbrk(line, L"eE")))) {
    if (*end == L'%') {
      end++;
    }
    if (0 <= integer && integer <= 9) {
      code[0] = integer + 1;
      return end - line;
    } else if (-32768 <= integer && integer <= 32767) {
      code[0] = 0x12;
      code[1] = integer & 0xff;
      code[2] = integer >> 8;
      return end - line;
    }
  }

  if (is_double) {
    code[0] = 0x18;
  } else {
    code[0] = 0x15;
  }
  if (fabs(frac) < pow(2, -128)) {
    frac = 0;
    code[1] = 0x00;
  } else {
    code[1] = 0x81 + (int)floor(log2(fabs(frac)));
    frac /= pow(2, code[1] - 0x81);
    frac *= 0x80;
  }
  code[2] = frac < 0 ? 0x80: 0x00;
  frac = fabs(frac);
  code[2] |= (int)frac & 0x7f;
  for (i = 0; i < code[0] - 0x10 - 2; i++) {
    frac -= (int)frac;
    frac *= 0x100;
    code[i + 3] = (int)frac;
  }
  return end - line;
}

bool dakuon(wchar_t src, wchar_t codew[])
{
  int i;
  const wchar_t dakuon[] =
    L"がぎぐげござじずぜぞだぢづでどばびぶべぼ"
    L"ガギグゲゴザジズゼゾダヂヅデドバビブベボヴ"
    L"ぱぴぷぺぽパピプペポ";
  const wchar_t seion[] =
    L"かきくけこさしすせそたちつてとはひふへほ"
    L"カキクケコサシスセソタチツテトハヒフヘホウ"
    L"はひふへほハヒフヘホ";

  for (i = 0; i < wcslen(dakuon); i++) {
    if (src == dakuon[i]) {
      codew[0] = seion[i];
      codew[1] = i < 41 ? L'゛' : L'゜';
      return true;
    }
  }
  return false;
}

wchar_t altchr(wchar_t src)
{
  int i;
  const wchar_t org[] =
    L" !\"#$%&'()*+,-./"
    L"0123456789:;<=>?"
    L"@ABCDEFGHIJKLMNO"
    L"PQRSTUVWXYZ[\\]^"
    L"abcdefghijklmnopqrstuvwxyz"

    L"。「」、・ヲァィゥェォャュョッ"
    L"ーアイウエオカキクケコサシスセソ"
    L"タチツテトナニヌネノハヒフヘホマ"
    L"ミムメモヤユヨラリルレロワン゛゜"

    L"┣┗┓┏┻┳┫┛━╋┃"
    L"\"#'[]^";

  const wchar_t alt[] =
    L"　！”＃＄％＆’（）＊＋，‐．／"
    L"０１２３４５６７８９：；＜＝＞？"
    L"＠ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯ"
    L"ＰＱＲＳＴＵＶＷＸＹＺ［＼］＾"
    L"ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚ"

    L"｡｢｣､･ｦｧｨｩｪｫｬｭｮｯ"
    L"ｰｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ"
    L"ﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ"
    L"ﾐﾑﾒﾓﾔﾕﾖﾗﾘﾙﾚﾛﾜﾝﾞﾟ"

    L"├└┐┌┴┬┤┘─┼│"
    L"“♯‘〔〕↑";
  
  for (i = 0; i < wcslen(org); i++) {
    if (src == alt[i]) return org[i];
  }
  return src;
}

unsigned char charset2(wchar_t src)
{
  int i;
  const wchar_t lower[] =
    L"abcdefghijklmnopqrstuvwxyz";
  const wchar_t hiragana[] =
    L"をぁぃぅぇぉゃゅょっーあいうえおかきくけこさしすせそたちつてとなにぬねのはひふへほまみむめもやゆよらりるれろわん";

  for (i = 0; i < wcslen(lower); i++) {
    if (src == lower[i]) {
      return 0x41 + i;
    }
  }
  for (i = 0; i < wcslen(hiragana); i++) {
    if (src == hiragana[i] && src != L'ー') {
      return 0x86 + i;
    }
  }
  return 0;
}

unsigned char mz700chr(wchar_t src)
{
  int i;
  const wchar_t mztbl[] = {
    L'\0', L'\0', L'\0', L'\0', L'\0', L'\0', L'\0', L'\0',
    L'\0', L'\0', L'\0', L'\0', L'©', L'\0', L'\0', L'\0',
    L'\0', L'\0', L'\0', L'\0', L'\0', L'\0', L'\0', L'\0',
    L'\0', L'\0', L'\0', L'\0', L'右', L'左', L'上', L'下',
    L' ', L'!', L'\"', L'#', L'$', L'%', L'&', L'\'',
    L'(', L')', L'*', L'+', L',', L'-', L'.', L'/',
    L'0', L'1', L'2', L'3', L'4', L'5', L'6', L'7',
    L'8', L'9', L':', L';', L'<', L'=', L'>', L'?',
    L'@', L'A', L'B', L'C', L'D', L'E', L'F', L'G',
    L'H', L'I', L'J', L'K', L'L', L'M', L'N', L'O',
    L'P', L'Q', L'R', L'S', L'T', L'U', L'V', L'W',
    L'X', L'Y', L'Z', L'[', L'\\', L']', L'^', L'←',
    L'\0', L'\0', L'\0', L'\0', L'\0', L'\0', L'\0', L'\0',
    L'\0', L'蛇', L'\0', L'\0', L'\0', L'\0', L'\0', L'\0',
    L'日', L'月', L'火', L'水', L'木', L'金', L'土', L'生',
    L'年', L'時', L'分', L'秒', L'円', L'￥', L'£', L'\0',
    L'↓', L'。', L'「', L'」', L'、', L'・', L'ヲ', L'ァ',
    L'ィ', L'ゥ', L'ェ', L'ォ', L'ャ', L'ュ', L'ョ', L'ッ',
    L'ー', L'ア', L'イ', L'ウ', L'エ', L'オ', L'カ', L'キ',
    L'ク', L'ケ', L'コ', L'サ', L'シ', L'ス', L'セ', L'ソ',
    L'タ', L'チ', L'ツ', L'テ', L'ト', L'ナ', L'ニ', L'ヌ',
    L'ネ', L'ノ', L'ハ', L'ヒ', L'フ', L'ヘ', L'ホ', L'マ',
    L'ミ', L'ム', L'メ', L'モ', L'ヤ', L'ユ', L'ヨ', L'ラ',
    L'リ', L'ル', L'レ', L'ロ', L'ワ', L'ン', L'゛', L'゜',
    L'→', L'\0', L'\0', L'\0', L'\0', L'\0', L'網', L'\0',
    L'■', L'\0', L'\0', L'┣', L'□', L'┗', L'┓', L'\0',
    L'┏', L'┻', L'┳', L'┫', L'\0', L'\0', L'\0', L'\0',
    L'\0', L'\0', L'\0', L'\0', L'\0', L'┛', L'\0', L'\0',
    L'━', L'▲', L'\0', L'\0', L'\0', L'\0', L'\0', L'\0',
    L'\0', L'\0', L'\0', L'\0', L'\0', L'\0', L'\0', L'\0',
    L'\0', L'●', L'\0', L'▼', L'\0', L'\0', L'×', L'○',
    L'★', L'\0', L'◆', L'╋', L'\0', L'┃', L'\0', L'π',
  };

  for (i = 0; i < 0x100; i++) {
    if (src == mztbl[i]) return i;
  }
  return src;
  return L' ';
}
