;BASIC interpreter for MZ-700
; by AKIKAWA, Hisashi 2025
;This software is released under the MIT license.
; (https://opensource.org/license/mit)

;for IOCS compatibility
DSPXY	equ	000eh		;cursor position, x(0-orgin), y(0-orgin)
E000VAL	equ	0594h		;output to e000h (for checking break key)
TONEVAL	equ	0c7ch		;8253 counter value for BEEP, 0c7c-0c7d
CMDNAME	equ	2dbeh		;command name table
CMDTBL	equ	315ch		;command jump table
GETCMER	equ	32b2h		;get comma and check error
INT1ARG	equ	63d6h		;get a 1-byte integer argument

;same address as ROM monitor work area
IBUFE	equ	10f0h		;information block buffer, 10f0-116f
FNAME	equ	10f1h		;file name
SIZE	equ	1102h		;program size
DTADR	equ	1104h		;program address
EXADR	equ	1106h		;execution address
ATSTRT	equ	1109h		;auto start
KANAF	equ	1170h		;input mode (0=alphanum, 1=kana, 2=graph)
FLASH	equ	118eh		;shunted character
CSRAD	equ	118fh		;cursor blink address
FLSST	equ	1191h		;cursor blink status (0=on, 1=off)
FLSDT	equ	1192h		;cursor character (display code)
AMPM	equ	119bh		;0=AM, 1=PM
TEMPW	equ	119eh		;tempo data (1=slow, 7=fast)

;work area
PROGAD	equ	8800h		;program address
LINENUM	equ	8802h		;line number
STOPAD	equ	8804h		;stop address
STOPLN	equ	8806h		;stop line number
DATAAD	equ	8808h		;data address

FBUFAD	equ	880ah		;file buffer address
BUFP	equ	880ch		;buffer pointer
FILES	equ	880eh		;max files

ERRVAL	equ	880fh		;return value of ERR function
ERLVAL	equ	8810h		;return value of ERL function

SELPRN	equ	8812h		;0=CRT, 1=printer
PRNMOD	equ	8813h		;printer mode (0=text,1=graphic)
PRNPOS	equ	8814h		;printer x position

PRTSPD	equ	8815h		;SPEED command
TRACEON	equ	8816h		;0=trace off, not0=trace on
RPTON	equ	8817h		;0=repeat off, not0=repeat on
CLICKON	equ	8818h		;0=click off, not0=click on

NOCSR	equ	8819h		;not0=stop curosr blink in GETC
CHKFKY	equ	881ah		;0=no data in function key buffer
INMON	equ	881bh		;in MON command?
DEV_DEF	equ	881ch		;default device number

ATTR	equ	8820h		;attribute
XMIN	equ	8821h		;console par3
YMIN	equ	8822h		;console par1
XMAX	equ	8823h		;console par3+par4-1
YMAX	equ	8824h		;console par1+par2-1
LINEST	equ	8825h		;line connection status, 8825-882b

OPRTR	equ	882fh		;operator, FAC1-1
FAC1	equ	8830h		;floating accumlator 1, 8830-8838
FAC2	equ	883ah		;floating accumlator 2, 883a-8842
FAC3	equ	8844h		;for string conversion, 8844-885a, 23bytes

TMP	equ	8860h		;temporary
MULWRK	equ	8862h		;for multiplication, 8862-886f

STRDSC1	equ	8870h		;temporary string descriptor, 8870-8872
STRDSC2	equ	8873h		;temporary string descriptor, 8873-8875
STRDSC3	equ	8876h		;temporary string descriptor, 8876-8878
STRDSC4	equ	8879h		;temporary string descriptor, 8879-887b
SDSCCNT	equ	887ch		;temporary strinr descriptor count

;
VARAD	equ	887fh		;variable area
STRPTR	equ	8881h		;string data area
STACK	equ	8887h		;stack pointer
BASICAD	equ	888bh		;BASIC program area
INPBUF	equ	888eh		;input buffer
BASINI	equ	898eh		;initial BASIC program area
;
TMPSTR	equ	8883h		;temporary string area
FREEAD	equ	8885h		;fre area start address
;
STKINI	equ	0ff00h		;initial stack pointer

;constant
WIDTH	equ	28h
HEIGHT	equ	19h
MAXINP	equ	BASINI-INPBUF-3	;max characters for input buffer
STKMGN	equ	0030h		;stack margin

;DEVCRT	equ	01h		;device number: CRT
;DEVSCR	equ	02h		;device number: screen
DEVCAS	equ	03h		;device number: cassete
;DEVPLT	equ	04h		;device number: plotter
;DEVLPT	equ	05h		;device number: printer
;DEVKEY	equ	06h		;device number: keyboard

;intermediate code
GOTO_	equ	80h
GOSUB_	equ	81h
RETURN_	equ	84h
RSTR_	equ	85h
RESUME_	equ	86h
LIST_	equ	87h
LLIST_	equ	88h
EDIT_	equ	8ch
FOR_	equ	8dh
NEXT_	equ	8eh
PRINT_	equ	8fh
LPRINT_	equ	90h
INPUT_	equ	91h
IF_	equ	93h
DATA_	equ	94h
REM_	equ	97h
ON_	equ	9dh
OFF_	equ	0a1h
WHILE_	equ	0a2h
WEND_	equ	0a3h
REPEAT_	equ	0a4h
ERROR_	equ	0c1h
ELSE_	equ	0c2h
TIME_	equ	0c4h
KEY_	equ	0c7h
LABEL_	equ	0cah
THEN_	equ	0e2h
TO_	equ	0e0h
STEP_	equ	0e1h
USING_	equ	0e3h
SUB_	equ	0e4h
BASE_	equ	0e5h
TAB_	equ	0e6h
SPC_	equ	0e7h
EQV_	equ	0e8h
NOT_	equ	0edh
EQ_	equ	0f4h
PLUS_	equ	0f7h
MINUS_	equ	0f8h
ASTRSK_	equ	0fch
DEVO_	equ	0aeh

;function
RND_	equ	88h
PEEK_	equ	89h
PAI_	equ	8eh
POS_	equ	99h
CHR_	equ	0a0h
CVI_	equ	0b0h
MID_	equ	0bch
MEM_	equ	0c0h
FN_	equ	0c7h
USR_	equ	0c8h

;;;;;;;;;;;;;;;;;;;;;;;;;

	org	1200h

COLD:
	di

;IOCS subroutine
	out	(0e0h),a	;0000-0fff:DRAM
	ld	hl,IOCSTBL
	ld	a,0+(IOCSTBLEND-IOCSTBL)/4
COLDLP1:
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
	ld	c,(hl)
	inc	hl
	ld	b,(hl)
	inc	hl
	ex	de,hl
	ld	(hl),0c3h
	inc	hl
	ld	(hl),c
	inc	hl
	ld	(hl),b
	ex	de,hl
	dec	a
	jr	nz,COLDLP1

;timer
	out	(0e3h),a	;d000-ffff:VRAM/IO
	ld	hl,0e007h
	ld	a,74h		;8253 counter1, low->high, mode2, binary
	ld	(hl),a
	ld	a,0b0h		;8253 counter2, low->high, mode0, binary
	ld	(hl),a
	dec	hl		;8253 counter2 value
	ld	(hl),0c0h	;60*60*12
	ld	(hl),0a8h
	dec	hl		;8253 counter1 value
	ld	(hl),54h	;15700
	ld	(hl),3dh

;clear VRAM
	ld	hl,0d000h
	ld	de,0d001h
	ld	bc,0fffh
	ld	(hl),l		;=0
	ldir

;interrupt
	im	1
	ld	a,05h
	ld	(0e003h),a	;8255 PC2=interrupt enable
	out	(0e1h),a	;d000-ffff:DRAM
	xor	a
	ld	(BANKST+1),a
	ld	(AMPM),a

;work area
	ld	hl,8800h
	ld	de,8801h
;	ld	bc,007dh
	ld	c,7dh		;b=0
	ld	(hl),a		;=0
	ldir			;8800-887e

;files
	inc	a
	ld	(FILES),a	;=1

;repeat
	ld	(RPTON),a	;repeat on

;default device
	ld	a,03h
	ld	(DEV_DEF),a	;3=CMT

;music/play
	inc	a
	ld	(TEMPW),a	;=4=8-4
	inc	a
	ld	(TONELEN+1),a	;=5

;key scan for break check
	ld	a,0f8h
	ld	(E000VAL),a

;stack
	ld	hl,STKINI
	ld	(STACK),hl
	ld	sp,hl

	ei

;screen
	call	SEDINICON
	call	CTLALP
	call	CSROFF

;BASIC
	ld	hl,BASINI
	ld	(BASICAD),hl
	call	NEW
	call	RESSTK

;function key
	ld	hl,FKEYLST
	ld	de,FKEYTBL+0010h
	ld	c,0ah*2
COLDLP2:
	ld	a,(hl)
	inc	hl
	cp	0feh
	jr	c,COLDC
	ld	b,(hl)
	inc	hl
COLDC:
	push	hl
	call	CNVASC
;	ld	b,10h
	ld	b,0fh
COLDLP3:
	dec	b
	ld	a,(hl)
	and	7fh
	ld	(de),a
	ld	a,(hl)
	inc	de
	inc	hl
	rlca
	jr	nc,COLDLP3
	pop	hl

	ldi
;	dec	b
	xor	a
COLDLP4:
	ld	(de),a
	inc	de
	djnz	COLDLP4

	dec	c
	jr	nz,COLDLP2

;message
	call	CLSMAIN
	ld	b,27h
COLDLP5:
	ld	a,0e0h
	call	PUTC
	djnz	COLDLP5
	call	PUTNL
	ld	hl,SYSNAME_STR
	call	PUTSNL
	ld	b,27h
COLDLP6:
	ld	a,0e0h
	call	PUTC
	djnz	COLDLP6
	call	PUTNL
	call	CHKFRE
	ex	de,hl
	ld	de,0-8		;adjust to ?fre(0)
	add	hl,de
	call	PUTI2
	ld	hl,BFREE_STR
	call	PUTSNL
	jp	EDIT


;system name
SYSNAME_STR:
	db	"BASIC INTERPRETER VER.251115 FOR MZ-700", 00h
BFREE_STR:
	db	" BYTES FREE", 00h

IOCSTBL:
	dw	0000h, COLD	;cold start
	dw	0008h, RDVRAM	;read VRAM
	dw	0010h, SETVRAM	;set VRAM
	dw	0038h, INTRPT	;interrupt
	dw	0c25h, BEEPON	;beep on
	dw	0c40h, BEEPOFF	;beep off
IOCSTBLEND:

;interrupt (8253 counter2=mode0)
INTRPT:
	push	af
	push	hl
	out	(0e3h),a	;d000-fff:VRAM/IO
	ld	hl,0e006h	;8253 counter2 value
	ld	(hl),0c0h	;60*60*12
	ld	(hl),0a8h
	ld	hl,AMPM
	ld	a,(hl)
	xor	01h
	ld	(hl),a
BANKST:
	ld	a,00h		;00=DRAM, 01=VRAM/IO
	or	a
	jr	nz,INTRPTEND
	out	(0e1h),a	;d000-ffff:DRAM
INTRPTEND:
	pop	hl
	pop	af
	ei
	ret


;wait for vertical blanking 1->0 (d000-ffff:VRAM/IO)
;destroy: af
VBLNK:
	push	hl
	ld	hl,0e002h
	xor	a
VBLNKLP1:
	or	(hl)
	jp	p,VBLNKLP1
VBLNKLP2:
	and	(hl)
	jp	m,VBLNKLP2
	pop	hl
	ret


;change d000-ffff to VRAM/IO
;destoy: af
CHGVRAM:
	pop	af		;return address
	ld	(SPBANK+1),sp
	di
	out	(0e3h),a	;d000-ffff:VRAM/IO
	ld	sp,IBUFE
	push	af
	ld	a,01h
	ld	(BANKST+1),a
	ei
	ret

;change d000-ffff to DRAM
;destoy: af
CHGDRAM:
	pop	af		;return address
	di
	out	(0e1h),a	;d000-ffff:DRAM
SPBANK:
	ld	sp,0000h
	push	af
	xor	a
	ld	(BANKST+1),a
	ei
	ret


;read VRAM
;input: hl=address
;output: a
;destroy: f,b
RDVRAM:
	call	CHGVRAM
	ld	b,(hl)
	call	CHGDRAM
	ld	a,b
	ret

;set VRAM
;input: a=value, hl=address
;destroy: af,b
SETVRAM:
	ld	b,a
	call	CHGVRAM
	ld	(hl),b
	call	CHGDRAM
	ret


;beep on
;input: (TONEVAL)
;destroy: af
BEEPON:
	push	hl
	call	CHGVRAM
	ld	a,36h
	ld	(0e007h),a	;couter0, low->high, mode3, binary
	ld	hl,(TONEVAL)
	ld	a,l
	ld	(0e004h),a
	ld	a,h
	ld	(0e004h),a
	ld	a,01h		;8253 gate=L
BEEPGATE:
	ld	(0e008h),a	;8253 gate
	call	CHGDRAM
	pop	hl
	ret

;beep off
;destroy: af
BEEPOFF:
	push	hl
	call	CHGVRAM
	xor	a		;8253 gate=L
	jr	BEEPGATE


;read 8253 timer (ch2,ch1)
;output: de=ch2, hl=ch1
;destroy: af
RDTIMER:
	call	CHGVRAM

	ld	a,80h		;8253 counter2, latch operation
	ld	hl,0e007h
	di
;	ld	(hl),a
	dec	hl
	ld	e,(hl)
	ld	d,(hl)
	inc	hl
	ld	a,40h		;8253 counter1, latch operation
;	ld	(hl),a
	dec	hl
	dec	hl
	ld	a,(hl)
	ld	h,(hl)
	ld	l,a
	ei
	call	CHGDRAM
	ret


;display code -> ASCII code
;input: a=display code
;output: a=ASCII code
;destroy: f
DACN:
	push	hl
	push	bc
	ld	hl,ADCNTBL
	ld	bc,0100h
	cpir
	ld	a,' '
	jr	nz,DACNNZ
	ld	a,0ffh
	sub	c
DACNNZ:
	pop	bc
	pop	hl
	ret

;ASCII code -> display code
;input: a=ASCII code
;output: a=display code
;destroy: f
ADCN:
	push	hl
	ld	hl,ADCNTBL
	add	a,l
	ld	l,a
	jr	nc,ADCNNC
	inc	h
ADCNNC:
	ld	a,(hl)
	pop	hl
	ret

ADCNTBL:
;
	db	0f0h,0f0h,0f0h,0f0h,0f0h,0f0h,0f0h,0f0h
;
	db	0f0h,0f0h,0f0h,0f0h,0f0h,0f0h,0f0h,0f0h
;		     down up   right left home clr
	db	0f0h,0c1h,0c2h,0c3h,0c4h,0c5h,0c6h,0f0h
;
	db	0f0h,0f0h,0f0h,0f0h,0f0h,0f0h,0f0h,0f0h
;		     !    "    #    $    %    &    '
	db	00h, 61h, 62h, 63h, 64h, 65h, 66h, 67h
;		(    )    *    +    ,    -    .    /
	db	68h, 69h, 6bh, 6ah, 2fh, 2ah, 2eh, 2dh
;		0    1    2    3    4    5    6    7
	db	20h, 21h, 22h, 23h, 24h, 25h, 26h, 27h
;		8    9    :    ;    <    =    >    ?
	db	28h, 29h, 4fh, 2ch, 51h, 2bh, 57h, 49h
;		@    A    B    C    D    E    F    G
	db	55h, 01h, 02h, 03h, 04h, 05h, 06h, 07h
;		H    I    J    K    L    M    N    O
	db	08h, 09h, 0ah, 0bh, 0ch, 0dh, 0eh, 0fh
;		P    Q    R    S    T    U    V    W
	db	10h, 11h, 12h, 13h, 14h, 15h, 16h, 17h
;		X    Y    Z    [    bksl ]    up   left
	db	18h, 19h, 1ah, 52h, 59h, 54h, 50h, 45h
;		ufo  car  car  man  man  man  man  face
	db	0c7h,0c8h,0c9h,0cah,0cbh,0cch,0cdh,0ceh
;		face snak diod diod npn  pnp  cap  cap
	db	0cfh,0dfh,0e7h,0e8h,0e9h,0eah,0ech,0edh
;		sun  mon  tue  wed  thu  fri  sat  birth
	db	0d0h,0d1h,0d2h,0d3h,0d4h,0d5h,0d6h,0d7h
;		year hour min  sec  yen  yen  pound play
	db	0d8h,0d9h,0dah,0dbh,0dch,0ddh,0deh,0c0h
;		down priod bra ket  comm dot  wo   xa
	db	80h, 0bdh,9dh, 0b1h,0b5h,0b9h,0b4h,9eh
;		xi   xu   xe   xo   xya  xyu  xyo  xtu
	db	0b2h,0b6h,0bah,0beh,9fh, 0b3h,0b7h,0bbh
;		long a    i    u    e    o    ka   ki
	db	0bfh,0a3h,85h, 0a4h,0a5h,0a6h,94h, 87h
;		ku   ke   ko   sa   shi  su   se   so
	db	88h, 9ch, 82h, 98h, 84h, 92h, 90h, 83h
;		ta   chi  tsu  te   to   na   ni   nu
	db	91h, 81h, 9ah, 97h, 93h, 95h, 89h, 0a1h
;		ne   no   ha   hi   fu   he   ho   ma
	db	0afh,8bh, 86h, 96h, 0a2h,0abh,0aah,8ah
;		mi   mu   me   mo   ya   yu   yo   ra
	db	8eh, 0b0h,0adh,8dh, 0a7h,0a8h,0a9h,8fh
;		ri   ru   re   ro   wa   nn  dakuten handakuten
	db	8ch, 0aeh,0ach,9bh, 0a0h,99h, 0bch,0b8h
;		right
	db	40h, 3bh, 3ah, 70h, 3ch, 71h, 0efh,3dh
	db	43h, 56h, 3fh, 1eh, 4ah, 1ch, 5dh, 3eh
	db	5ch, 1fh, 5fh, 5eh, 37h, 7bh, 7fh, 36h
	db	7ah, 7eh, 33h, 4bh, 4ch, 1dh, 6ch, 5bh
	db	78h, 41h, 35h, 34h, 74h, 30h, 38h, 75h
	db	39h, 4dh, 6fh, 6eh, 32h, 77h, 76h, 72h
	db	73h, 47h, 7ch, 53h, 31h, 4eh, 6dh, 48h
	db	46h, 7dh, 44h, 1bh, 58h, 79h, 42h, 60h


;MON command
C_MON:
	ld	(MONR+1),sp
MONLP:
	ld	a,01h
	ld	(INMON),a
	ld	sp,0000h
	call	PUTNLX
	call	MONINPT
	jr	c,MONLP
	ld	a,(hl)
	inc	hl
	cp	':'
	jr	z,MONCLN
	cp	'*'
	jr	nz,MONLP

	ld	a,(hl)
	inc	hl
	sub	'D'
	cp	'V'-'D'+1
	ld	de,MONCMD
	call	c,JPTBL
	jr	MONLP

;set
MONCLN:
	call	MONSET
	jr	MONLP


MONSET:	
	call	GETHEX2
	ret	nc
	ld	a,(hl)
	cp	'='
MONSETLP:
	ret	nz
	inc	hl
	call	GETHEX1
	ret	nc
	ld	(de),a
	inc	de
	ld	a,(hl)
	cp	' '
	jr	MONSETLP

;'D'-'V'
MONCMD:
	dw	MOND,  MONRET, MONF,   MONG,   MONRET, MONRET, MONRET, MONRET
	dw	MONL,  MONM,   MONRET, MONRET, MONP,   MONRET, MONR,   MONS
	dw	MONT,  MONRET, MONV

;dump
MOND:
	call	GETHEX2
	jr	nc,MONDAD
	ld	(MONDAD+1),de
	ld	a,(hl)
	cp	' '
	scf
	ccf
	jr	nz,MONDAD
	call	GETHEX2
	ex	de,hl
MONDAD:
	ld	de,0000h
	jr	c,MONDC
	ld	hl,007fh
	add	hl,de		;end address
;de=start, hl=end
MONDC:
	ex	de,hl
;hl=start, de=end
MONDLP1:
	ld	a,':'
	call	PUTC
	call	PUTHEX2
	ld	a,'='
	call	PUTC

	push	hl
	ld	b,08h
MONDLP2:
	ld	a,(hl)
	call	PUTHEX1
	ld	a,' '
	call	PUTC
	or	a
	sbc	hl,de
	add	hl,de
	jr	z,MONDZ
	inc	hl
	djnz	MONDLP2

MONDZ:
	ld	hl,(DSPXY)
	ld	l,1eh
	call	SETCSR

	pop	hl
	ld	a,'/'
	call	PUTC
	ld	b,08h
MONDLP3:
	ld	a,(hl)
	cp	20h
	jr	nc,MONDNC
	ld	a,'.'
MONDNC:
	call	PUTC
	or	a
	sbc	hl,de
	add	hl,de
	inc	hl
	ld	(MONDAD+1),hl
	jp	z,PUTNL
	djnz	MONDLP3
	call	PUTNL
	call	STOPESC
	jr	nz,MONDLP1
	jp	AFTBRK

;find
MONF:
	call	GETHEX2		;start
	ret	nc
	ld	a,(hl)
	cp	' '
	ret	nz
	push	de
	call	GETHEX2		;end
	ld	b,d
	ld	c,e
	pop	de
	ret	nc
	ld	a,(hl)
	cp	' '
	ret	nz
	inc	hl

;hl=input buffer,de=start,bc=end
MONFLP1:
	push	hl		;input buffer
	push	de		;start address

	push	bc
	call	GETHEX1
	pop	bc
	jr	nc,MONNOFND
MONFLP2:
	ex	de,hl
	cp	(hl)
	ex	de,hl
	jr	nz,MONFNEXT
	ld	a,(hl)
	cp	' '
	jr	nz,MONFFND
	push	bc
	call	GETHEX1
	pop	bc
	jr	nc,MONFFND
	ex	de,hl
	or	a
	sbc	hl,bc
	add	hl,bc
	jr	z,MONNOFND
	ex	de,hl
	inc	de
	jr	MONFLP2

;found
MONFFND:
	pop	hl		;start address
	push	hl		;start address
	push	bc		;end address
	call	MONDLP1
	pop	bc		;end address
;	jr	MONFNEXT

;next address
MONFNEXT:
	pop	de		;start address
	pop	hl		;input buffer address
	ex	de,hl
	or	a
	sbc	hl,bc
	add	hl,bc
	ex	de,hl
	ret	z		;not found
	inc	de
	call	CHKBRK
	jr	nz,MONFLP1
	jp	AFTBRK

;not found	
MONNOFND:
	pop	de		;start address
	pop	hl		;input buffer address
	ret


;gosub
MONG:
	call	GETHEX2
	ret	nc
	xor	a
	ld	(PLOTON+1),a
	ld	(INMON),a
	ex	de,hl
	jp	(hl)

;load
MONL:
	xor	a
	ld	(WITHAD+1),a
	call	SKIPSP
	or	a
	jr	z,MONLFN
	cp	':'
	jr	z,MONLZ1

	call	GETHEX2
	ret	nc
	ld	a,01h
	ld	(WITHAD+1),a

MONLZ1:
	call	SKIPSP
	cp	':'
	ld	b,00h
	jr	nz,MONLFN
	inc	hl
	ld	d,h
	ld	e,l
	dec	b
MONLLP:
	inc	b
	ld	a,(hl)
	inc	hl
	or	a
	jr	nz,MONLLP
MONLFN:
	push	de		;address
	ld	a,b
	call	SETFNAME
	call	LOADSRCH

	pop	de		;address

	dec	a
	jr	z,MONERR
	dec	a
	jr	z,MONBRK

	ld	a,(IBUFE)
	dec	a		;file attribute=1: binary
	jr	z,WITHAD
	ld	hl,ERR30	;bad file mode
	jp	PUTSNL
WITHAD:
	ld	a,00h		;with load address?
	or	a
	jr	z,MONLZ2
	ld	(DTADR),de
MONLZ2:
	call	RDD
	dec	a
	jr	z,MONERR
	dec	a
	jr	z,MONBRK
	ld	a,(ATSTRT)
	or	a
	ret	z
	ld	hl,(EXADR)
	jp	(hl)

MONERR:
	ld	hl,ERR29	;tape read error
	jr	MONERRSTOP
MONBRK:
	ld	hl,BREAK_STR
MONERRSTOP:
	call	PUTSNL
	jp	MONLP

;memory
MONM:
	call	GETHEX2
	jr	c,MONMLP
	ld	de,(MONDAD+1)
MONMLP:
	ld	hl,0007h
	add	hl,de
	push	hl
	call	MONDC
	ld	hl,(DSPXY)
	ld	de,0ff06h
	add	hl,de		;x+6,y-1
	call	SETCSR
	call	INPTSCR
	inc	hl
	call	MONSET
	pop	hl
	inc	hl
	ex	de,hl
	call	CHKBRK
	jr	nz,MONMLP
	jp	AFTBRK

;printer
MONP:
	ld	hl,PRNMOD
	ld	a,(hl)
	or	a
	jr	z,MONPOK
	ld	(hl),00h	;text mode
	ld	a,'A'
	call	PRINTER
MONPOK:
	ld	hl,PLOTON+1
	ld	a,(hl)
	cp	01h
	sbc	a,a		;0->ff,others->0
	ld	(hl),a
MONRET:
	ret

;return
MONR:
	ld	sp,0000h		;sp value before MON command
	ld	hl,INPBUF
	ld	(PROGAD),hl
	xor	a
	ld	(hl),a
	inc	hl
	ld	(hl),a
	inc	hl
	ld	(hl),a
	ld	(INMON),a
	ret

;save
MONS:
	call	GETHEX2
	ret	nc
	ld	(MONSSTRT+1),de
	ld	(MONSEX+1),de
	xor	a
	ld	(MONSAT+1),a
	ld	a,(hl)
	cp	' '
	ret	nz
	call	GETHEX2
	ret	nc
	ld	(MONSEND+1),de
	ld	a,(hl)
	cp	' '
	jr	nz,MONSNZ1
	call	GETHEX2
	jr	nc,MONSNZ1
	ld	(MONSEX+1),de
	ld	a,01h
	ld	(MONSAT+1),a
MONSNZ1:
	call	SKIPSP
	cp	':'
	ld	a,00h
	jr	nz,MONSNZ2
	inc	hl
	ld	d,h
	ld	e,l
	ld	b,0ffh
MONSLP:
	inc	b
	ld	a,(hl)
	inc	hl
	or	a
	jr	nz,MONSLP
MONSNZ2:
	ld	a,b
	call	SETFNAME
	ld	a,01h
	ld	(IBUFE),a	;file attribute=1: binary

MONSSTRT:
	ld	hl,0000h
	ld	(DTADR),hl
	ex	de,hl
MONSEND:
	ld	hl,0000h
	or	a
	sbc	hl,de
	inc	hl
	ld	(SIZE),hl
MONSEX:
	ld	hl,0000h
	ld	(EXADR),hl
MONSAT:
	ld	a,00h		;auto start?
	ld	(ATSTRT),a
	call	WRI
	jp	c,MONBRK
	call	WRD
	jp	c,MONBRK
	ret

;transfer
MONT:
	call	GETHEX2
	ret	nc
	ld	a,(hl)
	cp	','
	ret	nz
	inc	hl
	push	de		;source start
	call	GETHEX2
	pop	bc		;source start
	ret	nc
	ld	a,(hl)
	cp	','
	ret	nz
	push	bc		;source start
	push	de		;source end
	inc	hl
	call	GETHEX2		;destination
	pop	hl		;source end
	pop	bc		;source start
	ret	nc

	push	hl		;source end
	ld	h,b
	ld	l,c
	call	PUTHEX2
	ld	a,'-'
	call	PUTC
	pop	hl		;source end
	call	PUTHEX2
	ld	a,0c0h
	call	PUTC
	ex	de,hl
	call	PUTHEX2
	ex	de,hl
	call	PUTNL

	push	bc		;source start
	or	a
	sbc	hl,bc
	inc	hl
	ld	b,h		;source end - source start + 1
	ld	c,l
	pop	hl		;source start
	sbc	hl,de
	add	hl,de
	jr	c,MONTC

;destination < source start
	ldir
	ret

;source start < destination
MONTC:
	add	hl,bc
	dec	hl
	ex	de,hl
	add	hl,bc
	dec	hl
	ex	de,hl
	lddr
	ret


;verify
MONV:
	call	SKIPSP
	ld	b,0ffh
MONVLP:
	inc	b
	ld	a,(hl)
	inc	hl
	or	a
	jr	nz,MONVLP
	ld	a,b
	call	SETFNAME
	call	LOADSRCH
	dec	a
	jp	z,MONERR
	dec	a
	jp	z,MONBRK

	ld	a,(IBUFE)
	dec	a		;file attribute=1: binary
	jr	z,MONVZ
	ld	hl,ERR30	;bad file mode
	jp	PUTSNL
MONVZ:
	call	VRFY
	ret	nc
	dec	a
	jp	nz,MONBRK
	ld	hl,ERR29	;tape read error
	jp	PUTSNL


;input for MON command
;output: hl=INPBUF, c-flag(1=stop)
;destroy: af,bc,de
MONINPT:
	ld	a,'*'
	call	PUTC
	jp	INPTSCR


;put 2-byte hexadecimal value for MON command
;input: hl=data
;destroy: af
PUTHEX2:
	ld	a,h
	call	PUTHEX1
	ld	a,l
;	jr	PUTHEX1

;put 1-byte hexadecimal value for MON command
;input: a=data
;destroy: af
PUTHEX1:
	push	af
	rrca
	rrca
	rrca
	rrca
	call	PUTHEX
	pop	af
PUTHEX:
	and	0fh
	cp	0ah
	sbc	a,69h
	daa
	jp	PUTC


;get 1-byte hexadicmal value for MON command
;input: hl=input buffer address
;output: a=hex, hl=next address, c-flag(1=ok)
;destroy: af,bc
GETHEX1:
	call	SKIPSP
	ld	a,(hl)
	cp	';'
	jr	z,GETHEX1CHR
	xor	a
	ld	b,02h
GETHEX1LP:
	rlca
	rlca
	rlca
	rlca
	ld	c,a
	ld	a,(hl)
	inc	hl
	or	a
	ret	z
	sub	'0'
	cp	'9'-'0'+1
	jr	c,GETHEX1C
	sub	'A'-'0'
	cp	'F'-'A'+1
	ret	nc
	add	a,0ah
GETHEX1C:
	or	c
	djnz	GETHEX1LP
	scf
	ret

GETHEX1CHR:
	inc	hl
	ld	a,(hl)
	inc	hl
	scf
	ret


;get 2-byte hexadicmal value for MON command
;input: hl=input buffer address
;output: de=hex, hl=next address, c-flag(1=ok)
;destroy: af,b
GETHEX2:
	call	SKIPSP
	ld	de,0000h
	ld	b,04h
GETHEX2LP:
	ld	a,(hl)
	or	a
	ret	z
	sub	'0'
	cp	'9'-'0'+1
	jr	c,GETHEX2C
	sub	'A'-'0'
	cp	'F'-'A'+1
	ret	nc
	add	a,0ah
GETHEX2C:
	inc	hl
	ex	de,hl
	add	hl,hl		;*2
	add	hl,hl		;*4
	add	hl,hl		;*8
	add	hl,hl		;*16
	add	a,l
	ld	l,a
	ex	de,hl
	djnz	GETHEX2LP
	scf
	ret


;ELSE command
C_ELSE:
;	jr	C_REM

;REM command
C_REM:
	ld	c,00h
;	jr	DATREM

;input: c
DATREM:
	ld	b,00h
	call	SKIPTO
	jp	SETPRGAD

;	ld	bc,0000h
;	call	SKIPTO
;	jp	SETPRGAD


;LET command
C_LET:
	call	CHKGETV
	ld	hl,(PROGAD)
;	jp	VARIN

;put into variable
;input: b=variable name length, c=variable type,
;       de=variable address, hl=program address
;destroy: FAC1,FAC2,af,bc,de,hl
VARIN:
	call	SKIPSP
	cp	EQ_		;0f4h
	jp	nz,SNERR

	push	de		;variable address
	inc	hl
	ld	a,c
	cp	03h
	jr	z,VARSTR

;numerical
VARNUM:
	push	bc
	call	NARGMO
	pop	bc
	call	CNVTYP1
	ld	hl,FAC1+1
	ld	b,00h
	pop	de		;variable address
	ldir
	ret

;string
VARSTR:
	call	STRARG
	pop	hl		;variable address
	jp	MKSTRVAR


;with line number
RUNLN:
	call	RESSTK
	ld	hl,(PROGAD)
	call	GOTOMAIN
	ld	h,b
	ld	l,c
	jr	INTPRT

RUNEND:
	ld	a,(DURERRHND+1)
	or	a
	jp	z,EDIT
	jp	NRERR

;RUN command
C_RUN:
	call	SKIPSP
	cp	0bh
	jr	z,RUNLN		;with line number

	push	hl
	call	ARG
	pop	hl
	ld	a,(FAC1)
	cp	03h
	jr	nz,RUNBGN

;run with file name
RUNFN:
	xor	a
	call	LOADMAIN

;run from the beginning
RUNBGN:
	call	RESSTK
	ld	hl,(BASICAD)
;	jr	RUNLP

RUNLP:
	ld	a,(hl)
	inc	hl
	or	(hl)
	jr	z,RUNEND

	inc	hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	ld	(LINENUM),de

	ld	a,(TRACEON)	;0=trace off, not0=trace on
	or	a
	call	nz,TRACE

	inc	hl
INTPRT:
	call	STOPESC
	jp	z,STOP1

DURERRHND:
	ld	a,00h		;not0=during error handling
	or	a
	jr	nz,INTPRTNZ
	ld	(RESM0+1),hl
INTPRTNZ:
	ld	a,(hl)
	inc	hl
	ld	(PROGAD),hl

	or	a
	jp	m,COMMAND
	jr	z,RUNLP
	cp	':'
	jr	z,INTPRT
	cp	' '
	jr	z,INTPRT
	cp	27h		;apostrophe
	jr	z,APOS

	dec	hl
	cp	22h		;double quotation mark
	jp	z,GSUB
	call	C_LET
	jp	INTPEND

APOS:
	ld	a,REM_
COMMAND:
	cp	0feh
	jr	nc,COMMAND2
	cp	TO_
	ld	de,CMDTBL
	jp	c,CALLCMD
	jp	SNERR

COMMAND2:
	inc	a		;
	ld	a,(hl)
	inc	hl
	ld	(PROGAD),hl
	jr	nz,CMDFE	;
;ff
	ld	bc,INTPEND
	push	bc
	cp	MID_
	jp	z,C_MID
	cp	MEM_
	jp	z,C_MEM
	cp	TIME_
	jp	z,C_TIME
	jp	SNERR

CMDFE:
	cp	DEVO_+1
	jp	nc,SNERR
	ld	de,CMDTBLFE

CALLCMD:
	call	JPTBL
INTPEND:
	ld	hl,(TMPSTR)
	ld	(FREEAD),hl
	ld	hl,(PROGAD)
;	call	CHKCLN
;	jp	nz,SNERR
	jp	INTPRT


;FOR command
C_FOR:
	pop	af		;cancel return address
	call	CHKGETV
	ld	hl,(PROGAD)

	push	de		;variable value address
	push	bc		;c=variable type
	call	VARIN
	ld	hl,(PROGAD)
	call	CHKNUM
	ld	a,(hl)
	cp	TO_
	jp	nz,SNERR
	call	NARGMOINC
	pop	bc		;c=variable type
	push	hl
	call	CNVTYP1
	pop	de
	call	PUSHF1		;TO value
	ex	de,hl
	ld	a,(hl)
	cp	STEP_
	ld	(PROGAD),hl
	push	af
	call	nz,SETPLS1
	pop	af
	jr	nz,STEP1
	push	bc		;c=variable type
	call	NARGMOINC
	pop	bc		;c=variable type
	call	CNVTYP1
STEP1:
	call	POPF2		;TO value
	pop	de		;variable value address

	ld	hl,(PROGAD)
	push	hl		;address after FOR
	ld	hl,0002h	;address after FOR(2)
	add	hl,sp
FORLP1:
	ld	a,FOR_
	call	SRCHSTK2
	jr	nc,FOROK	;not found or gosub identifer found
	jr	z,FORZ		;for
;while,repeat
	ld	bc,0005h
	add	hl,bc
	jr	FORLP1

FORZ:
;check program address in stack
	ld	bc,0019h	;identifier(1)+varibale value address(2)+STEP(10)+TO(10)+line number(2)
	add	hl,bc
	ld	c,(hl)		;address after FOR
	inc	hl
	ld	b,(hl)
	inc	hl
	ex	(sp),hl
	or	a
	sbc	hl,bc		;compare address after FOR
	add	hl,bc
	ex	(sp),hl
	jr	nz,FORLP1

	ld	sp,hl
	push	bc		;address after FOR

FOROK:
	ld	hl,(LINENUM)
	push	hl		;line number(2)
	call	PUSHF2		;TO value(10)
	call	PUSHF1		;STEP value(10)
	push	de		;variable value address(2)
	ld	a,FOR_		;identifer(1)
	push	af
	inc	sp

;skip for-next loop?
	call	CHKSGN
	push	af
	ex	de,hl		;variable value address

	ld	de,FAC1+1
	ld	a,(FAC1)
	ld	c,a
	ld	b,00h
	ldir

	pop	af
	jr	z,FORSTEP0	;if STEP value = 0
	call	nc,EXFAC	;if STEP value > 0
	call	CMPF
	jr	c,FORSKIP	;skip for-next loop
;	jp	RECSP1

;record stack pointer for resume
RECSP1:
	ld	hl,(PROGAD)
RECSP2:
	ld	(SPERR+1),sp	;sp for resume
	jp	INTPRT

FORSTEP0:
	call	CMPF
	jr	nz,RECSP1

;skip for-next loop
FORSKIP:
	ld	hl,(PROGAD)
	ld	a,01h
	ld	(NESTCNT+1),a
	ld	bc,FOR_*256+NEXT_
FORLP2:
	call	SKIPNEST
	or	a
	jr	nz,NEXTFD	;NEXT found
	jp	FNERR

;another for-next loop
FORLP3:
	call	SKIPSPINC
	sub	'A'
	cp	'Z'-'A'+1
	jr	nc,FORLP2

	call	CHKVAR
	ld	hl,(PROGAD)
	call	SKIPSP
	cp	','
	jr	nz,FORLP2
;	jr	NEXTFD

;NEXT found
NEXTFD:
	ld	a,(NESTCNT+1)
	dec	a
	ld	(NESTCNT+1),a
	jr	nz,FORLP3	;another for-next loop

;corresponding next
	inc	sp		;identifier(1)
	call	SKIPSPINC
	sub	'A'
	cp	'Z'-'A'+1
	ld	(PROGAD),hl
	jr	nc,JPNXTEND

;compare variable
	call	CHKGETV
	pop	hl		;variable value address
	or	a
	sbc	hl,de
	jp	nz,NFERR
	push	hl		;dummy
JPNXTEND:
	jp	NEXTEND


;NEXT command
C_NEXT:
	pop	af		;cancel return address
	call	SKIPSP
	ld	(PROGAD),hl
	sub	'A'
	cp	'Z'-'A'+1
	ld	d,00h
	jr	nc,NEXTVLP	;no parameter

NEXTLP:
	inc	hl
	call	GETTYPE
	call	GETVAR

NEXTVLP:
	ld	a,FOR_
	call	SRCHSTK
	jp	nc,NFERR	;not found or gosub identifier found
	jr	nz,NEXTNZ	;found WHILE/REPEAT in stack
	inc	sp		;identifer
	pop	hl		;variable value address (FOR stack)
	ld	a,d		;variable value address (parameter)
	or	a
	jr	z,NEXTOK	;without vairable parameter
;	or	a
	sbc	hl,de
	add	hl,de
	jr	z,NEXTOK

;found other for-next loop
	ld	hl,0018h	;STEP(10)+TO(10)+line number(2)+program address(2)
	add	hl,sp
	ld	sp,hl
	jr	NEXTVLP

;found WHILE/REPEAT in stack
NEXTNZ:
	ld	hl,0005h
	add	hl,sp
	ld	sp,hl
	jr	NEXTVLP

NEXTOK:
;STEP->FAC2
	push	hl		;variable value address
	ld	hl,0003h
	ld	b,h		;=0
	add	hl,sp
	ld	a,(hl)		;type
	ld	de,FAC2
	ld	c,a
	ldir
	inc	c
	ldi
	pop	hl		;variable value address

	ld	de,(FAC2+1)
	push	hl		;variable value address
	push	de		;e=STEP exponent, d=STEP sign

;var->FAC1
	ld	de,FAC1
	ld	(de),a		;type
	inc	e		;inc de
;	ld	b,00h
	ld	c,a
	push	bc
	ldir

;var+STEP->FAC1
	call	ADDF
	pop	bc

	ld	a,(FAC1)	;type
	cp	c
	jp	nz,OVERR

;FAC1->var
	pop	hl		;l=STEP exponent, h=STEP sign
	pop	de		;variable value address
	push	de		;variable value address
	push	hl		;l=STEP exponent, h=STEP sign

	ld	hl,FAC1+1
	ldir

;TO->FAC2
	ld	hl,0010h
	add	hl,sp		;TO value
	ld	de,FAC2+1
;	ld	a,(FAC1)
;	ld	b,00h
	ld	c,a		;type
	ldir

;STEP sign
	pop	hl		;l=STEP exponent, h=STEP sign
	cp	02h
	ld	a,h		;integer
	jr	z,NEXTINT
	ld	a,l		;single or double
NEXTINT:
	or	l		;z-flag
	ld	a,h
	rlca			;c-flag
	jr	c,STEPMNS
	jr	z,STEP0

;STEP>0
STEPPLS:
	call	CMPF
	jp	c,NEXTCONT
	jr	z,NEXTCONT

NEXTEND:
	ld	hl,001ah	;FOR command
				;variable value address(2)+STEP(10)+TO(10)+line number(2)+program address(2)
	add	hl,sp
	ld	sp,hl

	ld	hl,(PROGAD)
	call	SKIPSP
	ld	(PROGAD),hl
	cp	','
	jp	nz,RECSP2
	call	SKIPSPINC
	sub	'A'
	cp	'Z'-'A'+1
	jp	c,NEXTLP
	jp	SNERR

;STEP=0
STEP0:
	call	CMPF
	jr	nz,NEXTCONT
	jr	NEXTEND

;STEP<0
STEPMNS:
	call	CMPF
	jr	c,NEXTEND

NEXTCONT:
	ld	hl,0016h	;variable value address(2)+STEP(10)+TO(10)
	add	hl,sp
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	ld	(LINENUM),de
	inc	hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	ex	de,hl		;address after FOR
	ld	a,FOR_
	push	af		;identifier
	inc	sp
	jp	INTPRT


;WRITE command
C_WRIT:
	ld	a,01h
	ld	(WRFLG+1),a
	jr	PRTFNUM

;LPRINT command
C_LPRT:
	call	SETPRN
;	jr	C_PRT

;PRINT command
C_PRT:
	xor	a
	ld	(WRFLG+1),a
PRTFNUM:
	xor	a
	ld	(PRTSYM+1),a
	ld	(WITHUS+1),a

	call	SKIPSP
	cp	'#'
	jr	nz,PRTLP1
	inc	hl
	call	INT1ARG
	call	SKIPSP
	cp	','
	jp	nz,SNERR
	inc	hl
	push	hl
	ld	a,(FILES)
	cp	e		;e=file number
	jp	c,FCERR
	ld	a,e
	or	a
	ld	hl,0000h
	jr	z,PRTZ1		;print #0
	call	GETBUFP
	jp	z,CFERR
PRTZ1:
	ld	(BUFP),hl
	pop	hl
	ld	a,01h
	ld	(PRTSYM+1),a

PRTLP1:
	xor	a
	ld	(NONL+1),a
	call	SKIPSP
PRTLP2:
	inc	hl
	cp	';'
	jp	z,PRTNEXT
	cp	','
	jp	z,PRTCMM
	ld	b,a
	ld	a,(WRFLG+1)	;WRITE command flag
	or	a
	jr	nz,PRTNZ
	ld	a,b
	cp	USING_
	jp	z,PRTUS
	cp	TAB_
	jp	z,PRTTAB
	sub	SPC_
	jp	z,PRTSPC
PRTNZ:
	dec	hl
	call	ARG
	ld	a,(FAC1)
	cp	03h
	jr	z,PRTSTR
	inc	a
	jp	z,PRTEOL
	jp	PRTNUM

;string
PRTSTR:
	ld	a,(WITHUS+1)
	or	a
	jp	nz,PRTSTRU
	ld	a,(WRFLG+1)	;WRITE command flag
	or	a
	push	af		;
	ld	a,22h
	call	nz,PUTC
	call	STRARG2
	ld	b,a
	call	PUTSPRT
PRTSEND:
	pop	af		;WRITE command flag
	ld	a,22h
	call	nz,PUTC
	ld	hl,(PROGAD)
	jr	PRTLP1


;put string for PRINT command
;input: b=length, de=address
;destroy: af,b,de
PUTSPRT:
	ld	a,b
	or	a
	ret	z
PRTSYM:
	ld	a,00h		;print symbol?
	or	a		;
	ld	a,(de)
	inc	de
	jr	z,PRTSTRZ	;

;print symbol
	cp	20h
	jr	nc,PRTSYMPUT
	call	BACN
	jr	z,PRTSYMPUT
	xor	a
PRTSYMPUT:
	call	PUTCH
	jr	PRTDJNZ

PRTSTRZ:
	call	PUTC
PRTDJNZ:
	djnz	PRTSYM
	ret


PRTNUM:
	call	CHKNUM
WITHUS:
	ld	a,00h		;USING?
	or	a
	jp	nz,PRTNUMU
	push	hl
	call	FTOA

	ld	a,WIDTH
	add	a,02h
	ld	c,a
	xor	a
	cpir
	ld	a,(DSPXY)
	cp	c
	call	nc,PUTNL
	ld	hl,FAC3
	ld	a,(WRFLG+1)	;WRITE command flag
	or	a
	push	af
	jr	z,PRTNUMZ
	ld	a,(hl)
	cp	' '
	jr	nz,PRTNUMZ
	inc	hl
PRTNUMZ:
	call	PUTS
	pop	af
	ld	a,' '
	call	z,PUTC
	pop	hl
	jp	PRTLP1

;comma
PRTCMM:
	ld	a,(WRFLG+1)	;WRITE command flag
	or	a
	jr	nz,PRTNEXT
	ld	a,(WITHUS+1)
	or	a
	jr	nz,PRTNEXT
	call	CMMMAIN
	call	nc,PUTNL

PRTNEXT:
WRFLG:
	ld	a,00h		;WRITE command flag
	or	a
	ld	a,','
	call	nz,PUTC
	ld	(NONL+1),a	;>0
	ld	a,(hl)
	jp	PRTLP2

CMMMAIN:
	ld	e,0ah
	ld	a,(SELPRN)
	or	a
	jr	z,CMM0

;printer
CMM1:
	ld	b,0ah
TAB1:
	ld	a,(PRNPOS)
TABLP:
	sub	e
	jr	c,PUTSPNEG
	djnz	TABLP
	ret

;CRT
CMM0:
	ld	b,03h
TAB0:
	ld	a,(DSPXY)
	jr	TABLP

PUTSPNEG:
	neg
PUTSP:
	ld	b,a
PUTSPLP:
	ld	a,' '
	call	PUTC
	djnz	PUTSPLP
	scf
	ret

PRTTAB:
	call	FNCNUM
	push	hl
	call	FTOI1
	pop	hl

	ld	b,01h
	ld	a,(SELPRN)
	or	a
	ld	a,(DSPXY)
	jr	z,PRTTABZ
	ld	a,(PRNPOS)
PRTTABZ:
	call	TABLP
	jr	PRTNEXT

PRTSPC:
	call	FNCNUM
	push	hl
	call	FTOI1
	pop	hl
	ld	a,e
	or	a
	call	nz,PUTSP
	jp	PRTLP1

PRTEOL:
NONL:
	ld	a,00h		;no new line
	or	a
	jr	nz,PRTEND
	ld	a,(PRTSPD)	;SPEED command
	or	a
	jr	z,PRTEOLZ
	ld	e,a
PRTEOLLP:
	ld	bc,15d8h
	call	WAIT
	dec	e
	jr	nz,PRTEOLLP
PRTEOLZ:
	call	PUTNL
PRTEND:
	ld	(PROGAD),hl

;print#/input# end
PIEND:
	xor	a
	ld	(SELPRN),a
	jp	CLRBUFP


;PRINT USING
PRTUS:
	call	STRARG
	or	a
	jp	z,FCERR
	ld	(WITHUS+1),a	;>0
	push	hl
	ld	hl,(FBUFAD)
	ld	(USNGAD+1),hl
	ex	de,hl
	ld	b,00h
	ld	c,a
	ldir
	xor	a
	ld	(de),a
	pop	hl
	ld	a,(hl)
	cp	','
	jr	z,PRTUSOK
	cp	';'
	jp	nz,SNERR
PRTUSOK:
	inc	hl
	push	hl
	call	ARG
	pop	hl
	jp	z,MOERR
	jp	PRTNEXT


;print string with using
PRTSTRU:
	push	hl		;program address
	ld	hl,(USNGAD+1)
	call	CHKUS
	jr	c,STRUOK	;string
	or	a
	jp	z,SNERR
	jp	TMERR

STRUOK:
	inc	hl
	ld	c,01h
STRULP:
	inc	c
	ld	a,(hl)
	inc	hl
	cp	' '
	jr	z,STRULP
	cp	'&'
	jp	nz,SNERR

	ex	(sp),hl		;program address
	push	bc
	call	STRARG2
	pop	bc
	ld	b,a
	ld	a,c
	sub	b
	jr	nc,STRUNC
	ld	b,c
	xor	a
STRUNC:
	ld	c,a		;
	call	PUTSPRT
	ld	a,c		;
	or	a
	call	nz,PUTSP
	ex	(sp),hl		;USING-string address

STRUEND:
	call	CHKUS
	or	a
	jr	nz,STRUNZ
	ld	hl,(FBUFAD)
STRUNZ:
	ld	(USNGAD+1),hl
	pop	hl		;program address
	jp	PRTLP1


;print number with USING
PRTNUMU:
	push	hl		;program address

	xor	a
	ld	(ALRDYPUT+1),a
	ld	(WITHEXP+1),a
	ld	(WITHYEN+1),a
	ld	(WITHCMM+1),a
	ld	(TAILSGN+1),a
	ld	h,a
	ld	l,a
	ld	(NUMUEXP+1),hl
	ld	a,' '
	ld	(NUMUSPC+1),a

	ld	a,(FAC1)
	cp	02h
	call	z,ITOS
	ld	a,(FAC1+1)	;exponent
	ld	(NUMURD+1),a

USNGAD:
	ld	hl,0000h	;USING-sring address
	call	CHKUS
	jp	c,TMERR
	or	a
	jp	z,SNERR
	ld	(NUMUFRST+1),a	;first character

	cp	'-'
	jr	z,NUMUMNS
	cp	'+'
	jr	nz,NUMUNZ1
;+
	inc	hl
	ld	a,(hl)

NUMUNZ1:
	cp	'*'
	jr	z,NUMUAST
	cp	7dh		;yen mark
	jr	nz,NUMUCNT1	

;\\
NUMUYEN:
	inc	hl
NUMUYEN2:
	ld	(WITHYEN+1),a
;-
NUMUMNS:
	inc	hl
	ld	a,(hl)
	jr	NUMUCNT1

;**
NUMUAST:
	ld	(NUMUSPC+1),a
	inc	hl
	inc	hl
	ld	a,(hl)
	cp	7dh		;yen mark
	jr	z,NUMUYEN2	;**\
;	jr	NUMUCNT1

;count # before .
NUMUCNT1:
	ld	d,00h
	cp	'.'
	jr	z,NUMUSPC

NUMULP1:
	inc	d
	inc	hl
	ld	a,(hl)
	cp	'#'
	jr	z,NUMULP1
	cp	','
	jr	nz,NUMUSPC
	ld	(WITHCMM+1),a	;>0
	jr	NUMULP1

;check ** and yen
NUMUSPC:
	ld	a,' '		;' ' or '*'
	cp	'*'
	jr	z,NUMUZ1
	ld	a,(WITHYEN+1)
	or	a
	jr	z,NUMUCNT2
	dec	d
NUMUZ1:
	inc	d
	inc	d

;count # after .
NUMUCNT2:
	ld	e,00h
	ld	a,(hl)
	cp	'.'
	jr	nz,NUMUCHKEXP
NUMULP2:
	inc	e
	inc	hl
	ld	a,(hl)
	cp	'#'
	jr	z,NUMULP2

;check exponential
NUMUCHKEXP:
	push	hl
	ld	b,04h
NUMULP3:
	ld	a,(hl)
	inc	hl
	cp	'^'
	jr	nz,NUMUNZ2
	djnz	NUMULP3
	ld	(WITHEXP+1),a	;>0
	xor	a
	ld	(WITHCMM+1),a
	ex	(sp),hl
NUMUNZ2:
	pop	hl
	ld	a,(hl)
	cp	'+'
	jr	z,NUMUZ2
	cp	'-'
	jr	nz,NUMUNZ3
NUMUZ2:
	ld	(TAILSGN+1),a
	inc	hl
	ld	a,(hl)
NUMUNZ3:
	ld	(USNGAD+1),hl

	push	hl
	ld	hl,FAC1+2
	ld	a,(hl)
	ld	(NUMUSGN+1),a
	res	7,(hl)		;negative->positive
	pop	hl
	rlca
	jr	nc,NUMUEXP10
	ld	a,(NUMUFRST+1)
	cp	'+'
	jr	z,NUMUEXP10
	cp	'-'
	jr	z,NUMUEXP10

	ld	a,(TAILSGN+1)
	or	a
	jr	nz,NUMUEXP10
	ld	a,d
	or	a
	jp	z,FOERR
	dec	d

;push 1,10,100,...,10^d
NUMUEXP10:
	call	CPYFAC		;target value
	ld	b,d

WITHCMM:
	ld	a,00h		;with comma?
	or	a
	jr	z,NUMUZ3
;with comma
	ld	a,d
	and	0fch
	rrca
	rrca
	ld	b,a
	ld	a,d
	sub	b
	ld	b,a

NUMUZ3:
	push	bc
	push	de
	call	SETPLS1
	pop	de
	pop	bc
	inc	b
NUMULP4:
	call	PUSHF1		;10^0, 10^1,...
	push	bc
	push	de
	call	PUSHF2		;
	call	MUL10
	call	POPF2		;
	pop	de
	pop	bc
	djnz	NUMULP4

WITHEXP:
	ld	a,00h		;with exponent?
	or	a
	jr	z,NUMUNOEXP

	ld	a,(FAC2+1)	;exponent
	or	a
	jr	z,NUMUNOEXP	;if 0

;alignment for ^^^^
	call	POPF1		;10^d
	call	PUSHF1

	push	de

;alignment for >= 10^(d-1)
NUMUAL1:
	call	PUSHF2
	call	DIV10		;10^(d-1)
	call	CPYFAC
	call	POPF1
	ld	bc,0000h
	push	bc
NUMULP5:
	call	CMPF
	pop	bc
	jr	c,NUMUAL2
	inc	b
	push	bc
	call	PUSHF2
	call	DIV10
	call	POPF2
	jr	NUMULP5

;fac1=target/(10^b), fac2=10^(d-1)

;alignment for < 10^(d-2)
NUMUAL2:
	push	bc
	call	PUSHF1		;
	call	EXFAC
	call	DIV10		;10^(d-2)
	call	CPYFAC
	call	POPF1		;

NUMULP6:
	call	CMPF
	pop	bc
	jr	nc,NUMUALEND
	inc	c
	push	bc
	call	PUSHF2
	call	MUL10
	call	POPF2
	jr	NUMULP6

;fac1=target*(10^c), fac2=10^(d-2)

NUMUALEND:
	ld	(NUMUEXP+1),bc
	call	CPYFAC
	pop	de

NUMUNOEXP:
	ld	a,(WITHCMM+1)
	or	a
	jr	z,NUMURD
	ld	a,d
	and	03h
	jr	nz,NUMURD
;for ,
	call	POPF1		;10^3 or 10^6 or ...
	call	PUSHF1
	call	PUSHF1

;for rounding
NUMURD:
	ld	a,00h		;exponent
	cp	0a0h
	jr	c,NUMURDC
	push	de
	call	EXFAC		;FAC1->FAC2
	pop	de
	jr	CHKFOERR

NUMURDC:
	call	PUSHF2		;
	ld	hl,HALFS
	push	de
	call	SETF1
	pop	de
	ld	b,e
	ld	a,e
	cp	02h
	jr	c,NUMRDEND
	dec	b
NUMULP7:
	push	bc
	push	de
	call	DIV10
	pop	de
	pop	bc
	djnz	NUMULP7
NUMRDEND:
	call	POPF2		;
	push	de
	call	ADDF
	pop	de

;check format over
CHKFOERR:
	call	POPF2		;10^d
	push	de
	call	CMPF
	jp	nc,FOERR
	pop	bc		;d->b

;put before .
;divide by 10^(b-1),10^(b-2),...
	ld	a,b
	or	a
	jr	nz,NUMULP8

;USING ".###..."
	ld	a,c
	dec	a
	jp	z,FCERR

	ld	a,(NUMUSGN+1)
	rlca
	jr	c,PUTMNS
;+
	ld	a,(NUMUFRST+1)
	cp	'+'
	jr	z,PUTPUTDOT
	cp	'-'
	ld	a,(NUMUSPC+1)
	jr	z,PUTPUTDOT
	jp	PUTDOT
;-
PUTMNS:
	ld	a,(TAILSGN+1)
	or	a
	jp	nz,PUTDOT
	ld	a,'-'
PUTPUTDOT:
	call	PUTC
	jp	PUTDOT

NUMULP8:
	call	POPF2		;10^n
	call	PUSHF1		;target value
	call	PUSHF2		;10^n

	push	bc
	call	DIVF
	call	FIX		;dividend
	call	FTOI1

ALRDYPUT:
	ld	a,00h		;already put figure?
	or	a
	jr	z,NUMUZ4

;already put figure
	ld	a,(WITHCMM+1)
	or	a
	ld	a,e
	jp	z,PUTFIG

	pop	af
	push	af
	and	03h
	ld	a,e
	jp	nz,PUTFIG

	pop	bc
	dec	b
	push	bc
	ld	a,','
	jr	PUTPUTFIG

;not put figure yet
NUMUZ4:
	ld	a,e
	or	a
	jr	z,NUMU0
	ld	(ALRDYPUT+1),a

;put first figure
NUMUSGN:
	ld	a,00h		;sign
	rlca
NUMUFRST:
	ld	a,00h		;first character
	jr	nc,NUMUSGNP	;sign

;negative
	cp	'+'
	jr	z,NUMUZ5
	cp	'-'
	jr	z,PUTPUTFIG
	ld	a,(TAILSGN+1)
	or	a
	jr	nz,WITHYEN
NUMUZ5:
	ld	a,'-'
PUTPUTFIG:
	call	PUTC
WITHYEN:
	ld	a,00h		;with yen mark?
	or	a
	call	nz,PUTC
	ld	a,e
	jr	PUTFIG

;positive
NUMUSGNP:
	cp	'+'
	jr	z,PUTPUTFIG
	cp	'-'
	ld	a,(NUMUSPC+1)
	jr	z,PUTPUTFIG
	jr	WITHYEN

;if '0'
NUMU0:
	pop	af
	push	af
	dec	a
	jr	z,NUMU0DOT

	ld	a,(WITHCMM+1)
	or	a
	ld	a,(NUMUSPC+1)
	jr	z,NUMUPUT
	pop	af		;a=digits before .
	push	af
	and	03h
	dec	a
	jr	nz,NUMUNZ4
	pop	bc		;b=digits before .
	dec	b
	push	bc
	ld	a,(NUMUSPC+1)
	call	PUTC
NUMUNZ4:
	ld	a,(NUMUSPC+1)
	jr	NUMUPUT

NUMU0DOT:
;0.
	ld	a,(NUMUSGN+1)
	rlca
	jr	nc,NUMUPLS0
;-0.
	ld	a,(NUMUFRST+1)
	cp	'+'
	jr	z,NUMUZ6
	cp	'-'
	jr	z,PUTPUT0
	ld	a,(TAILSGN+1)
	or	a
	jr	nz,PUT0
NUMUZ6:
	ld	a,'-'
	jr	PUTPUT0

;+0.
NUMUPLS0:
	ld	a,(NUMUFRST+1)
	cp	'+'
	jr	z,PUTPUT0

	cp	'-'
	jr	nz,PUT0

	ld	a,' '

PUTPUT0:
	call	PUTC
PUT0:
	ld	a,(WITHYEN+1)
	or	a
	call	nz,PUTC
	xor	a

;put figure (a=0-9)
PUTFIG:
	add	a,'0'
NUMUPUT:
	call	PUTC

	pop	bc
	call	POPF2		;10^n

	push	bc
	call	MULF
	pop	bc

	call	POPF2		;target value
	push	bc
	call	SUBF21		;remainder
	pop	bc
	dec	b
	jp	nz,NUMULP8

;put after .
	ld	a,c
	or	a
	jr	z,NUMUTAIL

PUTDOT:
	ld	a,'.'
	call	PUTC
	dec	c
	jr	z,NUMUTAIL

	call	PUSHF1		;target value
	push	bc
	call	SETPLS1
	pop	bc
	call	POPF2		;

;FAC1=10^-n, FAC2=target value
	ld	b,c
NUMULP9:
	call	PUSHF2		;
	push	bc
	call	DIV10		;10^-n
	pop	bc
	call	CPYFAC
	call	POPF1		;

	call	PUSHF2		;10^-n
	call	PUSHF1		;
	call	PUSHF2		;10^-n
	push	bc
	call	DIVF
	call	FIX
	call	FTOI1
	ld	a,e
	add	a,'0'
	call	PUTC
	pop	bc
	call	POPF2		;10^-n
	push	bc
	call	MULF
	pop	bc
	call	POPF2		;
	push	bc
	call	SUBF21
	pop	bc
	call	CPYFAC
	call	POPF1		;10^-n
	djnz	NUMULP9	

NUMUTAIL:
	ld	a,(WITHEXP+1)
	or	a
	jr	z,TAILSGN
;exp
	ld	a,'E'
	call	PUTC
NUMUEXP:
	ld	bc,0000h
	ld	a,c
	or	a
	ld	a,'+'
	jr	z,NUMUEXPPUT
	ld	a,'-'
	ld	b,c
NUMUEXPPUT:
	call	PUTC

	ld	a,b
	cp	100
	jp	nc,FOERR
	call	DIVI10
	call	PUTC
	ld	a,c
	call	PUTC

TAILSGN:
	ld	a,00h
	or	a
	jr	z,NUMUEND
	cp	'+'
	ld	a,(NUMUSGN+1)
	rlca
	ld	a,'-'
	jr	c,TAILSGNPUT
	ld	a,'+'
	jr	z,TAILSGNPUT
	ld	a,' '
TAILSGNPUT:
	call	PUTC

NUMUEND:
	ld	hl,(USNGAD+1)
	jp	STRUEND

;0.5
HALFS:
	db	05h, 80h, 00h, 00h, 00h, 00h


;check using string
;input: hl=address
;output: a=data, hl=address, c-flag(1=string)
CHKUS:

	ld	a,(hl)
	or	a
	ret	z

	push	hl
	cp	'&'
	jr	z,CHKUSSTR
	cp	'-'
	jr	z,CHKUSMNS
	cp	'+'
	jr	nz,CHKUSNZ
	inc	hl
	ld	a,(hl)
CHKUSNZ:
	cp	'*'
	jr	z,CHKUSAST
	cp	7dh		;yen mark
	jr	nz,CHKUSNUM

;yen-yen
CHKUSYEN:
	inc	hl
	cp	(hl)
	jr	z,CHKUSMNS

CHKUSPUT:
	pop	hl
CHKUSPUT2:
	ld	a,(hl)
	call	PUTC
	inc	hl
	jr	CHKUS

;**
CHKUSAST:
	inc	hl
	cp	(hl)
	jr	nz,CHKUSPUT
	inc	hl
	ld	a,(hl)
	cp	7dh		;yen mark
	jr	nz,CHKUSNUM

;-
CHKUSMNS:
	inc	hl
	ld	a,(hl)

;# or .
CHKUSNUM:
	cp	'#'
	jr	z,CHKUSEND
	cp	'.'
	jr	nz,CHKUSPUT
CHKUSEND:
	pop	hl
	ld	a,(hl)
;	or	a		;reset c-flag
	ret

;& &
CHKUSSTR:
	inc	hl
	ld	a,(hl)
	cp	' '
	jr	z,CHKUSSTR
	pop	hl
	cp	'&'
	ld	a,(hl)
	scf
	ret	z
	jr	CHKUSPUT2


;PAUSE command
C_PAUS:
	call	INT1ARG
	or	a
	ret	z
PAUSLP:
	ld	bc,15d8h
	call	WAIT
	dec	e
	jr	nz,PAUSLP
	ret


;SPEED command
C_SPED:
	call	NUMARG
	ld	e,00h
	call	nz,FTOI1	;if with numerical parameter
	ld	a,e
	ld	(PRTSPD),a
	ret


;STOP command
C_STOP:
	pop	af		;cancel return address
	call	CHKCLN
	jp	nz,SNERR

STOP1:
	ld	de,(LINENUM)
	ld	a,d
	and	e
	inc	e
	jr	z,STOPZ
STOP2:
	ld	(STOPAD),hl
STOP3:
	ld	hl,(LINENUM)
	ld	(STOPLN),hl
STOPZ:
	call	CLRBUFP
	xor	a
	ld	(SELPRN),a
	call	PUTNLX
	ld	hl,BREAK_STR
	call	PUTS
	jp	PUTLNUM

BREAK_STR:
	db	"BREAK", 00h
;	db	07h, "BREAK", 00h

;reset sp and stop
STOPSP:
	ld	sp,(STACK)
	ld	hl,0000h
	jr	STOP2


;CONT command
C_CONT:
	pop	af		;cancel return address
	ld	de,BASINI
	or	a
	sbc	hl,de
	jp	nc,CNERR	;cont in program

	ld	hl,(STOPAD)
	ld	a,h
	or	a
	jp	z,CNERR		;stop in direct command mode

	xor	a
	ld	(STOPAD+1),a	;higher byte
	ld	de,(STOPLN)
	ld	(LINENUM),de
	jp	INTPRT


;ERASE command
C_ERAS:
;GET command
C_GET:
;LSET command
C_LSET:
;NAME command
C_NAME:
;OFF command
C_OFF:
;PUSH command
C_PUSH:
;POP command
C_POP:
;RANDOMIZE command
C_RNDM:
;PUT command
C_PUT:
;RSET command
C_RSET:
;TRACE command
C_TRAC:
;VDIM command
C_VDIM:
	jp	RFERR


;ERROR command
C_ERR:
	pop	af		;cancel return address
	call	INT1ARG
	ld	b,a		;
	call	CHKCLN
	jp	nz,SNERR
	ld	a,b		;
	jp	ERROR


;END command
C_END:
	pop	af		;cancel return address
	call	CHKCLN
	jp	nz,SNERR
	call	CLOSALL
	jp	EDIT


;AUTO command
C_AUTO:
	pop	af		;cancel return address
	ld	bc,000ah	;2nd parameter
	call	GETLNSP		;1st parameter
	jr	z,AUTOP1OK
	ld	de,000ah
AUTOP1OK:
	call	GETCMM
	jr	nz,AUTONZ
AUTOPAR2:
	push	de		;1st parameter
	call	GETLNSP		;2nd parameter
	jp	nz,SNERR
	ld	b,d
	ld	c,e
	pop	de

AUTONZ:
	call	CHKCLN
	jp	nz,SNERR	

AUTOLP:
	push	bc		;2nd parameter
	push	de		;1st parameter
	ex	de,hl
	call	PUTI2
	pop	de
	push	de
	call	SRCHLN
	sbc	a,a
	and	'*'-' '
	add	a,' '		;'*' for c-flag=1, ' ' for c-flag=0
	call	PUTC
	call	INPT1
	jp	c,EDIT
	scf			;with line number
	call	CNVIL
	ld	hl,INPBUF
	or	a
	sbc	hl,bc
	jr	z,AUTOZ
	pop	de		;line number
	push	de		;line number
	call	CHGPRG
AUTOZ:
	pop	hl		;line number
	pop	bc		;2nd parameter
	inc	hl
	add	hl,bc
	jp	c,EDIT		;>=65535
	dec	hl
	ex	de,hl
	jr	AUTOLP


;NEW command
C_NEW:
	call	SKIPSP
	cp	ON_
	jr	nz,NEWEDIT
	inc	hl
	call	NUMARG
	ld	de,BASINI
	call	nz,FTOI2	;if with numerical parameter
	ld	hl,BASINI-1
	or	a
	sbc	hl,de
	jp	nc,FCERR
	ld	hl,(STACK)
	ld	bc,STKMGN
	or	a
	sbc	hl,bc
	sbc	hl,de
	jp	c,FCERR

	ld	(BASICAD),de
NEWEDIT:
	ld	hl,EDIT
	push	hl

;destroy: af,bc,de,hl
NEWRESSTK:
	ld	hl,RESSTK
	push	hl
;	jp	NEW

;destroy: af,hl
NEW:
	ld	hl,(BASICAD)
	dec	hl
	xor	a
	ld	(hl),a
	inc	hl
	ld	(hl),a
	inc	hl
	ld	(hl),a
	inc	hl
	ld	(VARAD),hl
	ret


;MAXFILES command
C_MAXF:
	call	INT1ARG
	cp	10h
	jp	nc,FCERR
	ld	(FILES),a
	jp	C_CLR


;CLR command
C_CLR:
	xor	a
	ld	(BASE+1),a	;option base 0
	ld	(STOPAD+1),a	;=0,higher byte
	ld	hl,(VARAD)
	ld	(hl),a		;=0
	inc	hl
	ld	(FBUFAD),hl
	ld	a,(FILES)
	inc	a
	ld	de,0110h
	ld	b,a
CLRLP1:
	ld	(hl),00h
	add	hl,de
	djnz	CLRLP1
	ld	(STRPTR),hl
	ld	(TMPSTR),hl
	ld	(FREEAD),hl

;numerical variable type
	ld	hl,DEFTYP
	ld	b,26
CLRLP2:
	ld	(hl),05h	;DEFSNG
	inc	hl
	djnz	CLRLP2

;random seed
CLRRND:
	call	RDTIMER
	ld	a,r
	ld	(SEED),a
	ld	(SEED+1),hl
	ld	(SEED+3),de
	or	h
	or	l
	or	d
	or	e
	jr	z,CLRRND

	ld	b,20h
CLRLP3:
	push	bc
	call	RNDSFT
	pop	bc
	djnz	CLRLP3

;restore
	call	RSTRNOPAR

;on error goto
CLRERRHND:
	xor	a
	ld	(ERRHND+2),a
	ld	(DURERRHND+1),a
	ret


;TRON command
C_TRON:
	db	3eh		;ld a,

;TROFF command
C_TROF:
	xor	a
	ld	(TRACEON),a
	ret

;input: de=line number
;destroy: af,bc,de
TRACE:
	push	hl
	ld	a,'['
	call	PUTC
	ex	de,hl
	call	PUTI2
	ld	a,']'
	call	PUTC
	pop	hl
	ret


;OPTION command
C_OPT:
	call	SKIPSP
	cp	BASE_
	jp	nz,SNERR
	inc	hl
	call	INT1ARG
	cp	02h
	jp	nc,FCERR
	ld	(BASE+1),a
	ret


;LINPUT command
C_LINP:
	call	PROMPT
	jr	nz,LINPSTRT	;no prompt
	jr	c,LINPSTRT	;with comma-terminated prompt
	ld	a,'?'
	call	PUTC
	ld	a,' '
	call	PUTC
LINPSTRT:
	call	CHKGETV
	ld	a,c
	cp	03h
	jp	nz,TMERR
	xor	a
	ld	(de),a		;length=0
	push	de		;variable value address
;	call	INPT1
	call	INPTSCR
	pop	de		;variable value address
	jr	c,LINPEND	;stop

	push	hl		;INPBUF
	ld	bc,0101h
	xor	a
	cpir
	sub	c
	ex	de,hl
	pop	de		;INPBUF
	call	MKSTRVAR

LINPEND:
	jp	CLRBUFP


;INPUT$() function
F_INPT:
	ld	(PROGAD),hl
	ld	a,(hl)
	cp	'$'
	jp	nz,SNERR
	inc	hl
	call	CHKLPAR
	call	INT1ARG
	ld	a,(hl)
	cp	','
	jr	z,FINPTDEV
	call	CHKRPAR

	xor	a
	ld	(STRDSC1),a
;	ld	a,e
;	or	a
	or	e
	jr	z,FINPTEND
	ld	b,e
FINPTLP:
	ld	h,b		;counter
	call	GETC
	cp	03h
	jr	z,FINPTEND
	push	hl		;h=counter
	call	ADDSTR1
	pop	bc
	djnz	FINPTLP

FINPTEND:
	call	INKYEND
	jp	FNCRTN


;INPUT$() function for CMT device
FINPTDEV:
	call	SKIPSPINC
	cp	'#'
	jr	nz,FINPTDZ
	inc	hl
FINPTDZ:
	push	de		;1st parameter
	call	INT1ARG
	call	CHKRPAR
	ld	a,e
	or	a
	jp	z,FCERR
	call	GETBUFP
	jp	z,CFERR
	ld	(BUFP),hl

	pop	de		;1st parameter
	xor	a
	ld	(STRDSC1),a
	ld	a,e
	or	a
	jr	z,FINPTDEND
	ld	b,e
FINPTDLP:
	push	bc
	call	GETDEV
	jp	c,IPERR
	call	ADDSTR1
	pop	bc
	djnz	FINPTDLP
FINPTDEND:
	call	INKYEND
	call	CLRBUFP
	jp	FNCRTN

;INPUT command
C_INPT:
	dec	hl
	ld	(STOPAD),hl
	inc	hl
	call	PROMPT
	push	hl		;address after prompt (for "REDO FROM START")
	jr	c,INPTSTRT	;with comma-terminated prompt
INPTQ:
	ld	de,(BUFP)
	ld	a,d
	or	e
	jr	nz,INPTSTRT
	ld	a,'?'
	call	PUTC
	ld	a,' '
	call	PUTC
INPTSTRT:
	ld	(PROGAD),hl
	call	INPT1
	pop	de		;address after prompt
	pop	bc		;return address
	jp	c,STOP3
	push	bc		;return address
	push	de		;address after prompt
	xor	a
	ld	(STOPAD+1),a	;higher byte

;analyze input buffer
INPTANA:
	ld	de,INPBUF
	ld	hl,(PROGAD)
INPTLP:
	push	de		;input buffer
	call	CHKGETV
	ld	a,(INPBUF)
	or	a
	jr	z,GONEXT	;no change in variable if no input
	pop	hl		;input buffer
	ld	a,c
	cp	03h
	jr	z,INPTSTR
INPTNUM:
	call	INPRDNUM
	call	SKIPSP
	ex	de,hl
	or	a
	jr	z,INPTNEXT
	cp	','
	jr	z,INPTNEXT
	ld	hl,REDO_STR
	call	PUTSNL
	pop	hl		;address after prompt
	push	hl
	jr	INPTQ

GONEXT:
	pop	de		;input buffer
	jr	INPTNEXT

INPTSTR:
	xor	a
	call	INPRDSTR
	call	INPRDSETV
	ex	de,hl

INPTNEXT:
	ld	hl,(PROGAD)
	call	SKIPSP
	cp	','
	jr	nz,INPTEND

;comma
	inc	hl
	ld	a,(INPBUF)
	or	a
	jr	z,INPTLP
	ld	a,(de)
	inc	de
	cp	','
	jr	z,INPTLP

	ld	de,(BUFP)
	ld	a,d
	or	e
	jr	nz,INPTQ

	ld	a,'?'
	call	PUTC
	jp	INPTQ

;not comma
INPTEND:
	pop	af		;cancel program address
	ex	de,hl
	ld	a,(hl)
	or	a
	ld	hl,EXTRA_STR
	call	nz,PUTSNL
	jp	PIEND

REDO_STR:
	db	"?REDO FROM START", 00h
EXTRA_STR:
	db	"EXTRA IGNORED", 00h


;prompt for INPUT/LINE INPUT
;input: hl=" address
;output: hl=address after "prompt"; or "prompt",
;        z-flag(1=with prompt), c-flag(1=, terminated prompt)
PROMPT:
	ld	de,(LINENUM)
	ld	a,d
	and	e
	inc	a
	jp	z,IDERR

	call	SKIPSP
	cp	'#'
	jr	z,INPDEV

	cp	22h		;double quotation
	scf
	ccf
	ret	nz		;c-flag=0
PPTLP:
	inc	hl
	ld	a,(hl)
	or	a
	jp	z,SNERR
	cp	22h		;double quotation
	jr	z,PPTEND
	call	PUTC
	jr	PPTLP
PPTEND:
	call	SKIPSPINC
	inc	hl
	cp	','
	scf
	ret	z		;c-flag=1 if comma
	cp	';'
	ret	z
	dec	hl
	cp	a		;reset c-flag, set z-flag
	ret


;INPUT/LINE INPUT for CMT device
INPDEV:
	inc	hl	
	call	INT1ARG
	call	SKIPSP
	cp	','
	jp	nz,SNERR
	inc	hl
	push	hl
	ld	a,(FILES)
	cp	e		;e=file number
	jp	c,FCERR
	ld	a,e
	or	a
	jp	z,FCERR
	call	GETBUFP
	jp	z,CFERR
	ld	(BUFP),hl
	pop	hl
	scf
	ret			;z-flag=0


;DEF command
C_DEF:
	call	SKIPSP
	inc	hl
	cp	KEY_
	jp	z,C_KEY

	inc	a
	jp	nz,SNERR
	ld	a,(hl)
	inc	hl
	cp	FN_
	jr	z,DEFFN
	cp	USR_
	jp	nz,SNERR

DEFUSR:
	call	SKIPSP
	ld	bc,0000h
	cp	0ah+1
	jr	nc,DEFUSR0
	inc	hl
	dec	a		;0-9
	add	a,a
	ld	c,a
DEFUSR0:
	push	bc		;0,2,4,...18
	call	SKIPSP
	cp	EQ_
	jp	nz,SNERR
	inc	hl
	call	INT2ARG		;de
	pop	bc		;0,2,4,...18
	ld	hl,USR0AD
	add	hl,bc
	ld	(hl),e
	inc	hl
	ld	(hl),d
	ret

USR0AD:
	dw	FCERR
USR1AD:
	dw	FCERR
USR2AD:
	dw	FCERR
USR3AD:
	dw	FCERR
USR4AD:
	dw	FCERR
USR5AD:
	dw	FCERR
USR6AD:
	dw	FCERR
USR7AD:
	dw	FCERR
USR8AD:
	dw	FCERR
USR9AD:
	dw	FCERR

DEFFN:
	ld	de,(LINENUM)
	ld	a,d
	and	e
	inc	a
	jp	z,IDERR
	call	CHKVAR		;b,c,z,
	ld	a,c
	or	a		;reset z-flag
	set	6,c		;fn mark
	call	GETVAR
	ld	hl,(PROGAD)
	call	SKIPSP
	ex	de,hl
	ld	(hl),e
	inc	hl
	ld	(hl),d
	inc	hl
	xor	a
	ld	(hl),a
	inc	hl
	ld	(hl),a
	ex	de,hl

	call	SKIPSP
	cp	'('
	jr	nz,DEFFNEND
DEFFNLP1:
	call	SKIPSPINC
	sub	'A'
	cp	'Z'-'A'+1
	jp	nc,SNERR
DEFFNLP2:
	inc	hl
	ld	a,(hl)
	call	ALPNUM
	jr	z,DEFFNLP2
	call	SKIPSP
	cp	'%'
	jr	z,DEFFNSUF
	cp	'!'
	jr	z,DEFFNSUF
	cp	'#'
	jr	z,DEFFNSUF
	cp	'$'
	jr	nz,DEFFNCHKCM

;with suffix
DEFFNSUF:
	call	SKIPSPINC

DEFFNCHKCM:
	cp	','
	jr	z,DEFFNLP1
	cp	')'
	jp	nz,SNERR
	inc	hl

DEFFNEND:
	call	SKIPSP
	cp	EQ_
	jp	nz,SNERR
	ld	bc,':'
	call	SKIPTO
	ld	(PROGAD),hl
	ret


;RESTORE command
C_RSTR:
	call	GETLNSP
	jr	z,RSTRLN
	cp	22h		;double quotation mark
	jr	nz,RSTRNOPAR
;label
RSTRLBL:
	call	SRCHLBL
	jr	RSTREND

;line number
RSTRLN:
	push	hl
	push	de
	call	SRCHLN
	jp	nc,ULERR
	pop	de
	pop	hl
RSTREND:
	call	ONSKIP
	ld	(PROGAD),hl
	ld	(F_DTL+1),de
	ld	(DATAAD),bc
	ret

;no parameter
RSTRNOPAR:
	ld	hl,(BASICAD)
	dec	hl
	ld	(DATAAD),hl
	ld	hl,0000h
	ld	(F_DTL+1),hl
	ret


;READ command
C_READ:
	call	CHKGETV
	push	bc		;b=variable name length, c=variable type
	push	de		;variable value address

	ld	hl,(DATAAD)
	ld	a,(hl)
	cp	','
	jr	z,READDATA1
	cp	':'
	jr	z,READNZ
	or	a
	jr	nz,READDATA2

;new line
READLP:
	inc	hl
	ld	a,(hl)
	inc	hl
	or	(hl)
	jp	z,ODERR
	inc	hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)

READNZ:
	inc	hl
	ld	bc,DATA_
	call	SKIPTO
	or	a
	jr	z,READLP
	ld	(F_DTL+1),de

READDATA1:
	inc	hl		;skip DATA or ,
READDATA2:
	pop	de		;variable value address
	pop	bc		;b=variable name length, c=variable type
	ld	a,c
	cp	03h
	jr	z,READSTR

READNUM:
	call	INPRDNUM
	jr	READNEXT

;de=variable value address
READSTR:
	ld	a,b		;>0
	call	INPRDSTR
	call	INPRDSETV

;next data?
READNEXT:
	call	SKIPSP
	ld	(DATAAD),hl
	ld	hl,(PROGAD)
	call	SKIPSP
	cp	','
	ret	nz
	inc	hl
	jr	C_READ

;for INPUT/READ
;input: b=variable name length, c=variable type,
;       de=variable adderss, hl=input buffer/data address
;output: hl=next input buffer/data address
;destroy: af,bc,de
INPRDNUM:
	push	bc		;b=variable name length, c=variable type
	push	de		;variable value address
	call	ATOF
	pop	de		;variable value address
	pop	bc		;b=variable name length, c=variable type
	push	hl		;next input buffer/data address
	push	de
	call	CNVTYP1
	pop	de
	ld	b,00h
	ld	hl,FAC1+1
	ldir
	pop	hl
	ret

;input: a=input(0)/read(>0), hl=input address
;output: a=length, hl=next address, bc=string start address
;destroy: f
INPRDSTR:
	ld	(INPRD+1),a
	ld	b,00h		;length
	call	SKIPSP
	cp	22h
	jr	nz,IRSNDQ	;no double quotation

;with double quotation
	inc	hl
	push	hl
IRSLP1:
	ld	a,(hl)
	or	a
	jr	z,IRSEND
	inc	hl
	cp	22h
	jr	z,IRSEND
	inc	b
	jp	z,LSERR
	jr	IRSLP1

;not double quotation
IRSNDQ:
	push	hl
IRSLP2:
	ld	a,(hl)
	or	a
	jr	z,IRSEND
	cp	','
	jr	z,IRSEND
	cp	':'
	jr	nz,IRSNZ
INPRD:
	ld	a,00h		;input(0)/read(>0)
	or	a
	jr	nz,IRSEND	;READ
IRSNZ:
	inc	hl
	inc	b
	jr	nz,IRSLP2
	jp	LSERR

IRSEND:
	ld	a,b		;length
	pop	bc
	ret


;copy string descriptor to variable for INPUT/READ command
;input: a=length, bc=string address, de=variable value address
;destroy: af,bc,de
INPRDSETV:
	push	hl
	ex	de,hl
	ld	d,b
	ld	e,c
	call	MKSTRVAR
	pop	hl
	ret


OK_STR:
	db	"OK",00h

EDIT:
	call	REMOVE
	call	PRTNLX
	ld	hl,OK_STR
	call	PUTSNL
EDIT2:
	call	PUTNLXPRN
	ld	hl,0ffffh
	ld	(LINENUM),hl
EDITLP:
	call	INPTSCR
	call	INPTPRG
	jr	z,EDITLP
	ld	hl,INPBUF
	jp	nc,INTPRT
	call	RESSTK
	jr	EDITLP


;input program or command
;input: hl=INPBUF
;output: z-flag(1=no input), c-flag(1=program changed, 0=command)
;destroy: af,bc,de,hl
INPTPRG:
	call	SKIPSP
	or	a
	ret	z		;no input
	call	CHKFIG
	push	af		;c-flag=1: with line number
	call	c,ATOLN
	pop	af		;c-flag=1: with line number
	push	af		;c-flag=1: with line number
	push	de		;line number
	call	CNVIL
	pop	de		;line number
	pop	af		;c-flag=1: with line number
	inc	a
	dec	a		;reset z-flag(a>0)
	ret	nc

;change program
CHGPRG:
	ld	a,d
	or	e
	jp	z,FCERR		;de=0
	ld	hl,INPBUF
	push	hl		;buffer address
	push	de		;line number
	push	hl		;buffer address
	push	bc		;end address
	call	SRCHLN
	call	c,DELPRG	;c-flag=1
	pop	de		;end address
	pop	hl		;buffer address
	inc	(hl)
	dec	(hl)		;set z-flag if only line number
	jr	nz,INSPRG
	jp	nc,ULERR
	pop	de		;line number
	pop	hl		;buffer address
	dec	h		;reset z-flag
	ret			;c-flag=1, delete line


;insert program
;bc=next line address, hl=buffer address, de=end address
INSPRG:
	ex	de,hl
	or	a
	sbc	hl,de		;<=253
	ld	a,l		;new line-5
	ld	de,0005h
	add	hl,de
	call	MOVEVAR
	push	de		;new line size
	push	hl		;old (TMPSTR)
	or	a
	sbc	hl,bc
	push	hl		;slide size
	add	hl,bc		;old (TMPSTR)
	add	hl,de		;new (TMPSTR)=old (TMPSTR)+new line size
	call	CHKSTK
	ld	(TMPSTR),hl
	ex	de,hl
	pop	bc		;slide size
	pop	hl		;old (TMPSTR)
	dec	hl
	dec	de
	lddr

	inc	hl		;adress to be inserted
	inc	hl
	pop	de		;new line size
	ld	(hl),e
	inc	hl
	ld	(hl),d
	inc	hl
	pop	de		;line number
	ld	(hl),e
	inc	hl
	ld	(hl),d
	inc	hl
	ex	de,hl
	pop	hl		;buffer address after line number
	ld	c,a		;new line size-5, b=0
	ldir
	xor	a
	ld	(de),a
	inc	a		;reset z-flag
	scf
	ret


;input: bc=address to be deleted, hl=next line address
;output: c-flag=1
;destroy: af,de,hl
DELPRG:
	push	bc		;address to be deleted
	push	hl		;next line address
	ex	de,hl
	ld	h,b
	ld	l,c
	or	a
	sbc	hl,de
	call	MOVEVAR
	pop	bc		;next line address
	push	bc		;next line address
	or	a
	sbc	hl,bc		;(TMPSTR)-next line address
	ld	b,h
	ld	c,l
	pop	hl		;next line address
	pop	de		;address to be deleted
	push	de
	ldir
	ld	(TMPSTR),de
	pop	bc
	scf
	ret


;move variable, file buffer, and string data area
;input: hl=diff
;output: hl=old (TMPSTR)
;destory: f,de,hl
MOVEVAR:
	ex	de,hl
	ld	hl,(VARAD)
	add	hl,de
	ld	(VARAD),hl

	ld	hl,(FBUFAD)
	add	hl,de
	ld	(FBUFAD),hl

	ld	hl,(BUFP)
	inc	h
	dec	h
	jr	z,MOVEVARZ	;if (BUFP)=0
	add	hl,de
	ld	(BUFP),hl

MOVEVARZ:
	ld	hl,(STRPTR)
	add	hl,de
	ld	(STRPTR),hl

	ld	hl,(TMPSTR)
	push	hl
	add	hl,de
	ld	(TMPSTR),hl
	ld	(FREEAD),hl
	pop	hl
	ret


;convert to intermediate language
;input: hl=input buffer address, c-flag(1=with line number)
;output: bc=end address
;destroy: af,de,hl
CNVIL:
	ld	a,00h
	ld	(TMPCS+1),a
	ex	de,hl
	jr	nc,ILSTRT	;no line number
ILLP1:
	dec	de
	ld	a,(de)
	cp	' '
	jr	z,ILLP1

	inc	de
	ld	a,(de)
	cp	' '
	jr	nz,ILSTRT
	inc	de

ILSTRT:
	ld	a,0ffh
	ld	(ILPRV+1),a
	ld	hl,INPBUF-1
	dec	de
ILLP2:
	inc	de
	ld	a,(de)
	call	CHK1011
	jr	z,ILLP2
	or	a
	jp	m,ILLP2		;skip kana etc.
	inc	hl
	jp	z,ILEND1	;00=line end

	call	CHKFIG
	jp	c,ILNUM
	cp	'.'
	jp	z,ILNUM
	cp	'&'
	jp	z,ILNUM

	cp	22h		;double quotation mark
	jp	z,ILDQ
	cp	27h		;apostrophe
	jr	z,ILAPOS
	cp	'*'
	jr	c,ILPUT		;00-29h
	cp	','
	jr	z,ILPUT
	cp	':'
	jr	z,ILPUT
	cp	';'
	jr	z,ILPUT
	cp	'?'
	jr	z,ILQ

;compare with command string
	push	hl
	cp	'L'
	jr	nz,ILNZ1
	inc	de
	ld	a,(de)
	cp	'?'
	ld	b,LPRINT_
	jp	z,ILFOUND1
	dec	de
ILNZ1:
	ld	hl,CMDNAME
	call	SRCHCMD
	jp	c,ILFOUND1
;	ld	hl,CMDNAMEFE
	call	SRCHCMD
	ld	a,0feh
	jp	c,ILFOUND2
;	ld	hl,FNCNAME
	call	SRCHCMD
	ld	a,0ffh
	jp	c,ILFOUND2

;not found
	pop	hl
	ld	a,(de)
	sub	'A'
	cp	'Z'-'A'+1
	ld	a,(de)
	jr	c,ILALP
ILPUT:
	ld	(hl),a
	jr	ILLP2

;alphabet (variable candidate)
ILALP:
	ld	(hl),a
	inc	hl
	ld	a,0ffh
	ld	(ILPRV+1),a
ILALPSKIP:
	inc	de
	ld	a,(de)
	call	CHK1011
	jr	z,ILALPSKIP
	call	ALPNUM
	jr	c,ILALP
	dec	de
	dec	hl
	jp	ILLP2

;question mark
ILQ:
	ld	a,PRINT_
	ld	(ILPRV+1),a
	jr	ILPUT

;apostrophe
ILAPOS:
	ld	(hl),':'
	or	a
	sbc	hl,de
	add	hl,de
	ld	a,01h
	inc	hl
	jr	c,ILAPOSC

	push	hl
	call	MVINPBUF
	pop	hl
	inc	de

ILAPOSC:
	ld	(hl),27h	;apostrophe
	xor	a
	jr	ILREM

;double quotation mark
ILDQ:
	ld	(hl),a
;REM
ILREM:
	ld	c,a		;double quotation mark or 0(REM)
	ld	b,01h		;double quotation counter

;REM,DATA,""
REMDTDQ:
	inc	de
	ld	a,(de)
	call	CHK1011
	dec	de
	jr	z,ILLP3
TMPCS:
	ld	a,00h		;10h=character set 1
	cp	10h
	jr	nz,ILLP3
	inc	hl
	jr	ILEND1

;loop until (de)=0 or (de)=c
ILLP3:
	inc	de
	inc	hl
	ld	a,(de)

ILEND1:
	ld	(hl),a
	or	a
	jr	z,ILEND2
	cp	22h
	jr	nz,ILNZ2
	inc	b
ILNZ2:
	bit	0,b
	jr	nz,ILLP3
	cp	c
	jr	nz,ILLP3
	jp	ILLP2

ILEND2:
	ld	b,h
	ld	c,l
	inc	hl
	ld	(hl),a		;a=0
	inc	hl
	ld	(hl),a		;a=0
	ret

;1-byte command found
ILFOUND1:
	pop	hl
	ld	a,b
	cp	MINUS_		;for LIST n-m
	jr	z,ILZ
	cp	TO_		;for GO TO
	jr	z,ILZ
	cp	SUB_		;for GO SUB
	jr	z,ILZ
	ld	(ILPRV+1),a
ILZ:
	cp	ELSE_
	jr	nz,ILNZ3
	ld	(hl),':'
	inc	hl
ILNZ3:
	ld	(hl),b
	sub	REM_
	jr	z,ILREM
	cp	DATA_-REM_
	jp	nz,ILLP2
;DATA
ILDATA:
	ld	bc,003ah	;b=0,c=':'
	jr	REMDTDQ

;2-byte command found
ILFOUND2:
	pop	hl
	ld	(hl),a
	inc	hl
	ld	(hl),b
	jp	ILLP2

ILNUM:
	push	hl		;out
	ex	de,hl
ILPRV:
	ld	a,0ffh		;previous intermediate code
	cp	EDIT_+1
	jr	c,ILLN		;GO-EDIT: line number
	cp	ELSE_
	jr	z,ILLN
	cp	THEN_
	jr	z,ILLN

	call	ATOICODE

ILCHKBF:
	ld	c,a		;length > 0
	pop	de		;out
	push	hl		;in_after
	or	a
	sbc	hl,de		;in_after-out < 256
	dec	a
	sub	l		;data length > in_after-out?
	jr	c,ILNUMC

	add	hl,de
	inc	a
	call	MVINPBUF
	pop	af		;cancel stack
	push	hl		;new in_after

ILNUMC:
	ld	hl,FAC1
	ld	b,00h
	ldir
	ex	de,hl
	dec	hl
	pop	de		;in_after
	dec	de
	jp	ILLP2

ILLN:
	ld	a,(hl)
	cp	'.'
	jr	z,NOTLN
	cp	'&'
	jr	nz,ILGETLN
NOTLN:
	ex	de,hl
	pop	hl		;out
	jp	ILPUT
ILGETLN:
	call	ATOLN
	ld	a,0bh
	ld	(FAC1),a
	ld	(FAC1+1),de
	ld	a,03h
	jr	ILCHKBF


;check 10h, 11h (change character set)
;input: a=data
;output: z-flag(1=10h or 11h)
;destroy: f
CHK1011:
	cp	10h
	jr	z,CHK1011Z
	cp	11h
	ret	nz
CHK1011Z:
	ld	(TMPCS+1),a
	ret


;move input buffer
;input: a=diff, hl=address
;output: hl=new address
;destroy: af,hl
MVINPBUF:
	push	de
	push	bc
	ld	b,a
	ld	c,00h
MVIBLP:
	inc	c
	ld	a,(hl)
	or	a
	jr	z,MVIBZ
	inc	hl
	jr	MVIBLP
MVIBZ:
	push	hl
	ld	a,b
	add	a,l
	ld	l,a
	jr	nc,MVIBNC
	inc	h
MVIBNC:
	ex	de,hl

	ld	hl,INPBUF+MAXINP	;253 characters
	or	a
	sbc	hl,de
	jp	c,BOERR

	pop	hl

	ld	b,00h
	lddr
	ex	de,hl
	inc	hl
	pop	bc
	pop	de
	ret


;search command name table
;input: de=program address, hl=table address
;output: c-flag(1=found),b=result,de=address of command last character
;destroy: af,c,hl
;output: c-flag(0=not round), hl=address after table
;destroy: af,bc
SRCHCMD:
	ld	b,80h
SCMDLP1:
	push	de
	ld	a,(de)
SCMDLP2:
	call	CHK1011
	jr	z,SCMDZ
	ld	c,(hl)
	res	7,c
	cp	c
	jr	nz,SCMDNZ	;not match
	bit	7,(hl)
	jr	nz,SCMDC	;match
	inc	hl
SCMDZ:
	inc	de
	sub	'A'
	cp	'Z'-'A'+1
	ld	a,(de)
	jr	nc,SCMDLP2	;not alphabet
	cp	'.'
	jr	nz,SCMDLP2

;match
SCMDC:
	pop	af		;cancel stack
	scf
	ret

;not match
SCMDNZ:
	inc	b
	pop	de
SCMDLP3:
	bit	7,(hl)
	inc	hl
	jr	z,SCMDLP3
	ld	a,(hl)
	or	a
	jr	nz,SCMDLP1
	inc	hl
	ret			;c-flag=0


;skip space and get line number
GETLNSP:
	call	SKIPSP
;	jr	GETLN

;get line number (intermediate code)
;input: hl=program address
;output: a=data, de=line number, hl=next address, z-flag(1=ok)
;destroy: f
GETLN:
	ld	de,0000h
	ld	a,(hl)
	cp	0bh
	ret	nz
	inc	hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
	ret

NFERR:
	ld	a,1
	db	01h		;ld bc,****
SNERR:
	ld	a,2
	db	01h		;ld bc,****
RGERR:
	ld	a,3
	db	01h		;ld bc,****
ODERR:
	ld	a,4
	db	01h		;ld bc,****
FCERR:
	ld	a,5
	db	01h		;ld bc,****
OVERR:
	ld	a,6
	db	01h		;ld bc,****
OMERR:
	ld	a,7
	db	01h		;ld bc,****
ULERR:
	ld	a,8
	db	01h		;ld bc,****
BSERR:
	ld	a,9
	db	01h		;ld bc,****
DDERR:
	ld	a,10
	db	01h		;ld bc,****
DIV0ERR:
	ld	a,11
	db	01h		;ld bc,****
IDERR:
	ld	a,12
	db	01h		;ld bc,****
TMERR:
	ld	a,13
	db	01h		;ld bc,****
LSERR:
	ld	a,15
	db	01h		;ld bc,****
STERR:
	ld	a,16
	db	01h		;ld bc,****
CNERR:
	ld	a,17
	db	01h		;ld bc,****
UFERR:
	ld	a,18
	db	01h		;ld bc,****
NRERR:
	ld	a,19
	db	01h		;ld bc,****
RWERR:
	ld	a,20
	db	01h		;ld bc,****
MOERR:
	ld	a,22
	db	01h		;ld bc,****
BOERR:
	ld	a,23
	db	01h		;ld bc,****
BPERR:
	ld	a,25
	db	01h		;ld bc,****
URERR:
	ld	a,26
	db	01h		;ld bc,****
TRERR:
	ld	a,29
	db	01h		;ld bc,****
BMERR:
	ld	a,30
	db	01h		;ld bc,****
WHERR:
	ld	a,32
	db	01h		;ld bc,****
WEERR:
	ld	a,33
	db	01h		;ld bc,****
RFERR:
	ld	a,34
	db	01h		;ld bc,****
FNERR:
	ld	a,35
	db	01h		;ld bc,****
FOERR:
	ld	a,36
	db	01h		;ld bc,****
AOERR:
	ld	a,54
	db	01h		;ld bc,****
IPERR:
	ld	a,61
	db	01h		;ld bc,****
FDERR:
	ld	a,65
	db	01h		;ld bc,****
CFERR:
	ld	a,71
	db	01h		;ld bc,****
DOERR:
	ld	a,73

;input: a=error number
ERROR:
	ld	(ERRVAL),a
	ld	hl,(LINENUM)
	ld	(ERLVAL),hl
	call	CLRBUFP
	xor	a
	ld	(SELPRN),a
	ld	(INMON),a
	ld	hl,(TMPSTR)
	ld	(FREEAD),hl

SPERR:
	ld	sp,STKINI
	ld	a,(ERRHND+2)
	or	a
	jr	z,PUTERR

	ld	a,(DURERRHND+1)
	or	a
	jr	z,ERRHND	;0=not during error handling

PUTERR:
	ld	hl,STRDSC2
	ld	bc,0a00h
ERRORLP1:
	ld	(hl),c		;=0
	inc	hl
	djnz	ERRORLP1
	ld	(STOPAD),bc	;=0

	ld	a,(ERRVAL)
	cp	74
	jr	c,ERRORC
	xor	a
ERRORC:
	ld	e,a
	inc	e
	call	PRTNLX
	ld	hl,ERROR_STRS
	ld	bc,0000h
ERRORLP2:
	xor	a
	cpir
	dec	e
	jr	nz,ERRORLP2
	ld	a,(hl)
	or	a
	jr	nz,ERRORNZ
	ld	hl,ERR00
ERRORNZ:
	call	PUTS

PUTLNUM:
	ld	hl,(LINENUM)
	ld	a,h
	and	l
	inc	a
	call	nz,INLNUM	;if (LINENUM)<>ffffh
	call	PUTNL
	jp	EDIT

;error handling
ERRHND:
	ld	hl,0000h	;error handling line address
	inc	a		;0->1
	ld	(DURERRHND+1),a
	jp	INTPRT


;print "in *****"
;input: hl
;destory: af,bc,de,hl
INLNUM:
	push	hl
	ld	hl,ERRIN
	call	PUTS
	pop	hl
;	jp	PUTI2


;print 2-byte integer in device (unsigned)
;input: hl
;destory: af,bc,de,hl
PUTI2:
	ld	bc,FAC3
	ex	de,hl
	call	I2TOA
	jp	PUTS

ERRIN:
	db	" IN ", 00h

ERROR_STRS:
	db	00h
ERR00:
	db	"UNPRINTABLE ERROR", 00h
ERR01:
	db	"NEXT WITHOUT FOR", 00h
ERR02:
	db	"SYNTAX ERROR", 00h
ERR03:
	db	"RETURN WITHOUT GOSUB", 00h
ERR04:
	db	"OUT OF DATA", 00h
ERR05:
	db	"ILLEGAL FUNCTION CALL", 00h
ERR06:
	db	"OVERFLOW", 00h
ERR07:
	db	"OUT OF MEMORY", 00h
ERR08:
	db	"UNDEFINED LABEL", 00h
ERR09:
	db	"SUBSCRIPT OUT OF RANGE", 00h
ERR10:
	db	"DUPLICATE DEFINITION", 00h
ERR11:
	db	"DIVISION BY ZERO", 00h
ERR12:
	db	"ILLEGAL DIRECT", 00h
ERR13:
	db	"TYPE MISMATCH", 00h
ERR14:
	db	00h
ERR15:
	db	"STRING TOO LONG", 00h
ERR16:
	db	"TOO COMPLEX", 00h
ERR17:
	db	"CAN", 27h, "T CONTINUE", 00h
ERR18:
	db	"UNDEFINED FUNCTION", 00h
ERR19:
	db	"NO RESUME", 00h
ERR20:
	db	"RESUME WITHOUT ERROR", 00h
ERR21:
	db	"ILLEGAL FORMAT", 00h
ERR22:
	db	"MISSING OPERAND", 00h
ERR23:
	db	"LINE BUFFER OVERFLOW", 00h
ERR24:
	db	00h
ERR25:
	db	"BAD PRINTER MODE", 00h
ERR26:
	db	"UNTIL WITHOUT REPEAT", 00h
ERR27:
	db	00h
ERR28:
	db	00h
ERR29:
	db	"TAPE READ ERROR", 00h
ERR30:
	db	"BAD FILE MODE", 00h
ERR31:
	db	"OUT OF STACK", 00h
ERR32:
	db	"WHILE WITHOUT WEND", 00h
ERR33:
	db	"WEND WITHOUT WHILE", 00h
ERR34:
	db	"RESERVED FEATURE", 00h
ERR35:
	db	"FOR WITHOUT NEXT", 00h
ERR36:
	db	"FORMAT OVER", 00h
ERR37:
	db	00h
ERR38:
	db	00h
ERR39:
	db	00h
ERR40:
	db	00h
ERR41:
	db	00h
ERR42:
	db	00h
ERR43:
	db	00h
ERR44:
	db	00h
ERR45:
	db	00h
ERR46:
	db	00h
ERR47:
	db	00h
ERR48:
	db	00h
ERR49:
	db	00h
ERR50:
	db	00h
ERR51:
	db	00h
ERR52:
	db	00h
ERR53:
	db	00h
ERR54:
	db	"ALREADY OPEN", 00h
ERR55:
	db	00h
ERR56:
	db	"DEVICE I/O ERROR", 00h
ERR57:
	db	"FILE ALREADY EXISTS", 00h
ERR58:
	db	00h
ERR59:
	db	00h
ERR60:
	db	"DEVICE FULL", 00h
ERR61:
	db	"INPUT PAST END", 00h
ERR62:
	db	00h
ERR63:
	db	00h
ERR64:
	db	"BAD ALLOCATION TABLE", 00h
ERR65:
	db	"BAD FILE DESCRIPTOR", 00h
ERR66:
	db	"BAD RECORD", 00h
ERR67:
	db	"BAD PASSWORD", 00h
ERR68:
	db	00h
ERR69:
	db	00h
ERR70:
	db	00h
ERR71:
	db	"FILE NOT OPEN", 00h
ERR72:
	db	"WRITE PROTECTED", 00h
ERR73:
	db	"DEVICE OFFLINE", 00h


OPRTBL:
	dw	O_EQV,  O_IMP,  O_XOR,  O_OR,   O_AND,  O_NOT,  O_GTLT, O_LTGT
	dw	O_EQLT, O_LTEQ, O_EQGT, O_GTEQ, O_EQ,   O_GT,   O_LT,   O_PLS
	dw	O_MNS,  O_MOD,  O_YEN,  O_DIV,  O_MUL,  O_POW


FNCTBL:
	dw	F_INT,  F_ABS,  F_SIN,  F_COS,  F_TAN,  F_LOG,  F_EXP,  F_SQR
	dw	F_RND,  F_PEEK, F_ATN,  F_SGN,  F_FRAC, F_FIX,  F_PAI,  F_RAD
	dw	F_INP,  F_CDBL, F_CSNG, F_CINT, F_DSKF, F_EOF,  F_FPOS, F_LOC
	dw	F_LOF,  F_POS,  F_FAC,  F_SUM,  F_FRE,  F_LPOS, F_JOY,  SNERR
	dw	F_CHR,  F_STR,  F_HEX,  F_OCT,  F_BIN,  F_MKI,  F_MKS,  F_MKD
	dw	F_SPAC, SNERR,  SNERR,  F_ASC,  F_LEN,  F_VAL,  F_CVS,  F_CVD
	dw	F_CVI,  SNERR,  SNERR,  F_ERR,  F_ERL,  F_CSRL, F_STRP, F_DTL
	dw	SNERR,  SNERR,  F_LEFT, F_RGT,  F_MID,  F_INKY, F_INST, F_HEXC
	dw	F_MEM,  F_SCRN, F_VARP, F_STRG, F_TIME, SNERR,  SNERR, F_FN
	dw	F_USR,  SNERR,  SNERR,  F_ATTR, SNERR,  F_CHAR


_CMDNAME:	ds	CMDNAME-_CMDNAME
	org	CMDNAME

	db	"GOT",'O'+80h,	"GOSU",'B'+80h,	"G",'O'+80h,	"RU",'N'+80h
	db	"RETUR",'N'+80h,"RESTOR",'E'+80h,"RESUM",'E'+80h,"LIS",'T'+80h
	db	"LLIS",'T'+80h,	"DELET",'E'+80h,"RENU",'M'+80h,	"AUT",'O'+80h
	db	"EDI",'T'+80h,	"FO",'R'+80h,	"NEX",'T'+80h,	"PRIN",'T'+80h
	db	"LPRIN",'T'+80h,"INPU",'T'+80h,	"LINPU",'T'+80h,"I",'F'+80h
	db	"DAT",'A'+80h,	"REA",'D'+80h,	"DI",'M'+80h,	"RE",'M'+80h
	db	"EN",'D'+80h,	"STO",'P'+80h,	"CON",'T'+80h,	"CL",'S'+80h
	db	"CLEA",'R'+80h,	"O",'N'+80h,	"LE",'T'+80h,	"NE",'W'+80h
	db	"POK",'E'+80h,	"OF",'F'+80h,	"WHIL",'E'+80h,	"WEN",'D'+80h
	db	"REPEA",'T'+80h,"UNTI",'L'+80h,	80h,		80h
	db	"TRAC",'E'+80h,	"TRO",'N'+80h,	"TROF",'F'+80h,	"SPEE",'D'+80h
	db	80h,		80h,		"DEFIN",'T'+80h,"DEFSN",'G'+80h
	db	"DEFDB",'L'+80h,"DEFST",'R'+80h,"DE",'F'+80h,	80h
	db	"LOA",'D'+80h,	"SAV",'E'+80h,	"MERG",'E'+80h,	"CHAI",'N'+80h
	db	"CONSOL",'E'+80h,80h,		"OU",'T'+80h,	"SEARC",'H'+80h
	db	"WAI",'T'+80h,	"PAUS",'E'+80h,	"WRIT",'E'+80h,	"SWA",'P'+80h
	db	"ERAS",'E'+80h,	"ERRO",'R'+80h,	"ELS",'E'+80h,	"CAL",'L'+80h
	db	"MO",'N'+80h,	"LOCAT",'E'+80h,"MOD",'E'+80h,	"KE",'Y'+80h
	db	"PUS",'H'+80h,	"PO",'P'+80h,	"LABE",'L'+80h,	"RANDOMIZ",'E'+80h
	db	"OPTIO",'N'+80h,"LIN",'E'+80h,	"OPE",'N'+80h,	"CLOS",'E'+80h
	db	80h,		"FIEL",'D'+80h,	"GE",'T'+80h,	"PU",'T'+80h
	db	"SE",'T'+80h,	"FILE",'S'+80h,	"LFILE",'S'+80h,"DEVIC",'E'+80h
	db	"NAM",'E'+80h,	"KIL",'L'+80h,	"LSE",'T'+80h,	"RSE",'T'+80h
	db	"INI",'T'+80h,	"VDI",'M'+80h,	"MAXFILE",'S'+80h, 80h
	db	"T",'O'+80h,	"STE",'P'+80h,	"THE",'N'+80h,	"USIN",'G'+80h
	db	"SU",'B'+80h,	"BAS",'E'+80h,	"TA",'B'+80h,	"SP",'C'+80h
	db	"EQ",'V'+80h,	"IM",'P'+80h,	"XO",'R'+80h,	"O",'R'+80h
	db	"AN",'D'+80h,	"NO",'T'+80h,	">",'<'+80h,	"<",'>'+80h
	db	"=",'<'+80h,	"<",'='+80h,	"=",'>'+80h,	">",'='+80h
	db	'='+80h,	'>'+80h,	'<'+80h,	'+'+80h
	db	'-'+80h,	"MO",'D'+80h,	7dh+80h,	'/'+80h
	db	'*'+80h,	'^'+80h
	db	00h

CMDNAMEFE:
	db	80h,		"PSE",'T'+80h,	"PRESE",'T'+80h,"COLO",'R'+80h
	db	80h,		80h,		80h,		80h
	db	80h,		80h,		80h,		"PLA",'Y'+80h
	db	80h,		"BEE",'P'+80h,	80h,		80h
	db	80h,		80h,		80h,		80h
	db	"CGE",'N'+80h,	"PCOLO",'R'+80h,"SKI",'P'+80h,	"RLIN",'E'+80h
	db	"MOV",'E'+80h,	"RMOV",'E'+80h,	"PHOM",'E'+80h,	"HSE",'T'+80h
	db	"GPRIN",'T'+80h,"AXI",'S'+80h,	"CIRCL",'E'+80h,"TES",'T'+80h
	db	"PLO",'T'+80h,	"PAG",'E'+80h,	"MUSI",'C'+80h,	"TEMP",'O'+80h
	db	"CURSO",'R'+80h,"VERIF",'Y'+80h,"CL",'R'+80h,	"LIMI",'T'+80h
	db	"KLIS",'T'+80h,	80h,		80h,		"CLIC",'K'+80h
	db	"BOO",'T'+80h,	"DEVI",'$'+80h,	"DEVO",'$'+80h
	db	00h

FNCNAME:
	db	"IN",'T'+80h,	"AB",'S'+80h,	"SI",'N'+80h,	"CO",'S'+80h
	db	"TA",'N'+80h,	"LO",'G'+80h,	"EX",'P'+80h,	"SQ",'R'+80h
	db	"RN",'D'+80h,	"PEE",'K'+80h,	"AT",'N'+80h,	"SG",'N'+80h
	db	"FRA",'C'+80h,	"FI",'X'+80h,	"PA",'I'+80h,	"RA",'D'+80h
	db	"IN",'P'+80h,	"CDB",'L'+80h,	"CSN",'G'+80h,	"CIN",'T'+80h
	db	"DSK",'F'+80h,	"EO",'F'+80h,	"FPO",'S'+80h,	"LO",'C'+80h
	db	"LO",'F'+80h,	"PO",'S'+80h,	"FA",'C'+80h,	"SU",'M'+80h
	db	"FR",'E'+80h,	"LPO",'S'+80h,	"JO",'Y'+80h,	80h
	db	"CHR",'$'+80h,	"STR",'$'+80h,	"HEX",'$'+80h,	"OCT",'$'+80h
	db	"BIN",'$'+80h,	"MKI",'$'+80h,	"MKS",'$'+80h,	"MKD",'$'+80h
	db	"SPACE",'$'+80h,80h,		80h,		"AS",'C'+80h
	db	"LE",'N'+80h,	"VA",'L'+80h,	"CV",'S'+80h,	"CV",'D'+80h
	db	"CV",'I'+80h,	80h,		80h,		"ER",'R'+80h
	db	"ER",'L'+80h,	"CSRLI",'N'+80h,"STRPT",'R'+80h,"DT",'L'+80h
	db	80h,		80h,		"LEFT",'$'+80h,	"RIGHT",'$'+80h
	db	"MID",'$'+80h,	"INKEY",'$'+80h,"INST",'R'+80h,	"HEXCHR",'$'+80h
	db	"MEM",'$'+80h,	"SCRN",'$'+80h,	"VARPT",'R'+80h,"STRING",'$'+80h
	db	"TIME",'$'+80h,	80h,		80h,		"F",'N'+80h
	db	"US",'R'+80h,	80h,		80h,		"ATTR",'$'+80h
	db	80h,	 	"CHARACTER",'$'+80h
	db	00h

_CMDTBL:ds	CMDTBL-_CMDTBL
	org	CMDTBL

	dw	C_GOTO, C_GSUB, C_GO,   C_RUN,  C_RET,  C_RSTR, C_RESM, C_LIST
	dw	C_LLST, C_DEL,  C_RENM, C_AUTO, C_EDIT, C_FOR,  C_NEXT, C_PRT
	dw	C_LPRT, C_INPT, C_LINP, C_IF,   C_DATA, C_READ, C_DIM,  C_REM
	dw	C_END,  C_STOP, C_CONT, C_CLS,  C_CLER, C_ON,   C_LET,  C_NEW
	dw	C_POKE, C_OFF,  C_WHL,  C_WEND, C_RPT,  C_UNTL, RFERR,  RFERR
	dw	C_TRAC, C_TRON, C_TROF, C_SPED, RFERR,  RFERR,  C_DINT, C_DSNG
	dw	C_DDBL, C_DSTR, C_DEF,  RFERR,  C_LOAD, C_SAVE, C_MERG, C_CHAI
	dw	C_CNSL, RFERR,  C_OUT,  C_SRCH, C_WAIT, C_PAUS, C_WRIT, C_SWAP
	dw	C_ERAS, C_ERR,  C_ELSE, C_CALL, C_MON,  C_LOCA, C_MODE, C_KEY
	dw	C_PUSH, C_POP,  C_LBL,  C_RNDM, C_OPT,  C_LINE, C_OPEN, C_CLOS
	dw	RFERR,  C_FLD,  C_GET,  C_PUT,  C_SET,  C_FLS,  C_LFLS, C_DEV
	dw	C_NAME, C_KILL, C_LSET, C_RSET, C_INIT, C_VDIM, C_MAXF, RFERR

;	dw	C_TO,   C_STEP, C_THEN, C_USNG, C_SUB,  C_BASE, F_TAB,  F_SPC

CMDTBLFE:
	dw	RFERR,  C_PSET, C_PRST, C_COLR, RFERR,  RFERR,  RFERR,  RFERR
	dw	RFERR,  RFERR,  RFERR,  C_PLAY, RFERR,  C_BEEP, RFERR,  RFERR
	dw	RFERR,  RFERR,  RFERR,  RFERR,  C_CGEN, C_PCOL, C_SKIP, C_RLIN
	dw	C_MOVE, C_RMOV, C_PHOM, C_HSET, C_GPRT, C_AXIS, C_CRCL, C_TEST
	dw	C_PLOT, C_PAGE, C_MSIC, C_TEMP, C_CRSR, C_VRFY, C_CLR,  C_LIM
	dw	C_KLST, RFERR,  RFERR,  C_CLCK, C_BOOT, C_DEVI, C_DEVO


;CALL command
C_CALL:
	call	NARGMO
	push	hl
	call	FTOI
	pop	hl
	ld	(CALLSP+1),sp
	ld	sp,1000h
	call	CALLDE
CALLSP:
	ld	sp,0000h
	ret


;skip space
;input: hl=program address
;output: a=data, hl=next address
;destroy: f
SKIPSP:
	ld	a,(hl)
	cp	' '
	ret	nz
SKIPSPINC:
	inc	hl
	jr	SKIPSP


;check colon and line end
;input: hl=program address
;output: a=(hl), z-flag(1= 00 or ':')
;destroy: af
CHKCLN:
	ld	a,(hl)
	or	a
	ret	z
	cp	':'
	ret	z
	cp	' '
	ret	nz
CHKCLNINC:
	inc	hl
	jr	CHKCLN


;get comma (no error)
;input: hl=program address
;output: hl=address after comma, a=(hl), z-flag(1=comma)
;destroy: af
GETCMM:
	ld	a,(hl)
	inc	hl
	cp	','
	ret	z
	cp	' '
	jr	z,GETCMM
	dec	hl
	ret


;get comma (error if no comma)
;input: hl=program address
;output: hl=address after comma, z-flag(1=comma)
;destroy: af
_GETCMER:ds	GETCMER-_GETCMER
	org	GETCMER

	call	SKIPSP
	cp	','
	jp	nz,SNERR
	inc	hl
	ret


;DATA command
;LABEL command
C_DATA:
C_LBL:
	ld	c,':'
	jp	DATREM


;search for line number
;input: de=line number
;output: z=1,c=1) bc=address, hl=next line address
;        z=0,c=0) bc=next line address, hl=next next line address
;        z=1,c=0) bc=hl=end address
;destroy: af
SRCHLN:
	ld	hl,(LINENUM)
	or	a
	sbc	hl,de
	add	hl,de
	jr	nc,SLNSTRT	;less than or equal to current line number
	ld	hl,(PROGAD)
	ld	bc,0000h
	call	SKIPTO
	inc	hl
	call	SLNLP
	ret	c

SLNSTRT:
	ld	hl,(BASICAD)
SLNLP:
	ld	b,h
	ld	c,l
	ld	a,(hl)
	inc	hl
	or	(hl)
	jr	z,SLNEND	;program end
	inc	hl
	inc	hl
	ld	a,(hl)
	dec	hl
	cp	d		;
	jr	nz,SLNNZ
	ld	a,(hl)
	cp	e		;
SLNNZ:
	push	af		;
	push	de
	dec	hl
	ld	d,(hl)
	dec	hl
	ld	e,(hl)
	add	hl,de
	pop	de
	pop	af		;

	jr	c,SLNLP		;<de
	inc	hl
	jr	nz,SLNEND	;>de
	scf
SLNEND:
	dec	hl
	dec	bc
	dec	hl
	ret


;get variable address
;input: hl=program address
;output: b=variable name length, c=variable type, z-flag(1=array),
;        hl=1st character address, (PROGAD)=address after variable name
;destroy: af,de
CHKVAR:
	ld	a,(hl)
	inc	hl
	cp	' '
	jr	z,CHKVAR
	sub	'A'
	cp	'Z'-'A'+1
	jp	nc,SNERR
;	call	GETTYPE
;	ret

;get variable type
;input: hl=2nd character address, a=1st character-'A'
;output: b=variable name length, c=variable type, z-flag(1=array)
;        hl=1st character address, (PROGAD)=address after variable name
;destroy: af,de
GETTYPE:
	push	hl

	ld	de,DEFTYP
	add	a,e
	ld	e,a
	jr	nc,GETTPNC1
	inc	d
GETTPNC1:

	ld	bc,0002h
GETTPLP:
	inc	b
	ld	a,240
	cp	b
	jr	nc,GETTPNC2
	ld	b,a		;240
GETTPNC2:
	ld	a,(hl)
	inc	hl
	call	ALPNUM
	jr	c,GETTPLP

	cp	'%'
	jr	z,GETTPEND

;for speedup
	jr	nc,GETTPZ
	or	a
	jr	z,GETTPZ
;

	inc	c
	cp	'$'
	jr	z,GETTPEND
	ld	c,05h
	cp	'!'
	jr	z,GETTPEND
	ld	c,08h
	cp	'#'
	jr	z,GETTPEND

GETTPZ:
	dec	hl
	ld	a,(de)		;default type
	ld	c,a

GETTPEND:
	ld	a,(hl)
	cp	'('
	ld	(PROGAD),hl
	pop	hl
	dec	hl
	ret


;call CHKVAR and GETVAR
CHKGETV:
	call	CHKVAR
;	jp	GETVAR

;get variable address
;input: b=variable name length, c=variable type, hl=1st character address
;       z-flag(1=array)
;output: de=variable value address, (PROGAD)=next program address
;destroy: FAC1,FAC2,af,hl
GETVAR:
	jp	z,GETARR
	push	bc
	call	SRCHVAR
	call	nc,MAKEVAR
	ld	(PROGAD),hl
	pop	bc
	ret


;search for variable
;input: b=variable name length, c=variable type, hl=1st character address,
;       (PROGAD)=address after variable name
;output: de=variable value address, hl=(PROGAD), c-flag(1=found)
;        de=variable area last address, hl=no change, c-flag(0=not found)
;destroy: af
SRCHVAR:
	ld	de,(VARAD)
SRCHVLP1:
	ld	a,(de)		;variable type
	or	a
	ret	z		;not found

	push	hl		;1st character address
	push	de		;variable address
	inc	de
	cp	c
	jr	nz,VARNEXT2
	ld	a,(de)		;variable name length
	cp	b
	jr	nz,VARNEXT2
	inc	de

	push	bc		;b=name length, c=type
SRCHVLP2:
	ld	a,(de)
	cp	(hl)
	jr	nz,VARNEXT1
	inc	de
	inc	hl
	djnz	SRCHVLP2

;found
	pop	bc
	pop	af		;cancel stack
	pop	af		;cancel stack
	ld	hl,(PROGAD)
	scf
	ret

VARNEXT1:
	pop	bc		;b=name length, c=type
VARNEXT2:
	pop	hl		;variable address
	ld	a,(hl)		;type<=8
	inc	hl
	rlca
	jr	c,VARNEXTARR
	rrca
	and	0bfh		;bit6=fn mark
	add	a,(hl)		;<=8+240
	inc	a		;<=249
	ld	d,00h
	ld	e,a
VARNEXTEND:
	add	hl,de
	ex	de,hl
	pop	hl
	jr	SRCHVLP1

;skip array variable
VARNEXTARR:
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	dec	hl
	jr	VARNEXTEND


;make variable
;input: b=variable name length, c=variable type,
;       de=variable address, hl=variable name 1st character address
;output: c=variable type, de=variable value address,
;        hl=next address
;destroy: af,bc
MAKEVAR:
	push	hl		;1st character address
	push	de		;variable address
	push	bc		;b=variable name length, c=type
	ld	a,c
	and	0bfh		;bit6=fn mark, <=8
	add	a,b		;b<=240
	add	a,02h		;<=250
	ld	e,a
	ld	d,00h		;de=type(1)+length(1)+name(b)+value(c)
	call	MOVESTR
	pop	bc		;b=variable name length, c=type
	pop	hl		;variable address

	ld	(hl),c		;type
	inc	hl
	ld	(hl),b		;variable name length
	inc	hl
	ld	a,c		;type
	and	0bfh		;bit6=fn mark
	ld	c,b
	ld	b,00h
	ex	de,hl
	pop	hl		;variable name 1st character address
	ldir

	push	de
	ld	b,a		;type
	inc	b
	xor	a
MAKEVLP1:
	ld	(de),a		;=0
	inc	de
	djnz	MAKEVLP1
	pop	de

MAKEVLP2:
	ld	a,(hl)
	inc	hl
	call	ALPNUM
	jr	c,MAKEVLP2

	cp	'%'+1
	jr	nc,MAKEVNC
	cp	'#'
	ret	nc		;#$%
	cp	'!'
	ret	z
MAKEVNC:
	dec	hl
	ret


;get array address
;input: b=variable name length, c=variable type, hl=1st character address
;       (PROGAD)=address of '('
;output: de=array value address, (PROGAD)=next address of ')'
;destroy: af,hl
GETARR:
	push	bc		;b=name length, c=type
	push	hl		;1st character address
	call	SRCHARR
	pop	hl		;1st character address
	jr	c,ARROK

	push	de		;array address
	push	hl		;1st character address
	call	DIMAUTO
	pop	hl		;1st character address
	pop	de		;array address
ARROK:
	ex	de,hl
	inc	hl		;skip type
	inc	hl		;skip total size
	inc	hl		;skip total size
	ld	b,00h
	ld	c,(hl)		;name length		;<=240
	inc	c		;skip name length
	add	hl,bc		;skip name
	ld	c,(hl)		;dimensions
	add	hl,bc
	add	hl,bc
	push	hl		;array address (data part)
	ld	a,c
	ex	af,af'		;dimensions
	push	af
	ld	c,b		;=0
	push	bc		;pointer
	push	hl		;array address (data part)
	ld	hl,0001h
	ld	(TMP),hl	;multiplied sizes

GETALP:
	ld	hl,(TMP)
	push	hl		;multiplied sizes
	ld	hl,(PROGAD)
	inc	hl
	call	INT2ARG		;get index
BASE:
	ld	a,00h		;change with "OPTION BASE"
	or	a
	jr	z,BASE0
	dec	de		;if base=1
BASE0:
	pop	hl		;multiplied sizes
	ld	(TMP),hl
	pop	hl		;array address
	push	de		;index
	ld	d,(hl)		;de=size of each dimension
	dec	hl
	ld	e,(hl)
	dec	hl
	ex	(sp),hl		;push array address, pop index
	or	a
	sbc	hl,de
	add	hl,de
	jp	nc,BSERR

	push	de		;size
	ld	de,(TMP)	;multiplided sizes
	call	MULINT2		;hl = multiplied sizes * size
	pop	bc		;size
	ex	de,hl
	pop	hl		;array address
	ex	(sp),hl		;push array address, pop pointer
	add	hl,de		;pointer += multiplied sizes * index
	push	hl		;pointer
	ld	hl,(TMP)	;multiplied sizes
	ld	d,b		;size
	ld	e,c
	call	MULINT2		;hl = multiplied sizes * size
	ld	(TMP),hl
	pop	hl		;pointer
	ex	(sp),hl		;push pointer, pop array address
	push	hl		;array address
	ex	af,af'		;dimensions
	dec	a
	ld	b,a
	ex	af,af'
	ld	hl,(PROGAD)
	ld	a,(hl)
	cp	','
	jr	nz,NOTCMM
	inc	b
;	dec	b
;	jr	nz,GETALP
	djnz	GETALP
	jp	BSERR

NOTCMM:
	inc	b
	dec	b
	jp	nz,BSERR
	call	CHKRPAR

	pop	de		;arrasy address
	pop	hl		;pointer
	pop	af
	ex	af,af'
	pop	de		;arrasy address
	pop	bc		;b=name length, c=type
	push	bc		;b=name length, c=type
	ld	a,c
	ld	b,h
	ld	c,l
	add	hl,hl		;*2
	cp	02h
	jr	z,NCMMINT
	cp	03h
	jr	z,NCMMSTR
	add	hl,hl		;*4
	cp	05h
	jr	z,NCMMSNG
	add	hl,hl		;*8
	jr	NCMMDBL
NCMMSTR:
NCMMSNG:
	add	hl,bc		;*3 or *5
NCMMINT:
NCMMDBL:
	add	hl,de
	inc	hl
	ex	de,hl
	pop	bc		;b=name length, c=type
	ret


;input: b=variable name length, c=type, de=array address,
;       hl=1st character address, (PROGAD)=address of '('
DIMAUTO:
	ld	a,c		;type
	exx
	push	hl
	push	de
	push	bc
	ld	hl,(PROGAD)	;address of '('
	push	hl
	ld	h,00h		;element bytes
	ld	l,a		;type
	exx

	ld	(TMP),hl	;1st character address
	ld	hl,(PROGAD)	;address of '('
	inc	hl
	push	de		;array address
	push	de		;array address
	push	bc		;b=name length, c=type

	ld	c,00h
DIMAUTOLP:
	push	bc		;c=dimensions
	call	INT2ARG		;skip index
	pop	bc		;c=dimensions
	inc	c
	bit	2,c
	jp	nz,BSERR	;over 3 dimensions

	ld	de,000bh	;size=10+1
	ld	a,(BASE+1)
	or	a
	jr	z,DIMAUTOZ	;option base 0
	inc	de		;option base 1
DIMAUTOZ:
	push	de
	exx
	pop	de
	call	MULINT2
	ld	a,d
	or	e
	jp	nz,OMERR
	exx

	ex	de,hl
	pop	de		;d=name length, e=type
	ex	(sp),hl		;push size, pop array address
	push	hl		;array address
	push	de		;d=name length, e=type

	ld	hl,(PROGAD)
	ld	a,(hl)
	inc	hl
	cp	','
	jr	z,DIMAUTOLP
	cp	')'
	jp	nz,SNERR
	call	MAKEARR

	exx
	pop	hl
	ld	(PROGAD),hl	;address of '('
	pop	bc
	pop	de
	pop	hl
	exx

	ret


;search for array variable
;input: b=variable name length, c=type, hl=1st character address
;output: de=array value address, c-flag(1=found),
;destroy: af,hl
SRCHARR:
	ld	de,(VARAD)
SRCHALP1:
	ld	a,(de)		;type
	or	a
	ret	z		;not found

	push	hl		;1st character address
	inc	de
	bit	7,a
	jr	z,SRCHAZ	;not array

	push	bc		;b=name length, c=type
	push	de		;array address+1

;array variable
	and	7fh
	cp	c		;type
	jr	nz,ARRNEXT
	inc	de		;skip total size
	inc	de		;skip total size
	ld	a,(de)		;name length
	cp	b
	jr	nz,ARRNEXT

;compare name
SRCHALP2:
	inc	de
	ld	a,(de)
	cp	(hl)
	jr	nz,ARRNEXT
	inc	hl
	djnz	SRCHALP2
	pop	de		;array address+1
	dec	de
	pop	bc		;b=name length, c=type
	pop	af		;cancel stack
	ld	a,(hl)
	scf
	ret			;found

ARRNEXT:
	pop	hl		;array address+1
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	dec	hl
	add	hl,de
	ex	de,hl
	pop	bc		;b=name length, c=type
	pop	hl		;1st character address
	jr	SRCHALP1


;normal variable
SRCHAZ:
	ex	de,hl
	ld	d,00h
	ld	e,(hl)		;name length
	inc	hl		;skip name length
	add	hl,de		;skip name
;	ld	d,00h
	and	3fh		;bit6=fn mark
	ld	e,a		;type
	add	hl,de
	ex	de,hl
	pop	hl		;1st character address
	jr	SRCHALP1


;make array
;input: c=dimensions, hl'=element bytes, (TMP)=1st character address
;pushed: array address, size*[1,255], array address, variable name length/type,
;destroy: af,bc,de,hl
MAKEARR:
	pop	hl		;return address
	ld	de,(TMP)	;1st character address
	ld	(TMP),hl

	pop	hl		;h=name length, l=type
	push	bc		;c=dimensions
	push	de		;1st character address
	push	hl		;h=name length, l=type

	ld	l,h		;name length
	ld	h,00h
	ld	d,h		;=0
	ld	e,c		;dimensions
	add	hl,de
	add	hl,de		;name length+dimensions*2
	ld	e,05h		;d=0
	add	hl,de		;name length+dimensions*2+5 < 10000h
	exx
	push	hl		;element bytes
	exx
	pop	de		;element bytes
	add	hl,de		;name length+dimensions*2+5+element bytes
	jp	c,OMERR

	push	hl
	exx
	pop	de
	dec	de		;name length+dimensions*2+4+element bytes
	exx
	ex	de,hl

	call	MOVESTR

	pop	bc		;b=name length, c=type
	pop	hl		;1st character address
	pop	de		;e=dimensions

	ld	a,e
	pop	de		;array addrss
	push	af		;dimensions
	call	SETARNM
	ex	de,hl
	pop	af		;dimensions
	ld	(hl),a
	ld	c,a		;dimensions
	inc	hl

	ld	b,c
MKALP:
	pop	de		;size
	ld	(hl),e
	inc	hl
	ld	(hl),d
	inc	hl
	djnz	MKALP

	ld	(hl),b		;=0
	ld	d,h
	ld	e,l
	inc	de
	exx
	push	hl		;element bytes
	exx
	pop	bc
	ldir

	exx
	pop	hl		;array address
	inc	hl		;skip type
	ld	(hl),e		;de=name length+dimensions*2+4+element bytes
	inc	hl
	ld	(hl),d
	exx

	ld	hl,(TMP)
JPHL:
	jp	(hl)


;set array name
;input: b=name length, c=type, de=array address, hl=1st character address
;output: de=array address (next of name)
;destroy: af,b,hl
SETARNM:
	ex	de,hl
	ld	(hl),c		;type
	set	7,(hl)		;array mark
	ld	a,c
	inc	hl
	inc	hl		;skip total size
	inc	hl		;skip total size
	ld	(hl),b		;name length
	inc	hl
	ex	de,hl
	ld	c,b
	ld	b,00h
	ldir
	ld	c,a		;type
	ret


;move file buffer and string data area
;input: de=slide size(>0)
;destroy: af,bc,de,hl
MOVESTR:
	ld	hl,(TMPSTR)
	push	hl		;old (TMPSTR)
	add	hl,de
	jp	c,OMERR
	push	de
	call	CHKSTK
	pop	de
	ld	(TMPSTR),hl	;new (TMPSTR)
	ld	(FREEAD),hl
	ld	hl,(BUFP)
	ld	a,h
	or	l
	jr	z,MOVESTRZ
	add	hl,de
	ld	(BUFP),hl
MOVESTRZ:
	ld	hl,(STRPTR)
	add	hl,de
	ld	(STRPTR),hl

	ld	hl,(FBUFAD)
	add	hl,de
	ld	(FBUFAD),hl
	ex	de,hl

	ld	hl,(TMPSTR)
;	or	a
	sbc	hl,de		;new (TMPSTR) - new (FBUFAD)
	ld	b,h
	ld	c,l
	add	hl,de		;new (TMPSTR)
	ex	de,hl
	pop	hl		;old (TMPSTR)

	ld	a,b
	or	c
	ret	z
	dec	de
	dec	hl
	lddr
	ret


;SWAP command
C_SWAP:
	call	CHKGETV		;c=type, de=variable value address
	push	de		;1st variable value address
	push	bc		;c=1st variable type
	ld	b,00h
	ld	hl,FAC1
	ld	(hl),c
	inc	hl
	ex	de,hl
	ldir			;FAC1<-1st variable value
	call	PUSHF1		;1st variable value

	ld	hl,(PROGAD)
	call	SKIPSP
	cp	','
	jp	nz,SNERR
	inc	hl

	call	CHKGETV
	push	de		;2nd variable value address
	push	bc		;c=2nd variable type
	ld	b,00h
	ld	hl,FAC2
	ld	(hl),c
	inc	hl
	ex	de,hl
	ldir			;FAC2<-2nd variable value

	pop	bc		;2nd variable type
	ld	a,c
	cp	03h
	jr	z,SWAPSTR

	call	CNVTYP1		;convert 1st variable
	pop	de		;2nd variable value address
	call	POPF1		;1st variable value
	ld	hl,FAC1+1
	ld	b,00h
	ldir			;2nd variable<-FAC1

	call	PUSHF2
	call	POPF1

	pop	bc		;1st variable type

	ld	a,c
	cp	03h
	jp	z,TMERR
	call	CNVTYP1		;convert 1st variable
	pop	de		;1st variable value address
	ld	hl,FAC1+1
	ld	b,00h
	ldir			;1st variable<-FAC1
	ret

SWAPSTR:
	pop	de		;2nd variable value address
	call	POPF1		;1st variable value
	ex	de,hl
	pop	bc		;1st variable type
	pop	de		;1st variable value address
	ld	a,c
	cp	03h
	jp	nz,TMERR

	ld	b,03h
SWAPLP:
	ld	a,(de)
	ld	c,(hl)
	ld	(hl),a
	ld	a,c
	ld	(de),a
	inc	hl
	inc	de
	djnz	SWAPLP
	ret


;OUT command
C_OUT:
	call	INT1ARG2
	out	(c),e
	ret


;reset stack and variables
;destroy: af,bc,de,hl
RESSTK:
	call	C_CLR
	jr	LIMMAIN


;CLEAR command
C_CLER:
	call	NUMARG
	jp	z,C_CLR		;no numerical parameter
	jr	LIM2

;LIMIT command
C_LIM:
	call	NARGMO
LIM2:
	call	FTOI2
	ld	hl,(TMPSTR)
	ld	bc,STKMGN-1
	add	hl,bc
;	or	a
	sbc	hl,de
	jp	nc,FCERR
	ld	hl,STKINI
	or	a
	sbc	hl,de
	jp	c,FCERR
	ld	(STACK),de
LIMMAIN:
	pop	bc		;return address
	ld	sp,(STACK)
	ld	(SPERR+1),sp
	xor	a
	ld	(ERRHND+2),a
	push	bc		;return address
	ret


;DELETE command
C_DEL:
	pop	af		;cancel return address
	call	CHKCLN
	jp	z,MOERR
	call	LISTPAR

;de=start line number, bc=end line number
DELSTRT:
	push	bc
	call	SRCHLN
	pop	de		;end line number
	push	bc		;start address
	call	SRCHLN
	jr	c,DELC
	ld	h,b		;end address
	ld	l,c
DELC:
	pop	de		;start address

	ex	de,hl
	or	a
	sbc	hl,de		;start-end
	add	hl,de
	jp	nc,EDIT
	ex	de,hl

	ld	b,d
	ld	c,e
	push	bc		;dummy,(bc)=0
	call	DELPRG
	jp	EDIT


;get parameters for LIST/DELETE
;input: hl=program address
;output: de=start line number, bc=end line number
LISTPAR:
	call	GETLN
	push	de		;start line number
	call	CHKCLN
	jr	z,LISTPEND

	cp	MINUS_
	jr	z,LISTPZ
	cp	','
	jp	nz,SNERR
LISTPZ:
	call	CHKCLNINC
	jr	nz,LISTPNZ
	pop	de
	ld	bc,65534
	ret

LISTPNZ:
	call	GETLN
	call	CHKCLN
	jp	nz,SNERR
LISTPEND:
	ld	b,d
	ld	c,e
	pop	de
	ret


;RENUM command
C_RENM:
	pop	af		;cancel return address
	ld	de,000ah	;default 1st parameter (new line number)
	ld	(RENMP3+1),de	;default 3rd parameter (increment)
	ld	b,d		;default 2nd parameter (start line number)
	ld	c,d

	call	GETLNSP		;de=1st parameter (new line number)
	jr	z,RENMP1OK
	ld	de,000ah
RENMP1OK:
	call	GETCMM
	jr	nz,RENMCHK
	push	de
	call	GETLNSP		;2nd parameter (start line number)
	jr	nz,RENMNOP2
	ld	b,d
	ld	c,e
RENMNOP2:
	pop	de
	call	GETCMM
	jr	nz,RENMCHK
	push	de
	call	GETLNSP		;3rd parameter (increment)
	jp	nz,FCERR
	ld	(RENMP3+1),de
	pop	de

;error check
RENMCHK:
	call	CHKCLN
	jp	nz,SNERR
	push	de		;1st parameter (new line number)
	push	bc		;2nd parameter (start line number)

	ld	hl,(BASICAD)
RENMLP1:
	ld	a,(hl)
	inc	hl
	or	(hl)
	push	hl		;
	jr	z,RENMSTRT

	inc	hl
	ld	a,(hl)
	inc	hl
	ld	h,(hl)
	ld	l,a
;	or	a
	sbc	hl,bc		;bc=2nd parameter (start line number)
	jr	nc,RENMNC

	add	hl,bc
	or	a
	sbc	hl,de
	add	hl,de
	jp	nc,FCERR
	jr	RENMCHKEND

RENMNC:
	ld	a,d
	and	e
	inc	a
	jp	z,FCERR		;de=65535

	pop	hl		;
	push	hl		;
	push	de
	ld	d,(hl)
	dec	hl
	ld	e,(hl)
	add	hl,de
	pop	de
	inc	hl
	ld	a,(hl)
	inc	hl
	or	(hl)
	jr	z,RENMSTRT

RENMP3:
	ld	hl,0000h	;3rd parameter (increment)
	add	hl,de
	jp	c,FCERR
	ex	de,hl

RENMCHKEND:
	pop	hl		;
	push	de
	ld	d,(hl)
	dec	hl
	ld	e,(hl)
	add	hl,de
	pop	de
	jr	RENMLP1

;line number -> address
RENMSTRT:
	pop	hl
	ld	hl,(BASICAD)
	dec	hl
RENMLP2:
	call	SRCHRNM
	jr	z,RENMCLN
	push	de		;current line number
	push	hl		;program address (0b)
	inc	hl
	ld	e,(hl)		;de=line number to be searched for
	inc	hl
	ld	d,(hl)
	call	SRCHLN
	jr	nc,RENMUL	;not found
	pop	hl		;program address
	pop	de		;current line number
;	inc	(hl)		;0b->0c

RENMAD:
	inc	hl
	ld	(hl),c		;line number -> address
	inc	hl
	ld	(hl),b
	inc	hl
	jr	RENMLP2

RENMUL:
	push	de		;line number to be searched for
	ld	hl,ERR08	;"UNDEFINED LABEL"
	call	PUTS
	pop	hl		;line number to be searched for
	call	PUTI2
	pop	hl		;program address
	ex	(sp),hl		;current line number
	call	INLNUM
	call	PUTNL
	pop	hl		;program address
	ld	bc,0000h
	jr	RENMAD

;change line number
RENMCLN:
	pop	de		;2nd parameter (start line number)
	call	SRCHLN
	inc	bc
	pop	de		;1st parameter (new line number)
RENMLP3:
	ld	h,b
	ld	l,c
	ld	c,(hl)
	inc	hl
	ld	b,(hl)
	ld	a,b
	or	c
	jr	z,RENMWLN
	inc	hl
	ld	(hl),e
	inc	hl
	ld	(hl),d
	dec	hl
	dec	hl
	dec	hl
	add	hl,bc
	ld	b,h
	ld	c,l
	ld	hl,(RENMP3+1)
	add	hl,de
	ex	de,hl
	jr	RENMLP3

;write new line number
RENMWLN:
	ld	hl,(BASICAD)
	dec	hl
RENMLP4:
	call	SRCHRNM
	jp	z,EDIT
;	dec	(hl)		;0c->0b
	inc	hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
	ld	a,d
	or	e
	jr	z,RENMLP4

	dec	hl
	dec	hl
	inc	de
	inc	de
	inc	de
	ex	de,hl
	ldi
	ldi
	ex	de,hl
	jr	RENMLP4

;search line number to be renumbered
;input: hl=address
;output: hl=address, de=line number, z-flag(1=program end)
;destroy: af,bc
SRCHRNM:
	ld	bc,000bh
	call	SKIPTO
	or	a
	ret	nz
	inc	hl
	ld	a,(hl)
	inc	hl
	or	(hl)
	ret	z
	inc	hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
	jr	SRCHRNM


;EDIT command
C_EDIT:
	pop	af		;cancel return address
	call	GETLNSP
	jr	z,CEDITZ
	ld	de,(ERLVAL)	;error line number
CEDITZ:
	call	CHKCLN
	jp	nz,SNERR
	call	SRCHLN
	jp	nc,EDIT
	ld	h,b
	ld	l,c
	inc	hl
	inc	hl
	inc	hl
	call	LIST1
	ld	hl,(DSPXY)
CEDITLP1:
	call	PREVPOS
	call	CHKLINE2
	jr	nz,CEDITLP1

	call	CHGVRAM
CEDITLP2:
	call	NEXTPOS
	push	hl
	call	XY2AD
	ld	a,(hl)
	or	a
	pop	hl
	jr	nz,CEDITLP2
	call	CHGDRAM

	call	NEXTPOS
	call	SETCSR
	jp	EDIT2


;UNTIL command
C_UNTL:
	pop	af		;cancel return address

	ld	a,REPEAT_
	call	SRCHSTK
	jp	nc,URERR	;not found or gosub,for,while identifer found
	inc	sp		;identifier
	ld	hl,(PROGAD)
	call	NARGMO
	ld	a,(FAC1)
	cp	02h
	jr	z,UNTLINT
;float
	ld	a,(FAC1+1)	;exponent
;	cp	80h
;	jr	nc,UNTLTRUE	;|value|>=0.5
	rlca
	jr	c,UNTLTRUE	;|value|>=0.5
	jr	UNTLFALSE
;integer
UNTLINT:
	ld	de,(FAC1+1)
	ld	a,d
	or	e
	jr	z,UNTLFALSE

UNTLTRUE:
	pop	af		;line number
	pop	af		;address after REPEAT
	jp	INTPEND

UNTLFALSE:
	pop	de		;line number
	ld	(LINENUM),de
	pop	hl		;address after REPEAT
	ld	(PROGAD),hl
	jp	RPTLP


;RETURN command
C_RET:
	pop	bc		;cancel return address
RETLP:
	ld	a,GOSUB_
	call	SRCHSTK
	jp	nc,RGERR	;no identifier
	jr	z,RETOK
	ld	hl,001bh	;FOR command
	cp	FOR_
	jr	z,RETZ
	ld	l,05h		;WHILE/REPEAT command
RETZ:
	add	hl,sp
	ld	sp,hl
	jr	RETLP
RETOK:
	inc	sp
	ld	hl,(PROGAD)
	call	SKIPSP
	cp	0bh
	jr	z,RETLNLBL		;line number
	cp	22h		;double quotation mark
	jp	nz,RETNOPAR

;with line number or label
RETLNLBL:
	pop	af		;cancel line number
	pop	af		;cancel program address
	call	GOTOMAIN	;bc=new program address
	ld	h,b
	ld	l,c
	jp	RECSP2

;no parameter
RETNOPAR:
	pop	hl		;line number
	ld	(LINENUM),hl
	pop	hl		;program address
	jp	RECSP2


;search stack for gosub/for/while/repeat identifier
;input: a=identifier to be searched for
;output: a=identifier, hl=sp (before calling)
;        z-flag=1,c-flag=1: found
;        z-flag=1,c-flag=0: no identifier
;        z-flag=0,c-flag=1: different identifer (larger) found
;        z-flag=0,c-flag=0: different identifer (smaller) found
;destroy: f,bc,hl
SRCHSTK:
        ld      hl,0002h
        add     hl,sp

;search (hl)
;destroy: f,bc
SRCHSTK2:
        ld      bc,(STACK)
	or	a
	sbc	hl,bc
	add	hl,bc
        ret     nc              ;(z-flag=1)
	cp	(hl)
        ld      a,(hl)
	ret	nz
	scf
        ret


;WHILE command
C_WHL:
	pop	af		;cancel return address
	ld	de,(LINENUM)

;hl=condition address, de=line number
WHLCND:
	push	hl		;WHILE condition address
	push	de		;line number
	call	NARGMO
	ld	a,(FAC1)
	cp	02h
	jr	z,WHLINT
;float
	ld	a,(FAC1+1)	;exponent
;	cp	80h
;	jr	nc,WHLTRUE	;|value|>=0.5
	rlca
	jr	c,WHLTRUE	;|value|>=0.5
	jr	WHLFALSE
;integer
WHLINT:
	ld	de,(FAC1+1)
	ld	a,d
	or	e
	jr	z,WHLFALSE

WHLTRUE:
	ld	a,WHILE_
	push	af
	inc	sp
;(sp)=identifier(1),line number(2),condition address(2)
	jp	RECSP2

WHLFALSE:
	pop	af		;line number
	pop	af		;WHILE condition address

	ld	a,01h
	ld	(NESTCNT+1),a
	ld	bc,WHILE_*256+WEND_
WHLLP:
	call	SKIPNEST
	or	a
	jp	z,WHERR
;wend found
	ld	a,(NESTCNT+1)
	dec	a
	ld	(NESTCNT+1),a
	inc	hl
	jr	nz,WHLLP
	ld	(PROGAD),hl
	jp	RECSP2


;WEND command
C_WEND:
	pop	af		;return address

WENDLP:
	ld	a,WHILE_
	call	SRCHSTK
	jp	nc,WEERR	;not found or gosub,for identifier found
	jr	z,WENDZ		;while
;repeat
	ld	hl,0005h
	add	hl,sp
	ld	sp,hl
	jr	WENDLP

WENDZ:
	inc	sp		;identifier
	pop	de		;line number
	ld	(LINENUM),de
	pop	hl		;condition address
	ld	(PROGAD),hl
	jp	WHLCND


;REPEAT command
C_RPT:
	call	SKIPSP
	cp	ON_
	jr	z,RPTONOFF
	sub	OFF_
	jr	z,RPTONOFF	;a=0

;repeat-until
	pop	af		;cancel return address
	ld	de,(LINENUM)
RPTLP:
	push	hl		;address after REPEAT
	push	de		;line number
	ld	a,REPEAT_
	push	af
	inc	sp
;(sp)=identifer(1),line number(2),address after REPEAT(2)
	ld	(PROGAD),hl
	jp	RECSP2

RPTONOFF:
	ld	(RPTON),a	;repeat off=0
	inc	hl
	ld	(PROGAD),hl
	ret


;GOSUB command
C_GSUB:
	pop	af		;cancel return address
GSUB:
	ld	de,(LINENUM)
	push	de		;line number
	call	GOTOMAIN	;bc=new program address
	call	ONSKIP

	pop	de		;line number
	push	hl		;address after GOSUB **
	push	de		;line number
	call	CHKFRE
	ld	a,GOSUB_
	push	af		;gosub identifier
	inc	sp
	ld	h,b
	ld	l,c
	jp	RECSP2


;RESUME command
C_RESM:
	pop	af		;cancel return address
	ld	a,(ERRHND+2)
	or	a
	jp	z,RWERR
	ld	a,(DURERRHND+1)
	or	a
	jp	z,RWERR

	call	SKIPSP
	cp	NEXT_
	jr	z,RESMNXT
	cp	22h
	jr	z,RESMLBL
	call	GETLN
	jr	nz,RESM0
	ld	a,d
	or	e
	jr	z,RESM0

;resume with line number
RESMLN:
	push	hl
	push	de
	call	SRCHLN
	pop	de
	pop	hl
RESMEND:
	ld	h,b
	ld	l,c
RESMEND2:
	xor	a
	ld	(DURERRHND+1),a
	jp	INTPRT

;resume with label
RESMLBL:
	call	SRCHLBL
	jr	RESMEND

;resume 0
RESM0:
	ld	hl,0000h		;error address
	jr	RESMEND2

;resume next
RESMNXT:
	xor	a
	ld	(DURERRHND+1),a
	ld	hl,(RESM0+1)
	ld	bc,':'
	call	SKIPTO
	or	a
	jp	nz,INTPRT
	inc	hl
	jp	RUNLP


;ON command
C_ON:
	call	SKIPSP
	cp	ERROR_
	jr	z,ONERR
	call	INT2ARG
	ld	a,(hl)
	ld	b,a		;
	cp	GOSUB_+1	;80h(GOTO) or 81h(GOSUB)
	jr	c,ONLP
	cp	RETURN_		;RETURN
	jr	z,ONLP
	cp	RSTR_		;RESTORE
	jr	z,ONLP
	cp	RESUME_		;RESUME
	jp	nz,SNERR

ONLP:
	inc	hl
	dec	de
	ld	a,d
	or	e
	jr	z,ONZ
	call	SKIPSP
	ld	a,(hl)
	inc	hl
	cp	22h		;double quotation mark
	jr	z,ONLBL
	cp	0bh
	jp	nz,SNERR
	inc	hl
	inc	hl
ONCHKCM:
	call	SKIPSP
	cp	','
	jr	z,ONLP
	ld	(PROGAD),hl
	ret

;label
ONLBL:
	call	SKIPDQ
	jr	ONCHKCM

ONZ:
	ld	(PROGAD),hl
	pop	af		;cancel return address
	ld	a,b		;
	jp	COMMAND

;skip for ON GOSUB/RESTORE/RESUME/ERROR GOTO
ONSKIP:
	ld	a,(hl)
	inc	hl
	cp	','
	jr	z,ONSKIP
	cp	' '
	jr	z,ONSKIP
	cp	0bh
	jr	z,ONSKIPLN
	dec	hl
	cp	22h
	ret	nz
;label
	inc	hl
	call	SKIPDQ
	jr	ONSKIP

;line number
ONSKIPLN:
	inc	hl
	inc	hl
	jr	ONSKIP

ONERR:
	call	SKIPSPINC
	cp	GOTO_
	jp	nz,SNERR
	call	SKIPSPINC
	cp	22h		;double quotation mark
	jr	z,ONERRLBL
	call	GETLN
	jp	nz,SNERR
	ld	a,d
	or	e
	jr	z,ONERR0	;on error goto 0
	push	hl
	push	de
	call	SRCHLN
	jp	nc,ULERR
	pop	de
	pop	hl
ONERREND:
	ld	(PROGAD),hl
	ld	(ERRHND+1),bc
	ret

ONERRLBL:
	call	SRCHLBL
	jr	ONERREND

ONERR0:
	ld	(PROGAD),hl
	ld	a,(DURERRHND+1)
	push	af
	call	CLRERRHND
	pop	af
	or	a
	ret	z	;not during error handling
;during error handling
	ld	sp,(SPERR+1)
	jp	PUTERR


;GO command
C_GO:
	call	SKIPSP
	inc	hl
	cp	SUB_		;GO SUB
	jp	z,C_GSUB
	cp	TO_
	jp	nz,SNERR
;	jp	C_GOTO		;GO TO


;GOTO command
C_GOTO:
	pop	af		;cancel return address
	call	GOTOMAIN
	ld	h,b
	ld	l,c
	jp	INTPRT

;input: hl=program address
;output: bc=(PROGAD)=address to go, hl=address after line number or label
;destroy: af,de
GOTOMAIN:
	call	SKIPSP
	cp	22h		;double quotation mark
	jr	z,GOTOLBL
	call	GETLN
	jp	nz,SNERR

;with line number
	ld	(PROGAD),hl
	push	hl
	call	SRCHLN
	pop	hl
	jr	c,GOTOEND
	jp	ULERR

;GOTO "LABEL"
GOTOLBL:
	call	SRCHLBL
GOTOEND:
	ld	(LINENUM),de
	ld	(PROGAD),bc
	ret


;search for label
;input: hl=" address
;output: bc=address to go, de=line number, hl=address after label parameter
;destroy: af
SRCHLBL:
	inc	hl
	push	hl		;
	ld	hl,(BASICAD)
	inc	hl
	inc	hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
SLBLLP1:
	ld	bc,LABEL_
	call	SKIPTO
	or	a
	jr	z,SLBLNXT

;label command found
	call	SKIPSPINC
	cp	22h		;double quotation mark
	jr	nz,SLBLLP1
	inc	hl
	pop	bc		;
	push	bc		;
SLBLLP2:
	ld	a,(bc)
	or	a
	jr	z,SLBLLAST
	inc	bc
	cp	22h		;double quotation mark
	jr	z,SLBLLAST
	cp	(hl)
	inc	hl
	jr	z,SLBLLP2
	dec	hl

SLBLDQ:
	call	SKIPDQ
	jr	nz,SLBLLP1
;	jr	SLBLNXT

;next line
SLBLNXT:
	inc	hl
	ld	a,(hl)
	inc	hl
	or	(hl)
	jp	z,ULERR
	inc	hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
	jr	SLBLLP1

;last character?
SLBLLAST:
	ld	a,(hl)
	or	a
	jr	z,SLBLEND
	inc	hl
	cp	22h
	jr	nz,SLBLDQ

;found
SLBLEND:
	pop	af		;cancel stack
	push	bc
	ld	b,h
	ld	c,l
	pop	hl
	ret


;skip to target or 00h
;input: hl=address, c=target1, b=target2
;output: hl=address, a=last data
;destroy: f
SKIPTO:
	call	SKIPSP
	or	a
	ret	z
	cp	':'
	jr	z,SKIPTOCLN
	cp	c
	ret	z
	cp	b
	ret	z

	inc	hl
	cp	20h
	jr	c,SKIPTONUM
	cp	22h		;double quotation mark
	jr	z,SKIPTODQ
	cp	DATA_
	jr	z,SKIPTODATA
	cp	REM_
	jr	z,SKIPTOREM
	cp	0feh
	jr	c,SKIPTO
	inc	hl
	jr	SKIPTO		;fe+xx, ff+xx

SKIPTOREM:
	ld	a,(hl)
	or	a
	ret	z
SKIPTOAP:
	inc	hl
	jr	SKIPTOREM

SKIPTOCLN:
	inc	hl
	ld	a,(hl)
	cp	27h		;apostrophe
	jr	z,SKIPTOAP
	ld	a,c
	cp	':'
	jr	z,SKIPTOZ
	ld	a,b
	cp	':'
	jr	nz,SKIPTO
SKIPTOZ:
	dec	hl
	ret

SKIPTONUM:
	call	SKIPNUM
	jr	SKIPTO

SKIPTODQ:
	call	SKIPDQ
	jr	SKIPTO

SKIPTODATA:
	ld	a,(hl)
	or	a
	ret	z
	cp	':'
	jr	z,SKIPTO
	inc	hl
	cp	22h
	jr	nz,SKIPTODATA
	call	SKIPDQ
	ret	z
	jr	SKIPTODATA

;skip number
;input: hl=number address+1, a=first data
;output: hl=address after number
;destroy: f
SKIPNUM:
	cp	0bh
	ret	c		;01-0a
	inc	hl
	inc	hl
	cp	15h
	ret	c		;0b,0d,0e,0f,12
	inc	hl
	inc	hl
	inc	hl
	ret	z		;15=single
	inc	hl
	inc	hl
	inc	hl
	ret			;18=double

;skip double quotation string
;input: hl=" address+1
;output: a=00h or 22h, hl=00h address or 22h address+1, z-flag(1:a=00h)
;destroy: f
SKIPDQ:
	ld	a,(hl)
	or	a
	ret	z
	inc	hl
	cp	22h
	jr	nz,SKIPDQ
	or	a
	ret


;IF command
C_IF:
	call	NARGMO
	ld	a,(hl)
	inc	hl
	cp	THEN_
	jr	z,IFTHEN
	cp	GOTO_
	jr	z,IFTHEN
	dec	hl
IFTHEN:
	ld	a,01h
	ld	(NESTCNT+1),a
	ld	(PROGAD),hl
	ld	a,(FAC1)
	cp	02h
	jr	z,IFINT
;float
	ld	a,(FAC1+1)	;exponent
;	cp	80h
;	jr	nc,IFTRUE	;|value|>=0.5
	rlca
	jr	c,IFTRUE	;|value|>=0.5
	jr	IFFALSE

IFINT:
	ld	de,(FAC1+1)
	ld	a,d
	or	e
	jr	z,IFFALSE

IFTRUE:
	call	SKIPSP
	ld	(PROGAD),hl
	cp	22h
	jp	z,C_GOTO
	cp	0bh
	jp	z,C_GOTO
	pop	af		;cancel return address
	jp	INTPRT		;skip colon check

IFFALSE:
	ld	bc,IF_*256+':'
	call	SKIPNESTIF
	ld	(PROGAD),hl
	or	a
	ret	z

	inc	hl
	ld	a,(hl)
	cp	ELSE_
	jr	nz,IFFALSE
	inc	hl
	ld	de,NESTCNT+1
	ld	a,(de)
	dec	a
	ld	(de),a
	jr	nz,IFFALSE
	jr	IFTRUE


;skip nest (b...c) for IF-ELSE
;input: b=loop-start value, c=loop-end value, hl=program address, (NESTCNT+1)
;output: a=00h(line end) or c, hl=00h address or c address+1, (PROGAD)
;destroy: f
SKIPNESTIF:
	ld	(PROGAD),hl
	call	SKIPTO
	or	a
	ret	z
	cp	c
	ret	z

;b found
	inc	hl
NESTCNT:
	ld	a,00h
	inc	a
	ld	(NESTCNT+1),a
	jr	SKIPNESTIF


;skip nest (b...c) for FOR-NEXT and WHILE-WEND
;input: b=loop-start value, c=loop-end value, hl=program address, (NESTCNT+1)
;output: a=00h(program end) or c, hl=3rd 00 address or c address+1, (PROGAD)
;destroy: f,de
SKIPNEST:
	call	SKIPNESTIF
	or	a
	ret	nz
;new line
	inc	hl
	ld	a,(hl)
	inc	hl
	or	(hl)
	ret	z
	inc	hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
	ld	(LINENUM),de
	jr	SKIPNEST


;DEFINT command
C_DINT:
	ld	c,02h
	db	11h		;ld de,

;DEFSTR command
C_DSTR:
	ld	c,03h
	db	11h		;ld de,

;DEFSNG command
C_DSNG:
	ld	c,05h
	db	11h		;ld de,

;DEFDBL command
C_DDBL:
	ld	c,08h
DEFLP1:
	call	SKIPSP
	cp	'A'
	jp	c,SNERR
	cp	'Z'+1
	jp	nc,SNERR
	ld	e,a		;

	call	SKIPSPINC
	cp	MINUS_
	ld	b,01h
	jr	nz,SETDEFTYP

	call	SKIPSPINC
	cp	'Z'+1
	jp	nc,SNERR
	sub	e		;
	jp	c,SNERR
	inc	hl
	ld	b,a
	inc	b

SETDEFTYP:
	push	hl
	ld	a,e		;
	sub	'A'
	ld	e,a
	ld	hl,DEFTYP
	ld	d,00h
	add	hl,de
DEFLP2:
	ld	(hl),c
	inc	hl
	djnz	DEFLP2
	pop	hl

	call	SKIPSP
	ld	(PROGAD),hl
	inc	hl
	cp	','
	jr	z,DEFLP1
	ret

DEFTYP:
	db	05h, 05h, 05h, 05h, 05h, 05h, 05h, 05h
	db	05h, 05h, 05h, 05h, 05h, 05h, 05h, 05h
	db	05h, 05h, 05h, 05h, 05h, 05h, 05h, 05h
	db	05h, 05h


;WAIT command
C_WAIT:
	call	INTARG
	ld	a,(hl)
	cp	','
	jp	nz,SNERR
	push	de		;1st parameter
	call	INT1INC
	push	af		;2nd parameter
	ld	a,(hl)
	cp	','
	ld	e,00h
	call	z,INT1INC	;e=3rd parameter
	pop	hl		;h=2nd parameter
	pop	bc		;bc=1st parameter
CWAITLP:
	in	a,(c)
	xor	e
	and	h
	jr	z,CWAITLP
	ret


;SKIP command
C_SKIP:
	call	CHKTMODE
	call	INT2ARG
	ld	hl,20
	add	hl,de
	ld	a,h
	or	a
	jp	nz,FCERR
	ld	a,l
	cp	40+1
	jp	nc,FCERR
	sub	20
	ret	z		;SKIP 0
	ld	b,a
	jp	m,SKIPNEG

;SKIP [1,20]
SKIPPOS:
	ld	a,0ah
	call	PRINTER
	djnz	SKIPPOS
	ret

;SKIP [-20,-1]
SKIPNEG:
	ld	a,03h
	call	PRINTER
	inc	b
	jr	nz,SKIPNEG
	ret	


;get x,y for plotter
;input: hl=program address
;output: de=x, hl=y
;destroy: af,bc
GETPXY:
	call	INT2ARG
	push	de		;x
	call	GETCMER
	call	INT2ARG
	call	CHKPY
	pop	hl		;x
	ex	de,hl
;	call	CHKPX
;	ret


;check x for plotter [-480,480]
;input: de=x
;destroy: af,bc
CHKPX:
	push	hl
	ld	hl,480
	add	hl,de
	ld	bc,480+480+1
	or	a
	sbc	hl,bc
	jp	nc,FCERR
	pop	hl
	ret


;check y for plotter [-999,999]
;input: de=y
;destroy: f,bc
CHKPY:
	push	hl
	ld	hl,999
	add	hl,de
	ld	bc,999+999+1
	or	a
	sbc	hl,bc
	jp	nc,FCERR
	pop	hl
	ret


;put x,y to printer
;input: de=x, hl=y
;destroy: af,bc,de,hl,FAC1
PUTPRNXY:
	push	hl		;y
	call	PUTPRNI2
	ld	a,','
	call	PRINTER
	pop	de		;y
;	jr	PUTPRNI2

;put integer to printer [-999,999]
;input: de
;destroy: af,bc,de,hl
PUTPRNI2:
	ex	de,hl
	call	SETI2
	call	ITOA
PUTPRNI2LP:
	ld	a,(hl)
	or	a
	ret	z
	inc	hl
	cp	' '
	jr	z,PUTPRNI2LP
	call	PRINTER	
	jr	PUTPRNI2LP


;RLINE command
C_RLIN:
	ld	a,'J'
RLIN2:
	push	af		;command: 'J' or 'D'
	call	CHKGMODE

	call	SKIPSP
	cp	'%'
	ld	de,00h		;continuous
	jr	nz,RLINNZ
	call	INT1INC
	call	GETCMER
	dec	e
	ld	a,e
	cp	15+1
	jp	nc,FCERR
RLINNZ:
	ld	a,'L'
	call	PRINTER
	push	hl
	call	PUTPRNI2
	pop	hl
RLINLP:
	call	GETPXY
	pop	af		;command: 'J' or 'D'
	push	af		;command: 'J' or 'D'
	call	PRINTER
	call	PUTPRNXY
	call	PUTPRNCR
	ld	hl,(PROGAD)
	call	GETCMM
	jr	z,RLINLP
	pop	af		;command: 'J' or 'D'
	ret


;MOVE command
C_MOVE:
	ld	a,'M'
MOVE2:
	push	af
	call	CHKGMODE
	call	GETPXY
	pop	af
	call	PRINTER
	call	PUTPRNXY
	jp	PUTPRNCR


;RMOVE command
C_RMOV:
	ld	a,'R'
	jr	MOVE2


;PCOLOR command
C_PCOL:
	call	INT1ARG
	cp	04h
	jp	nc,FCERR
	ld	a,(PRNMOD)
	or	a
	push	af		;
	ld	a,02h
	call	z,PRINTER	;chage to graphic mode if in text mode
	ld	a,'C'
	call	PRINTER
	ld	a,e
	add	a,'0'
	call	PRINTER
	pop	af		;
	ret	nz		;graphic mode
	ld	a,'A'
	jp	PRINTER	


;PHOME command
C_PHOM:
	call	CHKGMODE
	ld	a,'H'
	jp	PRINTER


;HSET command
C_HSET:
	call	CHKGMODE
	ld	a,'I'
	jp	PRINTER


;GPRINT command
C_GPRT:
	call	CHKGMODE
	call	SKIPSP
	cp	'['
	jr	nz,GPRTSTR
	call	INT1INC
	cp	3fh+1
	jp	nc,FCERR
	push	af		;width parameter
	call	GETCMER
	call	INT1ARG
	cp	03h+1
	jp	nc,FCERR
	ld	b,a		;rotation parameter
	call	SKIPSP
	cp	']'
	jp	nz,SNERR
	inc	hl
	call	GETCMER

	ld	a,'Q'
	call	PRINTER
	ld	a,b		;rotation parameter
	add	a,'0'
	call	PRINTER
	ld	a,'S'
	call	PRINTER
	pop	af		;width parameter
	call	DIVI10
	call	PRINTER
	ld	a,c
	call	PRINTER

GPRTSTR:
	call	STRARG	
	or	a		;string length
	ret	z
	ld	b,a
	ld	a,'P'
	call	PRINTER
GPRTLP:
	ld	a,(de)
	inc	de
	call	PRNCODE
	call	PRINTER
	djnz	GPRTLP
PUTPRNCR:
	ld	a,0dh
	jp	PRINTER


;divide by 10
;input: a
;output: a=quotient character code, c=remainder character code
;destroy:af,b
DIVI10:
	ld	b,0ffh
DIVI10LP:
	inc	b
	sub	0ah
	jr	nc,DIVI10LP
	add	a,0ah+'0'
	ld	c,a
	ld	a,b
	add	a,'0'
	ret


;AXIS command
C_AXIS:
	call	CHKGMODE
	call	INT1ARG
	cp	02h
	jp	nc,FCERR
	push	af		;1st parameter
	call	GETCMER
	call	INT2ARG
	push	de		;2nd parameter
	call	CHKPY
	call	GETCMER
	call	INT1ARG
	or	a
	jp	z,FCERR

	ld	a,'L'
	call	PRINTER
	ld	a,'0'		;solid line
	call	PRINTER

	ld	a,'X'
	call	PRINTER
	pop	hl		;2nd parameter
	pop	af		;1st parameter
	add	a,'0'
	call	PRINTER
	ld	a,','
	call	PRINTER		;1st
	push	de		;3nd parameter
	ex	de,hl
	call	PUTPRNI2	;2nd parameter
	ld	a,','
	call	PRINTER
	pop	de		;3rd parameter
	call	PUTPRNI2
	jr	PUTPRNCR


;CIRCLE command
C_CRCL:
	call	CHKGMODE
	call	GETPXY
	ld	(CRCLX+1),de	;x
	ld	(CRCLY+1),hl	;y
	ld	hl,(PROGAD)
	call	GETCMER
	call	INT2ARG
	ld	a,d
	rlca
	jp	c,FCERR		;r<0
	call	CHKPX
	ld	(CRCLR+1),de	;r
	call	GETCMER
	call	NARGMO
	call	PUSHF1		;start angle

	ld	hl,(PROGAD)
	call	GETCMER
	call	NARGMO
	call	POPF2		;start angle
	call	PUSHF2		;start angle
	call	SUBF
	call	PUSHF1		;end angle - start angle

	ld	hl,(PROGAD)
	call	GETCMER
	call	NARGMO		;step angle
	call	CHKSGN
	jp	z,FCERR		;=0
	jp	c,FCERR		;<0
	call	CPYFAC
	call	POPF1		;end angle - start angle
	call	PUSHF2		;step angle
	call	DIVF		;(end angle - start angle) / step angle
	call	FTOI2
	ld	a,d
	rlca
	jp	c,FCERR		;<0
	inc	de
	call	POPF2		;step angle
	call	POPF1		;start angle
	call	PUSHF2		;step angle
	call	PUSHF1		;start angle

	ld	a,'L'
	call	PRINTER
	ld	a,'0'		;solid line
	call	PRINTER

	ld	a,'M'
	ld	(CRCLCMD+1),a

CRCLLP:
	ld	(CRCLCNT+1),de	;loop count
	call	RAD		;degree->radian
	call	COS
	call	MULR
CRCLX:
	ld	hl,0000h	;x
	add	hl,de		;x+r*cos
	ex	de,hl
	call	CHKPX
	call	POPF1		;present angle
	call	PUSHF1		;present angle
	push	de		;x+r*cos
	call	RAD		;degree->radian
	call	SIN
	call	MULR
CRCLY:
	ld	hl,0000h	;y
	add	hl,de
	ex	de,hl
	call	CHKPY
	ex	de,hl
	pop	de		;x+r*cos

CRCLCMD:
	ld	a,'M'
	call	PRINTER
	call	PUTPRNXY
	call	PUTPRNCR

	ld	a,'D'
	ld	(CRCLCMD+1),a

	call	POPF1		;present angle
	call	POPF2		;step angle
	call	PUSHF2		;step angle
	call	ADDF		;present angle+step angle
	call	PUSHF1		;new angle

CRCLCNT:
	ld	de,0001h	;loop count
	dec	de
	ld	a,d
	or	e
	jp	nz,CRCLLP
	call	POPF1		;present angle
	call	POPF2		;step angle
	ret


;FAC1=FAC1*radius for CIRCLE command
;input: FAC1
;destroy: af,bc,de,hl,FAC2
MULR:
	call	CPYFAC
CRCLR:
	ld	hl,0000h
	call	I2TOF
	call	MULF
	jp	FTOI2


;TEST command
C_TEST:
	call	CHKTMODE
	ld	a,04h
	jp	PRINTER


;PAGE command
C_PAGE:
	call	CHKTMODE
	call	INT1ARG
	dec	a
	cp	72
	jp	nc,FCERR
	ld	a,09h
	call	PRINTER
	ld	a,09h
	call	PRINTER
	ld	a,'0'
	call	PRINTER
	ld	a,e
	call	DIVI10
	call	PRINTER
	ld	a,c
	jp	PRINTER	


;MODE command
C_MODE:
	call	SKIPSP
	inc	hl
	ld	b,(hl)
	inc	hl
	ld	(PROGAD),hl
	ld	hl,PRNMOD
	cp	'G'
	jr	z,MODEGR
	cp	'T'
	jp	nz,SNERR

;change to text mode
	ld	a,b
	ld	b,00h
	cp	'S'		;TS: 80characters/line=width 0
	jr	z,MODET
	inc	b
	cp	'N'		;TN: 40characters/line=width 1
	jr	z,MODET
	inc	b
	cp	'L'		;TL: 26characters/line=width 2
	jp	nz,SNERR
MODET:
	ld	a,(hl)		;(PRNMOD)
	or	a
	jr	z,MODETZ	;text mode
	ld	(hl),00h	;(PRNMOD)
	ld	a,'A'
	call	PRINTER
MODETZ:
	dec	b
	jr	z,MODETN
	inc	b
	jr	z,MODETS

;width 2
MODETL:
	call	PRNW0TO1	;09090b:0->1
	ld	a,0bh		;0b:1->2
	jp	PRINTER

;width 1
MODETN:
	ld	a,0ch		;0c:2->1
	call	PRINTER
PRNW0TO1:
	ld	a,09h		;09090b:0->1
	call	PRINTER
	ld	a,09h
	call	PRINTER
	ld	a,0bh
	jp	PRINTER

;width 0
MODETS:
	ld	a,0ch		;0c:2->1
	call	PRINTER
	ld	a,09h		;090909:1->0
	call	PRINTER
	ld	a,09h
	call	PRINTER
	ld	a,09h
	jp	PRINTER

MODEGR:
	ld	a,b
	cp	'R'
	jp	nz,SNERR
	ld	a,(PLOTON+1)
	or	a
	jp	nz,BPERR	;if plot on
	ld	a,(hl)		;(PRNMOD)
	or	a
	ret	nz		;graphic mode
	inc	a		;=1
	ld	(hl),a		;(PRNMOD)
;	ld	a,01h
	inc	a		;02: text->graphic
	jp	PRINTER


;PLOT command
C_PLOT:
	call	CHKTMODE
	call	SKIPSP
	cp	ON_
	jr	z,PLOTZ
	sub	OFF_
	jp	nz,SNERR
PLOTZ:
	ld	(PLOTON+1),a	;=0 if PLOT OFF
	inc	hl
	ld	(PROGAD),hl
	ret


;POKE command
C_POKE:
	call	SKIPSP
	cp	'@'
	push	af		;z-flag
	jr	nz,POKEVRAM	;poke
	inc	hl		;poke@
POKEVRAM:
	call	INTARG
	ld	a,(hl)
	cp	','
	jp	nz,SNERR
POKELP:
	push	de
	call	INT1INC
	pop	hl
	pop	af		;z-flag
	push	af		;z-flag
	jr	z,POKEAT

;poke (VRAM)
	call	CHGVRAM
	ld	(hl),e
	call	CHGDRAM
	jr	POKEEND

;poke@ (DRAM)
POKEAT:
	ld	(hl),e

POKEEND:
	inc	hl
	ex	de,hl
	ld	hl,(PROGAD)
	call	SKIPSP
	cp	','
	jr	z,POKELP
	pop	af
	ret


;LOCATE command
;CURSOR command
C_LOCA:
C_CRSR:
	call	GETXY
	ld	l,c
	ld	h,e
	jp	SETCSR


;no parameters
CNSLNOPAR:
	ld	bc,256*(WIDTH-1)
	ld	a,(CNSL1SCR)
	or	a
	jp	z,CNSLENDX

	push	bc
	xor	a
	ld	(CNSL1SCR+1),a

	call	CHGVRAM
	ld	hl,0d000h+WIDTH*HEIGHT
	ld	de,0d000h+WIDTH*HEIGHT+1
	ld	bc,WIDTH*HEIGHT-1
	ld	(hl),00h
	ldir
	ld	hl,0d800h+WIDTH*HEIGHT
	ld	de,0d800h+WIDTH*HEIGHT+1
	ld	bc,WIDTH*HEIGHT-1
	ld	a,(ATTR)
	ld	(hl),a
	ldir
	call	CHGDRAM

	ld	h,HEIGHT
	ld	a,0ffh
	ld	b,HEIGHT
CNSLNOPARLP:
	call	SETLINE
	inc	h
	djnz	CNSLNOPARLP
	pop	bc
	jp	CNSLENDX


;CONSOLE command
C_CNSL:
	ld	bc,256*(HEIGHT-1)
	push	bc

	call	SKIPSP
	cp	','
	jr	z,CNSLZ
	call	NUMARG
	jr	z,CNSLNOPAR	;not numerical parameter
CNSLZ:
	ld	a,01h
	ld	(CNSL1SCR+1),a	;>0
	
	pop	bc

	push	af		;z-flag(1=comma)
	ld	a,(YMIN)
	ld	c,a
	ld	a,(YMAX)
	ld	b,a
	pop	af		;z-flag(1=comma)
	push	bc		;c=(YMIN),b=(YMAX)

	jr	z,CNSLPAR2	;comma

	push	hl
	call	FTOI1
	pop	hl

	ld	a,HEIGHT-1
	cp	e
	jr	c,CNSLC
	ld	a,e
CNSLC:
	pop	bc
	ld	c,a
	push	bc

	ld	a,(hl)
	cp	','
	jr	nz,CNSLEND

CNSLPAR2:
	call	SKIPSPINC
	cp	','
	jr	z,CNSLPAR3
	call	INT1ARG
	pop	bc
	add	a,c
	jp	c,FCERR
	dec	a
	cp	HEIGHT		;temporary
	jp	nc,FCERR
	ld	b,a
	push	bc
	ld	a,(hl)
	cp	','
	jr	nz,CNSLEND

CNSLPAR3:
	ld	a,(XMIN)
	ld	c,a
	ld	a,(XMAX)
	ld	b,a

	push	bc		;c=(XMIN),b=(XMAX)
	call	SKIPSPINC
	cp	','
	jr	z,CNSLPAR4
	call	INT1ARG
	ld	a,WIDTH-1
	cp	e
	jr	c,CNSLC2
	ld	a,e
CNSLC2:
	pop	bc
	ld	c,a

	ld	a,(hl)
	cp	','
	jr	nz,CNSLENDX
	push	bc

CNSLPAR4:
	inc	hl
	call	INT1ARG
	pop	bc
	add	a,c
	jp	c,FCERR
	dec	a
	cp	WIDTH
	jp	nc,FCERR
	ld	b,a

CNSLENDX:
	ld	a,c
	ld	(XMIN),a
	ld	a,b
	ld	(XMAX),a
CNSLEND:
	pop	bc
	ld	a,c
	ld	(YMIN),a
	ld	a,b
	ld	(YMAX),a

	ld	a,(XMIN)
	call	CUTLINE2

	jp	CTLHOM


;key llist
KLLIST:
	call	SETPRN
;	jp	C_KLST

;KLIST command
C_KLST:
	ld	b,0ah
KLSTLP1:
	ld	hl,KLST_STR
	call	PUTS
	ld	a,0bh
	sub	b
	ld	c,a
	push	bc
	ld	l,c
	ld	h,00h
	call	PUTI2
	ld	hl,KLST_STR2
	call	PUTS
	pop	bc
	ld	a,c
	add	a,a		;*2
	add	a,a		;*4
	add	a,a		;*8
	add	a,a		;*16
	ld	e,a
	ld	d,00h
	ld	hl,FKEYTBL
	add	hl,de
	ld	a,(hl)
	or	a
	jr	z,KLSTZ1

KLSTLP2:
	cp	20h
	jr	c,KLSTC
KLSTNC:
	ld	a,22h
	call	PUTC
KLSTLP3:
	ld	a,(hl)
	call	PUTC
	inc	hl
	ld	a,(hl)
	or	a
	jr	z,KLSTZ2
	cp	20h
	jr	nc,KLSTLP3
	ld	a,22h
	call	PUTC
	ld	a,'+'
	call	PUTC
	ld	a,(hl)

KLSTC:
	push	hl
	ld	hl,CHR_STR
	call	PUTS
	ld	h,00h
	ld	l,a
	push	bc
	call	PUTI2
	pop	bc
	ld	a,')'
	call	PUTC
	pop	hl
	inc	hl
	ld	a,(hl)
	or	a
	jr	z,KLSTZ3
	ld	a,'+'
	call	PUTC
	ld	a,(hl)
	jr	KLSTLP2

KLSTZ1:
	ld	a,22h
	call	PUTC
KLSTZ2:
	ld	a,22h
	call	PUTC
KLSTZ3:
	call	PUTNL
	djnz	KLSTLP1
	xor	a
	ld	(SELPRN),a
	ret

KLST_STR:
	db	"KEY", 00h

KLST_STR2:
	db	", ", 00h

CHR_STR:
	db	"CHR$(", 00h


;KEY command
C_KEY:
	call	SKIPSP
	inc	hl
	ld	(PROGAD),hl
	cp	LIST_
	jp	z,C_KLST
	cp	LLIST_
	jp	z,KLLIST
	dec	hl
	call	INT1ARG
	cp	0bh
	jp	nc,FCERR
	push	af		;key n
	ex	de,hl
	add	a,a		;*2
	add	a,a		;*4
	add	a,a		;*8
	add	a,a		;*16
	ld	c,a
	ld	b,00h
	ld	hl,FKEYTBL
	add	hl,bc
	push	hl
	ex	de,hl

	ld	a,(hl)
	cp	','
	jp	nz,SNERR
	inc	hl
	call	STRARG		;a,de
	ld	c,a

	pop	hl
	ld	b,0eh
KEYLP1:
	ld	a,c
	or	a
	jr	z,KEYEND
	ld	a,(de)
	ld	(hl),a
	inc	de
	inc	hl
	dec	c
	djnz	KEYLP1
KEYEND:
	inc	b
	inc	b
	xor	a
KEYLP2:
	ld	(hl),a
	inc	hl
	djnz	KEYLP2

	pop	af		;key n
	or	a
	ret	nz
;key 0
	inc	a		;=1
	ld	(CHKFKY),a
	ld	hl,FKEYTBL
	ld	(FKYBUF+1),hl
	ret


;COLOR command
C_COLR:
	call	SKIPSP
	cp	','
	jr	z,COLRPAR2
	call	INT1ARG
	cp	07h+1
	jp	nc,FCERR
	rlca
	rlca
	rlca
	rlca
	ld	e,a
	ld	a,(ATTR)
	and	8fh
	or	e
	ld	(ATTR),a
	ld	a,(hl)
	cp	','
	ret	nz

COLRPAR2:
	inc	hl
	call	INT1ARG
	cp	07h+1
	jp	nc,FCERR
	ld	a,(ATTR)
	and	0f8h
	or	e
	ld	(ATTR),a
	ret


;CLS command
C_CLS:
	jp	CLSMAIN


;LINE command
C_LINE:
	call	SKIPSP
	cp	INPUT_
	inc	hl
	jp	z,C_LINP
	dec	hl
	ld	a,'D'
	jp	RLIN2


;CGEN command
C_CGEN:
	call	NUMARG
	ld	e,00h
	call	nz,FTOI1	;if with numerical parameter
	dec	e
	jp	z,CTLCS1	;CGEN 1
	inc	e
	jp	z,CTLCS0	;CGEN 0/CGEN
	jp	FCERR


;PLAY command
;MUSIC command
;TEMPO command
C_PLAY:
C_MSIC:
C_TEMP:
	call	ARG
	jp	z,MOERR

	ld	a,(FAC1)
	cp	03h
	jr	z,PLAYSTR
;tempo
	call	FTOI1
	ld	a,e
	dec	a
	cp	7
	jp	nc,FCERR	;1-7 ok
	ld	a,08h
	sub	e
	ld	(TEMPW),a

PLAYNEXT:
	ld	hl,(PROGAD)
	call	SKIPSP
	inc	hl
	cp	','
	jr	z,C_PLAY
	cp	';'
	jr	z,C_PLAY
	ret

PLAYSTR:
	call	STRARG2
	call	PLAYMAIN
	jr	PLAYNEXT


;BEEP command
C_BEEP:
	call	NUMARG
	jr	nz,BEEPNUM	;with numerical parameter

;destroy: af,bc,hl
BELL:
	call	BEEP1
	ld	bc,0800h
	call	WAIT
	jp	BEEPOFF

BEEPNUM:
	call	FTOI2
	ld	a,d
	or	e
	jp	z,BEEPOFF	;BEEP 0

BEEP1:
;	ld	hl,06aeh	;(4.5MHz/286*910/4)/4/523.25Hz=06ae (O5C)
	ld	hl,03f9h	;(4.5MHz/286*910/4)/4/880Hz=03f9h (O5A)
	ld	(TONEVAL),hl
	jp	BEEPON


;BOOT command
C_BOOT:
	call	CHKCLN
	jp	nz,SNERR

	ld	hl,PRNMOD
	ld	a,(hl)
	ld	(hl),00h	;text mode
	or	a
	ld	a,'A'
	call	nz,PRINTER

	call	CHGVRAM
	di
	out	(0e2h),a	;0000-0fff:ROM
	ld	hl,0000h
	ld	sp,hl
	jp	(hl)


;CLICK command
C_CLCK:
	call	SKIPSP
	sub	OFF_
	jr	z,CLCKZ
	sub	ON_-OFF_
	jp	nz,SNERR
	inc	a
CLCKZ:
	inc	hl
	ld	(PROGAD),hl
	ld	(CLICKON),a	;off=0, on=1
	ret


;PSET command
C_PSET:
	call	PSETPAR
	ld	b,e		;y
	push	bc		;;c=x,b=y
	call	SKIPSP
	cp	','
	ld	a,(ATTR)
	jr	nz,PSETNZ
	call	INT1INC
	cp	07h+1
	jp	nc,FCERR
	rlca
	rlca
	rlca
	rlca
	ld	b,a
	ld	a,(ATTR)
	and	07h
	or	b
PSETNZ:
	ld	d,a		;attribute
	call	CHKRPAR
	pop	hl		;;l=x,h=y

	call	DOTPAT
	ld	b,a
	call	XY2AD

	call	CHGVRAM
	ld	a,(hl)
	cp	0f0h
	jr	nc,PSETNC
	ld	a,0f0h
PSETNC:
	or	b
	ld	(hl),a
	set	3,h		;attribute
	ld	(hl),d
	call	CHGDRAM
	ret


;PRESET command
C_PRST:
	call	PSETPAR
	call	CHKRPAR
	ld	l,c
	ld	h,e
	call	DOTPAT
	cpl
	ld	b,a
	call	XY2AD
	call	CHGVRAM
	ld	a,(hl)
	cp	0f0h
	jr	c,PRSTC
;	ld	a,0f0h
	and	b
	ld	(hl),a
PRSTC:
	call	CHGDRAM
	ret


;get "(X,Y" parameters for PSET/PRESET
;input: hl=program address
;output: bc=x, de=y, hl=next address
;destroy: af
PSETPAR:
	call	CHKLPAR
	call	INT1ARG2
	ld	a,c		;x
	cp	WIDTH*2
	jp	nc,FCERR
	ld	a,e		;y
	cp	HEIGHT*2
	ret	c
	jp	FCERR

;get dot pattern and convert xy coordinates
;input: l=x(0-79), h=y(0-49)
;output: l=x(0-39), h=y(0-24), a=dot pattern(1,2,4,8)
;destroy: af,b
DOTPAT:
	xor	a
	srl	h
	rla
	srl	l
	rla
	inc	a		;1-4
	ld	b,a
	ld	a,80h
PSETLP:
	rlca
	djnz	PSETLP
	ret


;read information block from CMT
;output: a(0=ok, 1=checksum error, 2=shift+break), c-flag(1=error)
;destroy: af,bc,hl
RDI:
;for mz700win: ed f0 01=MON_rdinf
	or	80h		;for EmuZ-700:ed f0 is intepreted as nop+ret p
	db	0edh, 0f0h, 01h, 00h, 00h	;nop ld bc,0000h

	push	de
	ld	d,00h		;read
	call	MTON
	di

	call	CHGVRAM
	ld	de,2828h	;1*40+0*40
	call	TMARK
	jr	c,RDIEND	;shift+break

	ld	hl,IBUFE
	ld	de,0080h
	xor	a		;not verify
	call	RDMAIN
RDIEND:
	push	af		;
	pop	hl		;
	call	CHGDRAM
	ei
	push	hl		;
	call	MTOFF
	pop	af		;
	pop	de
	ret


;verify
;destroy: af,bc,hl
;output: a(0=ok, 1=error, 2=shift+break), c-flag(1=error)
VRFY:
;for mz700win: ed f0 05=MON_verfy (set c-flag=0,b=0)
	or	80h		;for EmuZ-700:ed f0 is intepreted as nop+ret p
	scf
	db	0edh, 0f0h, 05h	;nop dec b
	ret	nc

	ld	h,01h		;verify?
	jr	RDDMTON

;read data block from CMT
;output: a(0=ok, 1=checksum error, 2=shift+break), c-flag(1=error)
;destroy: af,bc,hl
RDD:
	ld	h,00h		;verify?

;for mz700win: ed f0 02=MON_rddata
	or	80h		;for EmuZ-700:ed f0 is intepreted as nop+ret p
	ld	bc,TMP
	db	0edh, 0f0h, 02h	;nop ld (bc),a

;input: h(0=load,1=verify)
RDDMTON:
	push	de
	push	hl
	ld	d,00h		;read
	call	MTON
	pop	hl
	di

	call	CHGVRAM
	ld	de,1414h	;1*20+0*20
	call	TMARK
	jr	c,RDIEND	;shift+break

	ld	a,h		;verify?
	ld	hl,(DTADR)
	ld	de,(SIZE)
	call	RDMAIN
	jr	RDIEND


;read CMT data
;input: hl=address, de=size, a(0=load, not0=verify)
;output: a(0=no error, 1=check sum error, 2=shift+break), c-flag(1=error)
;destroy: af,bc,de,hl
RDMAIN:
	ld	(RDVRFY+1),a
	ld	b,02h
RDMAINLP1:
	push	hl
	push	de
	push	bc

RDMAINLP2:
	call	GETCMTBIT
	jr	z,RDBRK		;shift+brak
;	jr	nc,RDMAINLP2	;wait for bit=1

	ld	bc,0000h
	ld	(CHKSUMR+1),bc
RDMAINLP3:
	call	GETCMT
	jr	z,RDBRK		;shift+break
	out	(0e1h),a	;d000-ffff:DRAM
RDVRFY:
	ld	c,00h
	inc	c
	dec	c
	jr	z,RDMAINZ
	cp	(hl)
	jr	nz,RDBAD
RDMAINZ:
	ld	(hl),a
	out	(0e3h),a	;d000-ffff:VRAM/IO
	inc	hl
	dec	de
	ld	a,d
	or	e
	jr	nz,RDMAINLP3

CHKSUMR:
	ld	de,0000h
	call	GETCMT		;check sum high
	jr	z,RDBRK		;shift+break
	ld	h,a
	call	GETCMT		;check sum low
	jr	z,RDBRK
	ld	l,a
	or	a
	sbc	hl,de
	jr	nz,RDERR
	pop	bc
	pop	de
	pop	hl
	xor	a
	ret

;verify error
RDBAD:
	out	(0e3h),a	;d000-ffff:VRAM/IO
RDERR:
	pop	bc
	pop	de
	pop	hl
	djnz	RDMAINLP1
	ld	a,01h
	scf
	ret

RDBRK:
	pop	bc
	pop	de
	pop	hl
	ld	a,02h
	scf
	ret


;wait for tape mark
;input: d='1' counts, e='0' counts
;output: a=2 if shift+break, c-flag(1=shift+break)
;destroy: af,b
TMARK:
	ld	b,d
TMARKLP1:
	call	GETCMTBIT
	jr	z,TMARKBRK	;shift+break
	jr	nc,TMARK	;wait for bit=1
	djnz	TMARKLP1

	ld	b,e
TMARKLP2:
	call	GETCMTBIT
	jr	z,TMARKBRK	;shift+break
	jr	c,TMARK		;wait for bit=0
	djnz	TMARKLP2
	ret			;c-flag=0

TMARKBRK:
	ld	a,02h
	scf
	ret


;get 1 character from CMT (d000-ffff:VRAM/IO)
;output: a=data, z-flag(1=shift+break)
;destroy: af,bc
GETCMT:
	call	GETCMTBIT
	ret	z
	jr	nc,GETCMT	;wait for start bit(1)

	ld	bc,0800h
GETCMTLP:
	call	GETCMTBIT
	ret	z		;shift+break
	jr	nc,GETCMTNC
	push	hl
	ld	hl,(CHKSUMR+1)
	inc	hl
	ld	(CHKSUMR+1),hl
	pop	hl
GETCMTNC:
	ld	a,c
	rla
	ld	c,a
	djnz	GETCMTLP
	ret			;z-flag=0


;get CMT bit data (d000-ffff:VRAM/IO)
;output: c-flag(data), z-flag(1=shift+break)
;destroy: af
GETCMTBIT:
	call	EDGE
	ret	z
	push	bc
	ld	bc,0013h
	call	WAIT
	inc	b		;reset z-flag
	pop	bc
	ld	a,(0e002h)	;bit5=RDATA
	rlca
	rlca
	rlca
	ret


;wait for RDATA 0->1 edge (d000-ffff:VRAM/IO)
;output: z-flag(1=shift+break)
;destroy: af
EDGE:
	call	CHKBRK2
	ret	z
	ld	a,(0e002h)	;bit5=RDATA
	and	20h
	jr	nz,EDGE
EDGELP:
	call	CHKBRK2
	ret	z
	ld	a,(0e002h)	;bit5=RDATA
	and	20h
	jr	z,EDGELP
	ret			;z-flag=0


;write infromation block to CMT
;output: c-flag(1=shift+break)
;destroy: af,bc,hl
WRI:
;for mz700win: ed f0 03=MON_wrinf
	or	80h		;for EmuZ-700:ed f0 is intepreted as nop+ret p
	db	0edh, 0f0h, 03h	;nop inc bc

	push	de
	ld	d,01h		;write
	call	MTON
	di

	call	CHGVRAM
	ld	bc,22000
WRILP1:
	call	PWM0
	dec	bc
	ld	a,b
	or	c
	jr	nz,WRILP1

	ld	bc,2828h
	call	PUTTMARK
	jp	c,RDIEND

	ld	hl,IBUFE
	ld	de,0080h
	call	WRMAIN
	jp	RDIEND


;write data block to CMT
;output: c-flag(1=shift+break)
;destroy: af,bc,hl
WRD:
;for mz700win: ed f0 04=MON_wrdata
	or	80h		;for EmuZ-700:ed f0 is intepreted as nop+ret p
	db	0edh, 0f0h, 04h	;nop inc b

	push	de
	ld	d,01h		;write
	call	MTON
	di

	call	CHGVRAM
	ld	bc,11000
WRDLP1:
	call	PWM0
	dec	bc
	ld	a,b
	or	c
	jr	nz,WRDLP1

	ld	bc,1414h
	call	PUTTMARK
	jp	c,RDIEND

	ld	hl,(DTADR)
	ld	de,(SIZE)
	call	WRMAIN
	jp	RDIEND


;write data to CMT
;input: hl=address, de=size, a(0=load, not0=verify)
;output: c-flag(1=error)
;destroy: af,bc,de,hl
WRMAIN:
	ld	b,02h
WRMAINLP1:
	push	hl
	push	de
	push	bc

	call	PWM1
	ld	bc,0000h
	ld	(CHKSUMW+1),bc
WRMAINLP2:
	out	(0e1h),a	;d000-ffff:DRAM
	ld	a,(hl)
	inc	hl
	out	(0e3h),a	;d000-ffff:VRAM/IO
	call	PUTCMT
	jr	z,WRBRK
	dec	de
	ld	a,d
	or	e
	jr	nz,WRMAINLP2

CHKSUMW:
	ld	hl,0000h
	ld	a,h
	call	PUTCMT
	jr	z,WRBRK
	ld	a,l
	call	PUTCMT
	jr	z,WRBRK
	call	PWM1

	ld	b,0100h
WRMAINLP3:
	call	PWM0
	djnz	WRMAINLP3

	pop	bc
	pop	de
	pop	hl
	djnz	WRMAINLP1
	or	a		;reset c-flag
	ret

WRBRK:
	scf
	pop	bc
	pop	de
	pop	hl
	ret


;put tape mark
;input: b='1' counts, c='0' counts
;output: c-flag(1=shift+break)
;destroy: af,b,d
PUTTMARK:
	call	PWM1
	djnz	PUTTMARK
	ld	b,c
PUTTMARKLP:
	call	PWM0
	djnz	PUTTMARKLP
	ret


;put 1 character to CMT (d000-ffff:VRAM/IO)
;input: a=data
;output: z-flag(1=shift+break)
;destroy: af,bc
PUTCMT:
	ld	c,a
	call	CHKBRK2
	ret	z
	call	PWM1		;start bit
	ld	b,08h
PUTCMTLP:
	rlc	c
	jr	nc,PUTCMTNC
	call	PWM1
	push	hl
	ld	hl,(CHKSUMW+1)
	inc	hl
	ld	(CHKSUMW+1),hl
	pop	hl
	jr	PUTCMTEND
PUTCMTNC:
	call	PWM0
PUTCMTEND:
	djnz	PUTCMTLP
	inc	b		;reset z-flag
	ret


;put a pulse to CMT (bit=0)
;destory: af
PWM0:
	push	de
	ld	d,0ch
	jr	PWM

;put a pulse to CMT (bit=1)
;destory: af
PWM1:
	push	de
	ld	d,19h
PWM:
	push	bc
	ld	b,00h
	ld	c,d
	ld	a,03h
	ld	(0e003h),a	;WDATA on
	call	WAIT
	ld	a,02h
	ld	(0e003h),a	;WDATA off
;	ld	b,00h
	ld	c,d
	call	WAIT
	pop	bc
	pop	de
	ret


;motor on
;input: d(0=read, not0=write)
;destroy: af,de,hl
MTON:
	ld	e,04h
	call	CHGVRAM
MTONLP:
	ld	a,(0e002h)
	and	10h
	jr	nz,MTEND	;if on
	call	MTPULSE
	push	bc
	ld	bc,0100h
	call	WAIT
	pop	bc
	dec	e
	jr	nz,MTONLP

	call	CHGDRAM
	ld	hl,PLAY_STR
	ld	a,d
	or	a
	jr	z,MTONZ
	ld	hl,REC_STR
MTONZ:
	jp	PUTSNL

PLAY_STR:
	db	7fh, " PLAY", 00h
REC_STR:
	db	7fh, " RECORD,PLAY", 00h


;motor off
;destroy: af
MTOFF:
	call	CHGVRAM
	ld	a,(0e002h)
	and	10h
	call	nz,MTPULSE
MTEND:
	call	CHGDRAM
	ret

;motor on/off pulse (d000-ffff:VRAM/IO)
MTPULSE:
	ld	a,06h
	ld	(0e003h),a
	inc	a
	ld	(0e003h),a	;0->1 edge
	dec	a
	ld	(0e003h),a
	ret


;open file#0
;input: b=file attribute, d=file open mode(1=read,2=write)
;destroy: af,hl
FOPEN0:
	ld	hl,(FBUFAD)
	ld	(BUFP),hl
;	jr	FOPEN

;input: b=file attribute, d=file open mode(1=read,2=write), hl=buffer pointer
;destroy: af
FOPEN:
	ld	(hl),d		;+0: file open mode
	inc	hl
	ld	a,d
	sub	02h
	ld	(hl),a		;+1: to next block? (0 if write)
	inc	hl
	ld	(hl),00h	;+2: position
	ld	a,(IBUFE)
	cp	04h		;file attribute=4: ASCII
	ret	z
	jp	BMERR


;close file#0
;destroy: af,de,hl
FCLOSE0:
	ld	e,00h
;	jr	FCLOSE

;close file
;input: e=file number
;destroy: af,de,hl
FCLOSE:
	call	GETBUFP
	ret	z

;input: hl=buffer pointer
FCLOSE2:
	ld	a,(hl)		;+0: file open mode
	dec	a
	jr	z,FCLOSEEND	;1=read
;write
	push	hl
	inc	hl
	ld	a,(hl)		;+1: to next block?
	or	a		;
	inc	hl
	ld	a,(hl)		;+2: pos
	ld	de,000eh
	add	hl,de
	jr	nz,FCLOSENZ	;if to next block

	ld	e,a		;pos
	add	hl,de		;+16+pos
	ld	(hl),1ah
;	or	a
	sbc	hl,de		;+16

FCLOSENZ:
	dec	hl
	ld	(hl),0ffh	;+15: record number(high)
	dec	hl
	ld	(hl),0ffh	;+14: record number(low)
	ld	(DTADR),hl
	ld	hl,0102h
	ld	(SIZE),hl
	call	WRD
	pop	hl

FCLOSEEND:
	ld	(hl),00h	;+0: file open mode
	inc	hl
	ld	(hl),00h	;+1: to next block?
	ret


;input: e=file number
;output: a=file open mode, hl=buffer pointer, z-flag(0=open, 1=close)
;destroy: f,de
GETBUFP:
	ld	a,(FILES)
	cp	e
	jp	c,FCERR
	ld	hl,(FBUFAD)
	ld	d,e
	ld	a,e
	add	a,a
	add	a,a
	add	a,a
	add	a,a
	ld	e,a
	add	hl,de		;(FBUFAD)+0110h*e
	ld	a,(hl)		;+0: file open mode
	or	a
	ret


REMOVE:
	ld	hl,(BUFP)
	ld	a,h
	or	l
	ret	z
	ld	a,(hl)		;+0: file open mode
	dec	a
	call	nz,CLOSALL	;for ASCII save

;destroy: hl
CLRBUFP:
	ld	hl,0000h
	ld	(BUFP),hl
	ret


;DEVI$ command
C_DEVI:
	push	hl
	call	FCLOSE0
	pop	hl
	call	STRARG		;device name
	push	hl
	call	GETDEVNUM
	cp	DEVCAS		;CAS:
	jp	nz,FDERR
	pop	hl
	call	GETCMER
	call	NARGMO
	push	hl
	call	FTOI
	pop	hl
	push	de		;record number
	call	GETCMER
	call	CHKGETV
	ld	a,c
	cp	03h
	jp	nz,TMERR
	pop	bc		;record number
	push	de		;1st variable value address
	push	bc		;record number
	ld	hl,(PROGAD)
	call	GETCMER
	call	CHKGETV
	ld	a,c
	cp	03h
	jp	nz,TMERR
	pop	bc		;record number
	pop	hl		;1st variable value address
	push	de		;2nd variable value address
	push	hl		;1st variable value address

DEVILP:
	ld	hl,0102h
	ld	(SIZE),hl
	ld	hl,(FBUFAD)	;file buffer #0
	ld	de,000eh	;+14: record number
	add	hl,de
	ld	(DTADR),hl
	push	bc		;record number
	push	hl
	call	RDD
	pop	hl
	pop	bc		;record number
	dec	a
	jp	z,TRERR
	dec	a
	jp	z,STOPSP
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
	ex	de,hl
	sbc	hl,bc		;compare record number
	jr	nz,DEVILP

	ld	a,80h
	pop	hl		;1st variable value address
	call	MKSTRVAR

	ld	hl,(FBUFAD)	;file buffer #0
	ld	bc,0090h
	add	hl,bc
	ex	de,hl
	ld	a,80h
	pop	hl		;2nd variable value address 
	jp	MKSTRVAR


;DEVO$ command
C_DEVO:
	push	hl
	call	FCLOSE0
	ld	hl,(FBUFAD)	;file buffer #0
	ld	d,h
	ld	e,l
	inc	de
	ld	bc,010fh
	ld	(hl),00h
	ldir
	pop	hl

;device name
	call	STRARG
	push	hl
	call	GETDEVNUM
	cp	DEVCAS		;CAS:
	jp	nz,FDERR
	pop	hl
	call	GETCMER

;record number
	call	NARGMO
	push	hl
	call	FTOI
	ld	hl,(FBUFAD)
	ld	bc,000eh
	add	hl,bc
	ld	(DTADR),hl
	ld	(hl),e		;+14: record number(low)
	inc	hl
	ld	(hl),d		;+15: record number(low)
	pop	hl
	call	GETCMER

;1st string
	call	STRARG
	or	a		;length
	jr	z,DEVOZ1
	cp	81h
	jr	c,DEVOC1
	ld	a,80h
DEVOC1:
	push	hl
	ld	hl,(FBUFAD)
	ld	bc,0010h
	add	hl,bc
	ex	de,hl
	ld	b,00h
	ld	c,a
	ldir
	pop	hl
DEVOZ1:
	call	GETCMER

;2nd string
	call	STRARG
	or	a		;length
	jr	z,DEVOZ2
	cp	81h
	jr	c,DEVOC2
	ld	a,80h
DEVOC2:
	ld	hl,(FBUFAD)
	ld	bc,0090h
	add	hl,bc
	ex	de,hl
	ld	b,00h
	ld	c,a
	ldir
DEVOZ2:
	ld	hl,0102h
	ld	(SIZE),hl
	call	WRD
	jp	c,STOPSP
	ret


;DEVICE command
C_DEV:
	call	STRARG
	call	GETDEVNUM
	ld	(DEV_DEF),a
	ret


;CHAIN command
C_CHAI:
;FIELD command
C_FLD:
;SET command
C_SET:
;KILL command
C_KILL:
	jp	SNERR


;SEARCH command
C_SRCH:
	call	STRARG

	ld	hl,(BASICAD)
SRCHLP1:
	push	de		;string address
	push	af		;length
	push	hl		;program address

	ld	b,04h		;file attribute=4: ASCII
	ld	hl,IBUFE
	ld	(hl),b
	ld	d,02h		;file open mode=2: output
	call	FOPEN0

	pop	hl		;program address
	ld	a,(hl)
	inc	hl
	or	(hl)
	jr	z,SRCHEND
	inc	hl
	call	LIST1
	xor	a
	call	PUTC

	pop	bc		;b=length
	pop	de		;string address

	push	hl		;line end address (00)

	ld	hl,(BUFP)
	push	de
	ld	de,0010h
	add	hl,de
	pop	de
SRCHLP2:
	ld	a,(hl)
	inc	hl
	cp	' '
	jr	nz,SRCHLP2
SRCHLP3:
	push	hl		;list address
	push	de		;string address
	push	bc		;b=length
	ld	a,b
	or	a
	jr	z,SRCHFND
SRCHLP4:	
	ld	a,(hl)
	or	a
	jr	z,SRCHZ		;line end
	ld	a,(de)
	cp	(hl)
	jr	nz,SRCHNZ
	inc	de
	inc	hl
	djnz	SRCHLP4
;found
SRCHFND:
	ld	hl,(BUFP)
	ld	de,0000h
	ld	(hl),d		;+0: file open mode
	ld	(BUFP),de
	ld	e,10h
	add	hl,de
	call	PUTSNL
SRCHZ:
	pop	af		;length
	pop	de		;string address
	pop	hl		;list address
	pop	hl		;line end address(00)
	inc	hl
	jr	SRCHLP1

SRCHNZ:
	pop	bc		;length
	pop	de		;string address
	pop	hl		;list address
	inc	hl
	jr	SRCHLP3

;end
SRCHEND:
	pop	bc		;b=length
	pop	de		;string address
	xor	a
	ld	hl,(BUFP)
	ld	(hl),a		;+0: file open mode
	jp	CLRBUFP


;LLIST command
C_LLST:
	call	SETPRN
;	jp	C_LIST


;LIST command
C_LIST:
	ld	a,h		;>0
	ld	(PRTSYM+1),a
	pop	af		;cancel return address
	call	PUTNLX

;with line number?
	call	SKIPSP
	cp	0bh
	jr	z,LISTRANGE

;with file descriptor? (device + file name)
	push	hl
	call	ARG
	pop	de
	ex	de,hl
	ld	a,(FAC1)
	cp	03h
	jr	nz,LISTRANGE

;check file descriptor
	ex	de,hl
	call	STRARG2

	push	hl
	push	de
	push	af		;length
	call	CHKDEV
	pop	bc		;b=length
	pop	de
	or	a
	jr	nz,LISTNZ
	ld	a,(DEV_DEF)
LISTNZ:
	dec	a
	jr	z,LISTCRT	;1=CRT:
	dec	a
	jr	z,LISTSCR	;2=SCR:
	dec	a
	jr	z,LISTCAS	;3=CAS:
	dec	a
	jr	z,LISTPLT	;4=PLT:
	dec	a
	jp	nz,FDERR	;5=LPT:, 6=KEY:

;"LPT:"
LISTLPT:
;	xor	a
	ld	(PRTSYM+1),a
;"PLT:"
LISTPLT:
	call	SETPRN
	jr	LISTAFTFD

;tape
LISTCAS:
	ld	a,b		;length
	call	SETFNAME
	call	ASCOPEN
;	jr	LISTSCR		;(PRTSYM+1)=0

LISTSCR:
	xor	a
	ld	(PRTSYM+1),a
LISTCRT:
;	xor	a
	ld	(SELPRN),a
;	jr	LISTAFTFD

;check comma after file descriptor
LISTAFTFD:
	pop	hl

	call	GETCMM
	jr	z,LISTRANGE
	xor	a		;set z-flag
	jr	LISTALL

;check list range
LISTRANGE:
	ld	de,(ERLVAL)	;error line number
	ld	b,d
	ld	c,e
	call	SKIPSP
	cp	'.'
	jr	z,LISTSTRT

	call	CHKCLN
LISTALL:
	ld	de,0000h
	ld	bc,65534
	call	nz,LISTPAR

;de=start line number, bc=end line number
LISTSTRT:
	push	bc
	call	SRCHLN
	ld	h,b
	ld	l,c
LISTLP:
	call	STOPESC
	jp	z,STOPSP

	inc	hl
	ld	a,(hl)
	inc	hl
	or	(hl)
	jr	z,LISTEND2

	pop	de		;end line number
	push	de
	inc	hl
	push	hl		;line number address
	ld	a,(hl)
	inc	hl
	ld	h,(hl)
	ld	l,a

	inc	de
	or	a
	sbc	hl,de
	add	hl,de
	jr	nc,LISTEND

	pop	hl
	call	LIST1
	call	PUTNL
	jr	LISTLP

LISTEND:
	pop	af		;cancel stack
LISTEND2:
	pop	af		;cancel stack
	xor	a
	ld	(SELPRN),a
	jp	EDIT


;list 1 line
;input: hl=line number address
;output: hl=line end address (00)
;destroy: af,bc,de
LIST1:
	ld	a,(hl)
	inc	hl
	push	hl
	ld	h,(hl)
	ld	l,a
	call	PUTI2
	ld	a,' '
	call	PUTC
	pop	hl

	ld	de,0000h	;double quotation mark and DATA counter
	ld	c,e		;REM counter
LS1LP1:
	inc	hl
	ld	a,(hl)
	or	a
	ret	z

	cp	22h
	jr	z,LISTDQ

	bit	0,d
	jr	nz,LISTPUT

	inc	e
	dec	e
	jr	nz,LISTCHKCLN

	inc	c
	dec	c
	jr	nz,LISTPUT

	cp	':'
	jr	z,LISTCLN

	cp	' '
	jr	c,LISTNUM

	cp	REM_
	jr	z,LISTREM
	cp	DATA_
	jr	z,LISTDATA
	or	a
	jp	m,LISTCMD

LISTPUT:
	push	hl
	ld	hl,(PRTSYM+1)
	inc	l
	dec	l
	call	nz,BACN		;CRT:, PLT:
	ld	b,l
	ld	hl,(SELPRN)
	inc	l
	dec	l
	pop	hl
	jr	nz,LSPUTNZ1	;PLT:, LPT:
	inc	b
	dec	b
	jr	nz,LSPUTNZ2	;CRT:

;SCR:, CAS:
	call	PUTC
	jr	LS1LP1

;PLT:, LPT:
LSPUTNZ1
	inc	b
	dec	b
	call	nz,PRNCODE	;PLT:

LSPUTNZ2:
	call	PUTCH
	jr	LS1LP1

;double quotation
LISTDQ:
	inc	d
	jr	LISTPUT


;in DATA
LISTCHKCLN:
	cp	':'
	jr	nz,LISTPUT
	dec	e
	jr	LISTPUT

LISTCLN:
	inc	hl
	ld	a,(hl)
	cp	27h		;apostrophe
	jr	z,LISTAPOS
	cp	ELSE_
	jr	z,LISTCMD
	dec	hl
	ld	a,(hl)
	jr	LISTPUT

LISTAPOS:
	call	PUTC
	inc	c
	jr	LS1LP1

LISTREM:
	inc	c
LISTDATA:
	inc	e
LISTCMD:
	cp	0feh
	jr	c,LSCMDC
	inc	hl
	ld	b,(hl)
LSCMDC:
	push	hl
	inc	hl
	call	CNVASC

LS1LP2:
	ld	a,(hl)
	and	7fh
	call	PUTC
	ld	a,(hl)
	rlca
	inc	hl
	jr	nc,LS1LP2
	pop	hl
	jp	LS1LP1

LISTNUM:
	cp	0bh
	jr	c,LIST09	;0-9
	inc	hl
	jr	z,LISTLN	;line number
	cp	12h
	jr	c,LISTOBH	;0d=octal, 0e=binary, 0f=hexadecimal
	jr	z,LISTINT	;integer
	cp	15h
	jr	z,LISTSNG	;single
	cp	18h
	jr	z,LISTDBL	;double

;0-9
LIST09:
	push	hl
	dec	a
	ld	h,00h
	jr	LISTI2

;line number
LISTLN:
	ld	a,(hl)
	inc	hl
	push	hl
	ld	h,(hl)
LISTI2:
	ld	l,a
	push	de
	push	bc
	call	PUTI2
	pop	bc
	pop	de
	pop	hl
	jp	LS1LP1

;a=0dh(octal),0e(binary),0f(hexadecimal)
LISTOBH:
	push	bc
	push	de
	push	hl
	sub	0dh
	ld	c,a		;0-2
	ld	b,00h
	ld	hl,OBHCHR
	add	hl,bc
	ld	a,'&'
	call	PUTC
	ld	a,(hl)
	call	PUTC
	pop	hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	push	hl
	sla	c
	ld	hl,OBHTBL
	add	hl,bc
	ld	a,(hl)
	inc	hl
	ld	h,(hl)
	ld	l,a
	call	JPHL
	ld	b,a
LISTOBHLP:
	ld	a,(hl)
	call	PUTC
	inc	hl
	djnz	LISTOBHLP
	pop	hl
	pop	de
	pop	bc
	jp	LS1LP1

OBHCHR:
	db	"OBH"
OBHTBL:
	dw	OCTMAIN, BINMAIN, HEXMAIN


LISTINT:
LISTSNG:
LISTDBL:
	push	bc
	push	de
	sub	10h
	ld	de,FAC1
	ld	(de),a
	inc	de
	ld	b,00h
	ld	c,a
	ldir
	dec	hl
	push	hl
	call	FTOA
	call	SKIPSP
	call	PUTS
	ld	a,(FAC1)
	cp	02h		;integer
	jr	z,LISTINTEND
	cp	08h
	jr	z,LISTINTZ
	call	FRAC
	ld	a,(FAC1+1)	;exponent
	or	a
	jr	nz,LISTINTEND
	ld	a,(FAC1)	;05 or 08
LISTINTZ:
	dec	a
	and	02h		;00 or 02
	add	a,'!'		;'!' or '#'
	call	PUTC

LISTINTEND:
	pop	hl
	pop	de
	pop	bc
	jp	LS1LP1

;convert intermediate code to command/function ASCII characters
;input: a=1st code, (b=2nd code)
;output: hl=address
;destroy: af,b,hl
CNVASC:
	ld	hl,CMDNAME
	cp	0feh
	jr	c,CNVASCZ
	ld	a,b
	ld	hl,FNCNAME
	jr	nz,CNVASCZ
	ld	hl,CMDNAMEFE
CNVASCZ:
	sub	80h
	ret	z
	ld	b,a
CNVASCLP:
	ld	a,(hl)
	rlca
	inc	hl
	jr	nc,CNVASCLP
	djnz	CNVASCLP
	ret


;VERIFY command
C_VRFY:
	ld	a,01h		;1=verify
	jr	LOADNZ

;LOAD command
C_LOAD:
	xor	a
LOADMERG:
	ld	(LOADASC+1),a
	ld	a,(hl)
	cp	'M'
	jp	z,LOADM
	call	SKIPSP
	cp	PRINT_		;load?
	ld	a,00h
	jr	nz,LOADNZ	;0=load
	ld	a,01h		;1=verify
	inc	hl
LOADNZ:
	push	af
	call	LOADMAIN
	pop	af
	or	a
	call	z,RESSTK
	jp	EDIT

;input: a(0=load, 1=verify)
LOADMAIN:
	ld	(LOADMODE+1),a
	call	SARGNOVAL
	call	SETFNAME

CPYFNAME:
	call	LOADSRCH
	dec	a
	jp	z,TRERR
	dec	a
	jp	z,STOPSP
LOADMODE:
	ld	a,00h
	ld	b,a		;load mode
	cp	02h		;2=loadm
	ld	a,(IBUFE)	;file attribute
	jr	nc,CHKATR2	;2=loadm, 3=loadm+address
	dec	b
	jr	z,CHKATR1	;1=verify
;0=load
	cp	04h
	jp	z,LOADASC	;file attribute=4: ASCII
CHKATR1:
	sub	02h
	ld	hl,(BASICAD)
	jr	z,LOADSETAD	;file attribute=2: BASIC
	jp	BMERR

;loadm,loadm+1
CHKATR2:
	dec	a
	jp	nz,BMERR	;file attribute=1: binary
	ld	a,b		;load mode
	cp	02h
	jr	z,NOAD		;2=loadm
LOADAD:
	ld	hl,0000h	;load address

LOADSETAD:
	ld	(DTADR),hl
NOAD:
	ld	a,(LOADMODE+1)	;0=load, 1=verify, 2=loadm, 3=loadm+address
	ld	d,a
	dec	a
	call	z,VRFY		;1=verify
	dec	d
	call	nz,RDD
	dec	a
	jp	z,LOADERR
	dec	a
	jp	z,LOADSTOP

	ld	a,(LOADMODE+1)
	dec	a
	ret	z		;1=verify
	inc	a
;	jr	nz,LOADMEND	;2=loadm, 3=loadm+address
	ret	nz		;2=loadm, 3=loadm+address
;0=load
	ld	hl,(SIZE)
	ld	de,(DTADR)
	add	hl,de
	call	CHKSTK
	ld	(VARAD),hl
	ret

LOADERR:
	call	CHKVRF
	jp	TRERR
LOADSTOP:
	call	CHKVRF
	jp	STOPSP

;verify or not
CHKVRF:
	ld	a,(LOADMODE+1)
	dec	a
	ret	z		;mode1=verify
	jp	NEW


;LOADM command
LOADM:
	inc	hl
	ld	a,02h
	ld	(LOADMODE+1),a
	push	af		;auto start?

;file name
	call	CHKCLN
	ld	a,00h		;file name length=0
	jr	z,LOADMZ1
	ld	a,(hl)
	sub	','		;file name length=0 if comma
	call	nz,STRARG
LOADMZ1:
	push	hl
	call	SETFNAME
	pop	hl
	call	GETCMM
	jr	nz,LOADMEND

;address
	call	GETCMM
	jr	z,LOADMZ2
	call	INTARG
	ld	(LOADAD+1),de
	ld	a,03h
	ld	(LOADMODE+1),a	;3=loadm+address
	call	GETCMM
	jr	nz,LOADMEND

;auto start
LOADMZ2:
	call	SKIPSP
	cp	'R'
	jp	nz,SNERR
	pop	bc		;cancel stack
	push	af		;auto start
	inc	hl

LOADMEND:
	ld	(PROGAD),hl
	call	CPYFNAME
	pop	af		;auto start?
	cp	'R'
	ret	nz
	ld	hl,(EXADR)
	jp	(hl)


;search file for LOAD
;output: a(0=ok, 1=checksum error, 2=shift+break), c-flag(1=error)
LOADSRCH:
	ld	hl,FNAME
	ld	de,FNAMEBK
	ld	bc,0011h
	ldir

	ld	d,12h
LOADLP1:
	dec	hl
	dec	d
	jr	z,LOADLP2
	ld	a,(hl)
	cp	' '
	jr	z,LOADLP1

LOADLP2:
	call	RDI
	ret	c

;compare file name
	ld	a,d		;file name length
	or	a
	jr	z,LOADSRCHEND

;file name(13)
	ld	de,FNAMEBK
	ld	hl,FNAME
	ld	b,0dh
LOADLP3:
	ld	a,(de)
	cp	(hl)
	jr	nz,LOADSKIP
	inc	de
	inc	hl
	djnz	LOADLP3

;suffix(3)
	ld	de,FNAMEBK+0dh
	ld	hl,FNAME+0dh
	ld	b,03h
LOADLP4:
	ld	a,(de)
	cp	(hl)
	jr	nz,LOADSKIP
	inc	de
	inc	hl
	djnz	LOADLP4
;password(1)
	ld	a,(FNAMEBK+10h)
	ld	hl,FNAME+10h
	cp	(hl)
	jr	z,LOADSRCHEND
LOADSKIP:
	ld	hl,SKIP_STR
	call	PUTFN
	jr	LOADLP2

LOADSRCHEND:
	ld	hl,FOUND_STR
	call	PUTFN
	xor	a		;reset c-flag
	ret


;ASCII load
LOADASC:
	ld	a,00h
	or	a
	call	z,NEW
	ld	b,04h		;file attribute=4: ASCII
	ld	d,01h		;file open mode=1: input
	call	FOPEN0
	ld	hl,INPBUF

LDASCLP1:
	call	GETDEV
	jp	c,CLRBUFP
	cp	0dh
	jr	z,LDASCLP1
	or	a
	jr	z,LDASCZ
	call	CHKFIG
	jp	nc,BMERR
LDASCZ:
	ld	hl,INPBUF
	ld	b,MAXINP
	jr	LDASCPUT

LDASCLP2:
	call	GETDEV
	jp	c,CLRBUFP
	cp	0dh
	jr	z,LOADCNV
	or	a
	jr	z,LOADCNV
LDASCPUT:
	ld	(hl),a
	inc	hl
	djnz	LDASCLP2

LOADCNV:
	push	af		;
	ld	(hl),00h
	ld	hl,INPBUF
	call	INPTPRG
	pop	af		;
	or	a
	jr	z,LDASCZ
	jr	LDASCLP1

FNAMEBK:
	ds	11h
FOUND_STR:
	db	"FOUND  ", 00h
SKIP_STR:
	db	"SKIP  ",00h


;put message and file name in device
;input: hl=message address
;destroy: af,hl,(bc,de,FAC1)
PUTFN:
	call	PUTS
	ld	a,22h
	call	PUTC
	ld	hl,FNAME+0dh
	ld	b,0dh
PUTFNLP1:
	dec	hl
	ld	a,(hl)
	cp	21h
	jr	nc,PUTFNNC
	djnz	PUTFNLP1
	jr	PUTFNSFX

PUTFNNC:
	ld	hl,FNAME
PUTFNLP2:
	ld	a,(hl)
	inc	hl
	call	PUTC
	djnz	PUTFNLP2

;suffix
PUTFNSFX:
	ld	hl,FNAME+0dh
	ld	a,(hl)
	cp	21h
	jr	c,PUTFNEND
	ld	a,'.'
	call	PUTC

	ld	b,03h
SFXLP:
	ld	a,(hl)
	cp	21h
	jr	c,PUTFNEND
	call	PUTC
	inc	hl
	djnz	SFXLP

PUTFNEND:
	ld	a,22h
	call	PUTC
	jp	PUTNL


;MERGE command
C_MERG:
	ld	a,01h
	jp	LOADMERG

;SAVEM command
SAVEM:
	inc	hl
	call	ARG
	jp	z,MOERR
	ld	a,(FAC1)
	cp	03h
	jr	z,SAVEMFN	;with file name

;no file name, with address
	xor	a
	push	hl
	call	SETFNAME
	pop	hl
	jr	SAVEMAD

;with file name
SAVEMFN:
	call	STRARG2
	call	SETFNAME
	ld	hl,(PROGAD)
	ld	a,(hl)
	cp	','
	jp	nz,MOERR
	inc	hl
	ld	(PROGAD),hl
	call	ARG
	jp	z,MOERR

SAVEMAD:
	call	CHKNUM
	push	hl
	call	FTOI
	ex	de,hl
	ld	(DTADR),hl
	ld	(EXADR),hl
	pop	hl
	ld	a,(hl)
	cp	','
	jp	nz,MOERR
	inc	hl
	call	INTARG
	push	hl		;
	ld	hl,(DTADR)
	ex	de,hl
	or	a
	sbc	hl,de
	inc	hl
	ld	(SIZE),hl
	pop	hl		;
	ld	a,(hl)
	cp	','
	jr	nz,SAVEMNZ
	inc	hl
	call	INTARG
;	ld	a,01h
;	ld	(ATSTRT),a
SAVEMNZ:
	ld	(EXADR),de
	ld	a,01h
	ld	(IBUFE),a	;file attribute=1: binary
	jr	SAVEMAIN


;SAVE command
C_SAVE:
	ld	a,(hl)
	cp	'M'
	jr	z,SAVEM
	call	SARGNOVAL
	call	SETFNAME
	ld	hl,(PROGAD)
	call	SKIPSP
	cp	','
	jr	z,SAVEASC
	ld	a,02h
	ld	(IBUFE),a	;file attribute=2: BASIC
	ld	hl,(BASICAD)
	ex	de,hl
	ld	hl,(VARAD)
	or	a
	sbc	hl,de
	ld	(SIZE),hl
;	ld	a,0dh
;	ld	(FNAME+10h),a	;temporary

SAVEMAIN:
	call	WRI
	jp	c,STOPSP
	ex	de,hl
	ld	(DTADR),hl
	call	WRD
	jp	c,STOPSP
	call	CLOSALL
	ret

ASCOPEN:
	ld	a,04h
	ld	(IBUFE),a	;file attribute=4: ASCII
	call	WRI
	ld	b,04h		;file attribute=4: ASCII
	ld	d,02h		;file open mode=2: output
	call	FOPEN0
	jp	c,STOPSP
	ret

;ASCII save
SAVEASC:
	call	SKIPSPINC
	cp	'A'
	jp	nz,SNERR
	inc	hl
	ld	(PROGAD),hl

	call	ASCOPEN

	pop	af		;cancel return address

	xor	a
	ld	(PRTSYM+1),a
	jp	LISTALL		;z-flag=1


;set file name
;input: a=length, de=address
;destroy: af,bc,de,hl
SETFNAME:
	ld	hl,FNAME
	ld	b,11h
SETFNLP1:
	ld	(hl),' '
	inc	hl
	djnz	SETFNLP1

	ld	b,6eh
SETFNLP2:
	ld	(hl),00h
	inc	hl
	djnz	SETFNLP2

	or	a
	jr	z,CHKDEVCAS
	cp	20h
	jr	nc,SETFNFCERR

;device
	push	af		;length
	ld	b,':'
	call	FNCPIR
	jr	nz,NODEV
	pop	af

	push	bc
	call	CHKDEV
	cp	DEVCAS		;CAS:
	pop	bc
	ld	a,c
	jr	z,CHKPSWD
	jr	SETFNFDERR

NODEV:
	call	CHKDEVCAS
	pop	af		;length

;password?
CHKPSWD:
	or	a
	ret	z
	push	af
	ld	b,';'
	call	FNCPIR
	jr	nz,NOPSWD

	pop	af
	inc	c
	sub	c
	push	af
	dec	c
	jr	z,NOPSWD

	ld	a,(hl)
	ld	a,20h		;temporary
	ld	(FNAME+10h),a

;suffix?
NOPSWD:
	pop	af		;length
	or	a
	ret	z
	push	af
	ld	b,'.'
	call	FNCPIR
	jr	nz,NOSUF

	pop	af
	inc	c
	sub	c
	push	af
	dec	c
	jr	z,NOSUF

	ld	bc,FNAME+0dh
	ld	a,(hl)
	ld	(bc),a
	inc	hl
	inc	bc
	ld	a,(hl)
	ld	(bc),a
	inc	hl
	inc	bc
	ld	a,(hl)
	ld	(bc),a

;file name
NOSUF:
	pop	af		;length
	or	a
	ret	z
	cp	0eh
	jr	nc,SETFNFDERR
	ld	c,a
	ld	b,00h
	ex	de,hl
	ld	de,FNAME
	ldir
	ret

;check if default device name is "CAS:"
;destroy: af
CHKDEVCAS:
	ld	a,(DEV_DEF)
	cp	DEVCAS		;CAS:
	ret	z
	jp	FDERR


;cpir or SETFNAME
;input: a=length, de=address, b=target
;output: c=position, hl=address, z-flag(1=found)
;destroy: af,b
FNCPIR:
	ld	h,d
	ld	l,e
	ld	c,a
	ld	a,b
	ld	b,00h
	cpir
	ret

;error
SETFNFCERR:
	ld	a,(INMON)
	or	a
	jp	z,FCERR
	ld	hl,ERR05
	jp	MONERRSTOP

SETFNFDERR:
	ld	a,(INMON)
	or	a
	jp	z,FDERR
	ld	hl,ERR65
	jp	MONERRSTOP


;CLOSE command
C_CLOS:
	call	SKIPSP
	cp	'#'
	jr	z,CLOSLP1
	call	NUMARG
	jr	nz,CLOSNUM
CLOSALL:
	ld	a,(FILES)
	ld	e,a
CLOSALP:
	push	de
	call	FCLOSE
	pop	de
	dec	e
	jp	p,CLOSALP
	ret

CLOSLP1:
	inc	hl
CLOSLP2:
	call	ARG
	jp	z,SNERR		;no number
	ld	a,(FAC1)
	cp	03h
	jp	z,SNERR
CLOSNUM:
	push	hl
	call	FTOI1
	ld	a,(FILES)
	cp	e
	jp	c,FCERR
	call	FCLOSE
	pop	hl
	ld	a,(hl)
	cp	','
	ret	nz
	call	SKIPSPINC
	cp	'#'
	jr	nz,CLOSLP2
	jr	z,CLOSLP1


;OPEN command
C_OPEN:
	call	STRARG
	dec	a
	jp	nz,FCERR
	ld	a,(de)
	cp	'I'		;read
	jr	z,OPENOK
	cp	'O'		;write
	jp	nz,SNERR
OPENOK:
	push	af		;'I' or 'O'
	call	GETCMER
	call	SKIPSP
	cp	'#'
	jr	nz,OPENNZ1
	inc	hl
OPENNZ1:
	call	INT1ARG
	ld	a,(FILES)
	cp	e		;e=file number
	jp	c,FCERR
	ld	a,e
	or	a
	jp	z,FCERR
	push	hl
	call	GETBUFP
	jp	nz,AOERR
	ex	(sp),hl		;buffer address
	ld	a,(hl)
	cp	','
	ld	a,00h
	jr	nz,OPENNZ2
	inc	hl
	call	STRARG
OPENNZ2:
	call	SETFNAME
	pop	de		;buffer address

	pop	af		;'I'=read, 'O'=write
	cp	'O'
	jr	z,WOPEN

ROPEN:
	xor	a
	push	de		;buffer address
	call	LOADSRCH
	pop	hl		;buffer address
	dec	a
	jp	z,TRERR
	dec	a
	jp	z,STOPSP
	ld	b,04h		;file attribute=4: ASCII
	ld	d,01h		;file open mode(1=read)
	jp	FOPEN

WOPEN:
	ld	a,04h
	ld	(IBUFE),a
	call	WRI
	ex	de,hl
	ld	b,04h
	ld	d,02h		;file open mode(2=write)
	jp	FOPEN


;INIT command
C_INIT:
	call	SARGNOVAL
	jr	nz,INITOK	;no parameter
	call	STRARG2
	call	GETDEVNUM
	cp	03h		;SCR: or CRT:
	jp	nc,FDERR
INITOK:
	ld	hl,(LINENUM)
	ld	a,h
	and	l
	inc	a
	jr	nz,INITNZ
	ld	hl,SURE_STR
	call	PUTS
	call	GETC
	ld	b,a
	call	PUTC
	call	PUTNL
	ld	a,b
	cp	'Y'
	ret	nz
INITNZ:
	jp	SEDINICON

SURE_STR:
	db	"ARE YOU SURE? ", 00h


;get device number (default device number if "")
;input: a=length, de=string address
;output: a=device number(0=not found), de=de+4 if found, hl=descriptor string
;destroy: af,bc,hl
GETDEVNUM:
	or	a
	jr	nz,CHKDEVERR
	ld	a,(DEV_DEF)
	ret


;check device name and error
;input: a=length, de=string address
;output: a=device number, de=de+4 if found, hl=descriptor string
;destroy: af,bc,hl
CHKDEVERR:
	call	CHKDEV
	or	a
	ret	nz
	jp	FDERR


;check device name
;input: a=length, de=string address
;output: a=device number(0=not found), de=de+4 if found, hl=descriptor string
;destroy: f,bc,hl
CHKDEV:
	cp	04h
	ld	a,00h
	ret	c

	ld	hl,DEV_KEY
	ld	c,06h
CHKDLP1:
	push	de
	push	hl
	ld	b,04h
CHKDLP2:
	ld	a,(de)
	cp	(hl)
	jr	nz,CHKDNZ
	inc	de
	inc	hl
	djnz	CHKDLP2
;found
	pop	hl
	pop	af		;cancel stack
	ld	a,c
	ret

CHKDNZ:
	pop	hl
	ld	de,0005h
	add	hl,de
	pop	de
	dec	c
	jr	nz,CHKDLP1
	xor	a		;not found
	ret


;device name string
;6:keyboard
DEV_KEY:
	db	"KEY:",00h
;5:printer
DEV_LPT:
	db	"LPT:",00h
;4:plotter
DEV_PLT:
	db	"PLT:",00h
;3:cassete
DEV_CAS:
	db	"CAS:",00h
;2:screen
DEV_SCR:
	db	"SCR:",00h
;1:CRT (symbol, print#0)
DEV_CRT:
	db	"CRT:",00h


;LFILES command
C_LFLS:
	call	SETPRN
;	jr	C_FLS

;FILES command
C_FLS:
	call	PUTNLX
	call	SARGNOVAL
	jr	z,FLSZ1		;with string

	call	CHKDEVCAS
	jr	FLSRDI

FLSZ1:
	call	GETDEVNUM
	cp	DEVCAS		;CAS:
	jp	nz,FDERR

FLSRDI:
	call	RDI
	dec	a
	jp	z,TRERR
	dec	a
	jp	z,STOPSP
	ld	a,(IBUFE)
	ld	hl,BIN_STR
	cp	01h
	jr	z,FLSZ2
	ld	hl,BAS_STR
	cp	02h
	jr	z,FLSZ2
	ld	hl,ASC_STR
	cp	04h
	jr	z,FLSZ2
	ld	hl,UNKNOWN_STR
FLSZ2:
	call	PUTS
	ld	a,' '
	call	PUTC
	ld	hl,DEV_CAS
	call	PUTS
	ld	hl,IBUFE+0001h
	ld	b,10h
FLSLP1:
	ld	a,(hl)
	cp	0dh
	jr	nz,FLSNZ
	ld	a,' '
FLSNZ:
	call	PUTC
	ld	a,b
	cp	04h
	ld	a,'.'
	call	z,PUTC
	inc	hl
	djnz	FLSLP1

	ld	hl,IBUFE+0012h
	ld	b,03h
FLSLP3:
	push	bc
	ld	a,':'
	call	PUTC
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
	push	hl
	call	HEXMAIN
	ld	b,a
	sub	04h
	jr	z,FLSLP5
	ld	c,a
FLSLP4:
	ld	a,'0'
	call	PUTC
	inc	c
	jr	nz,FLSLP4
FLSLP5:
	ld	a,(hl)
	inc	hl
	call	PUTC
	djnz	FLSLP5
	pop	hl
	pop	bc
	djnz	FLSLP3

	xor	a
	ld	(SELPRN),a
	ret

BIN_STR:
	db	"BIN", 00h
BAS_STR:
	db	"BAS", 00h
ASC_STR:
	db	"ASC", 00h
UNKNOWN_STR:
	db	"???", 00h


;INT() function
;(round toward minus infinity)
F_INT:
	call	CHKNUM
INT:
	call	CPYFAC
	call	FIX
	call	CMPF
	ret	z
	ret	c
	call	CPYFAC
	call	SETMNS1
	jp	ADDF


;ABS() function
F_ABS:
	call	CHKNUM
ABS:
	ld	a,(FAC1)	;type
	cp	02h
	jr	z,ABSI
	ld	hl,FAC1+2	;sign
	res	7,(hl)
	ret

ABSI:
	ld	hl,(FAC1+1)
	ld	a,h
	rlca
	ret	nc
	xor	a
	sub	l
	ld	l,a
	sbc	a,h
	sub	l
	ld	h,a
	ld	(FAC1+1),hl
	ret


;SIN() function
;x-x^3/3!+x^5/5!-...=x(1-x^2/(2*3)(1-x^2/(4*5)(...(1-x^2/(12*13)))))))))))
F_SIN:
	call	CHKNUM
	ld	a,(FAC1)	;type
	cp	02h
	call	z,ITOS
SIN:
	call	PUSHF1		;x
	ld	hl,PI2S
	call	SNGORDBL
	call	SETF2
	call	DIVF		;x/2pi
	call	FIX		;int(x/2pi)
	ld	hl,PI2S
	call	SNGORDBL
	call	SETMULF		;int(x/2pi)*2pi
	call	POPF2		;x
	call	SUBF21		;x'=x-int(x/2pi)*2pi

	call	SINRANGE	;[-pi,+pi]
	call	SINRANGE	;[-pi/2,+pi/2]

	call	PUSHF1		;x'
	call	CPYFAC
	call	MULF		;x'^2

	call	CPYFAC
	call	SETPLS1		;y=1

	ld	a,(FAC1)	;5 or 8
	add	a,a
	add	a,02h
	ld	b,a		;12 or 18
SINLP:
	call	PUSHF2		;x'^2
	push	bc
	call	MULF		;x'^2*y
	pop	bc
	push	bc
	call	PUSHF1		;x'^2*y

	ld	a,b
	inc	b
	call	MULINT1		;hl=b(b+1)
	ld	(FAC1+1),hl
	ld	a,(FAC1)
	call	ITOSD

	call	CPYFAC
	call	POPF1		;x'^2*y
	call	DIVF		;x'^2/b/(b+1)*y
	call	NEGABS		;-x'^2/b/(b+1)*y (y>0)
	call	INCF1		;y=1-x'^2/b/(b+1)*y

	pop	bc
	call	POPF2		;x'^2
	dec	b
	djnz	SINLP

	call	POPF2		;x'
	call	MULF
	ld	a,(FAC1+1)	;exponent
	sub	81h
	ret	nz

;if |result|>=1
	ld	hl,FAC1
	ld	e,(hl)		;type
	ld	d,00h
	add	hl,de		;FAC1+5or8
	ld	(hl),d		;=0
	ret


;[-pi/2,+pi/2]
SINRANGE:
	ld	hl,PIDIV2S	;pi/2
	call	SNGORDBL
	call	SETCMPF
	jr	nc,RANGENC
	ld	hl,FAC2+2	;sign
	set	7,(hl)		;-pi/2
	call	CMPF
	ret	nc
RANGENC:
	ld	hl,FAC2+1	;exponent
	inc	(hl)		;+-pi
	jp	SUBF21


;COS() function
;cos(x)=sin(pi/2-|x|)
F_COS:
	call	CHKNUM
	ld	a,(FAC1)	;type
	cp	02h
	call	z,ITOS
COS:
	call	NEGABS
	ld	hl,PIDIV2S
	call	SNGORDBL
	call	SETADDF		;pi/2-|x|
	jp	SIN


;TAN() function
;tan(x)=sin(x)/cos(2)
F_TAN:
	call	CHKNUM
	ld	a,(FAC1)	;type
	cp	02h
	call	z,ITOS
	call	PUSHF1		;x
	call	COS		;cos(x)
	call	CPYFAC
	call	POPF1		;x
	call	PUSHF2		;cos(x)
	call	SIN		;sin(x)
	call	POPF2		;cos(x)
	jp	DIVF		;sin(x)/cos(x)


;LOG() function
;log(x)=log(a*2^n)=log(a)+n*log(2)
;a'=(a-1)/(a+1)
;log(a)=2*(a'+a'^3/3+a'^5/5+...)=2a'(1+1/3*a'^2(1+3/5*a'^2(...(1+19/21*a'^2))))))))))
F_LOG:
	call	CHKNUM
	ld	a,(FAC1)	;type
	cp	02h
	call	z,ITOS
LOG:
	call	CHKSGNSD
	jp	z,FCERR		;x=0
	jp	c,FCERR		;x<0

	ld	hl,FAC1+1
	ld	a,(hl)		;exponent
	sub	81h
	push	af		;n
	ld	(hl),81h	;a
	call	PUSHF1		;a
	call	INCF1
	call	POPF2		;a
	call	PUSHF1		;a+1
	call	SETMNS1
	call	ADDF		;a-1
	call	POPF2		;a+1
	call	DIVF		;a'=(a-1)/(a+1)
	call	PUSHF1		;a'
	call	CPYFAC
	call	MULF		;a'^2
	call	CPYFAC		;a'^2
	call	SETPLS1		;y0

	ld	bc,0012h
	ld	a,(FAC2)
	cp	08h
	jr	nz,LOGLP
	ld	bc,0020h
LOGLP:
	call	PUSHF2		;a'^2
	push	bc
	call	MULF		;y*a'^2
	call	CPYFAC
	pop	hl
	push	hl
	inc	l
	call	I2TOF
	call	EXFAC
	call	DIVF		;y*a'^2/(c+1)
	call	CPYFAC
	pop	hl
	dec	l
	push	hl
	call	I2TOF		;c-1
	call	MULF		;y*a'^2*(c-1)/(c+1)
	call	INCF1		;y=1+y*a'^2+(c-1)/(c+1)
	pop	bc
	call	POPF2		;a'^2
	dec	c
	jr	nz,LOGLP

	call	POPF2		;a'^2
	call	MULF		;a'*y
	ld	hl,FAC1+1
	ld	a,(hl)		;exponent
	or	a
	jr	z,LOGZ
	inc	(hl)		;2*a'*y
	jp	z,OVERR
LOGZ:
	pop	bc		;b=n
	call	PUSHF1		;2*a'*y
	ld	a,b
	call	I1TOF
	ld	hl,LOG2S
	call	SNGORDBL
	call	SETMULF		;n*log(2)
	call	POPF2		;2*a'*y
	jp	ADDF		;n*log(2)+2*a'*y


;EXP() function
;exp(x)=2^n * e^a, n=[x/log2], a=x-n*log2
F_EXP:
	call	CHKNUM
	ld	a,(FAC1)	;type
	cp	02h
	call	z,ITOS
EXP:
	call	PUSHF1		;x
	ld	hl,LOG2S
	call	SNGORDBL
	call	SETF2
	call	DIVF		;x/log2

	ld	hl,(FAC1+1)
	ld	a,l		;exponent
	cp	89h
	jr	nc,EXPLARGE	;|x/log2|>=256

	call	INT		;n=[x/log2]
	call	FTOI
	push	de		;n
	ld	hl,LOG2S
	call	SNGORDBL
	call	SETMULF		;n*log2
	pop	de		;n
	call	POPF2		;x
	push	de		;n
	call	SUBF21		;a=x-n*log2

;1+a(1+a/2(1+a/3(1+...(1+a/10))))))))))
	call	CPYFAC
	call	SETPLS1

	ld	a,(FAC2)	;5 or 8
	add	a,a
	ld	b,00h
	ld	c,a		;10 or 16
EXPLP:
	call	PUSHF2		;a
	push	bc
	call	MULF		;a*y
	call	CPYFAC
	pop	hl
	push	hl
	call	I2TOF
	call	EXFAC
	call	DIVF		;a*y/b
	call	INCF1		;a*y/b+1
	pop	bc
	call	POPF2		;a
	dec	c
	jr	nz,EXPLP

	pop	de		;-256<=n<=255
;	ld	a,(FAC1+1)	;=81h?
	ld	a,81h
	add	a,e
	ld	(FAC1+1),a	;exponent
	bit	7,d
	jr	nz,EXPNZ
	ret	nc		;0<=n<=126
EXPOV:
	jp	OVERR		;127<=n<=255
EXPNZ:
	dec	a
	ret	p		;-128<=n<=-1
EXPZERO:
	jp	SETZERO		;-256<=n<=-129

EXPLARGE:
	bit	7,h		;(FAC1+2)
	jr	z,EXPOV		;x/log2>=256
	call	POPF1
	jr	EXPZERO		;x/log2<=-256

;log(2)
LOG2S:
	db	05h, 80h, 31h, 72h, 17h, 0f8h
LOG2D:
	db	08h, 80h, 31h, 72h, 17h, 0f7h, 0d1h, 0cfh, 7ah


;SQR() function
;y_n+1=(y_n + x/y_n)/2
F_SQR:
	call	CHKNUM
	ld	a,(FAC1)	;type
	cp	02h
	call	z,ITOS
	call	CHKSGNSD
	ret	z		;sqr(0)=0
	jp	c,FCERR		;sqr(-x)

	call	PUSHF1
	ld	hl,FAC1+1
	ld	a,(hl)		;exponent
	sub	81h
	rra			;=sra a, c-flag=bit7
	add	a,81h
	ld	(hl),a		;exponent of y_0 = (exponent of x)/2

	ld	b,05h
SQRLP:
	call	CPYFAC
	call	POPF1		;x
	call	PUSHF1		;x
	push	bc
	call	PUSHF2		;;y
	call	DIVF		;x/y
	call	POPF2		;;y
	call	ADDF		;x/y+y
	ld	hl,FAC1+1
	dec	(hl)		;y=(x/y+y)/2
	pop	bc
	djnz	SQRLP
	call	POPF2
	ret


;RND function
F_RND:
	call	SKIPSP
	ld	(PROGAD),hl
	cp	'('
	ld	a,05h
	ld	(FAC1),a	;default type=single
	call	z,FNCNUM
	ld	hl,FAC1
	ld	a,(hl)
	cp	02h
	jr	nz,RNDNZ
	ld	a,05h
	ld	(hl),a
RNDNZ:
	call	RNDSFT
	ld	hl,SEED
	ld	de,FAC1+2
	ld	a,(FAC1)	;type
	dec	a
	ld	b,00h
	ld	c,a		;4 or 7
	ldir
	call	INTTOF

	ld	a,(FAC1)	;5 or 8
	dec	a
	add	a,a
	add	a,a
	add	a,a
	ld	b,a		;20h or 38h

	ld	hl,FAC1+1	;exponent
	ld	a,(hl)
	or	a
	ret	z

	sub	b		;1/(2^32) or 1/(2^56)
	ld	(hl),a
	ret

;xorshift
RNDSFT:
	ld	c,07h
	call	SFTL
	ld	c,09h
;	jr	SFTR

;shift right
;input: c
;destroy: af,bc,de,hl
SFTR:
	call	GETSEED
SFTRLP1:
	ld	hl,FAC1+1
	ld	b,08h
	or	a
SFTRLP2:
	rr	(hl)
	inc	hl
	djnz	SFTRLP2
	dec	c
	jr	nz,SFTRLP1
;	jr	XORSEED

;seed ^= shifted value
;input: seed, [FAC1+1,FAC1+8]
;output: seed
;destroy: af,b,de,hl
XORSEED:
	ld	de,FAC1+1
	ld	hl,SEED
	ld	b,08h
XORSEEDLP:
	ld	a,(de)
	xor	(hl)
	ld	(hl),a
	inc	de
	inc	hl
	djnz	XORSEEDLP
	ret

;shift left
;input: c
;destroy: af,bc,de,hl
SFTL:
	call	GETSEED
SFTLLP1:
	ld	hl,FAC1+8
	ld	b,08h
	or	a
SFTLLP2:
	rl	(hl)
	dec	hl
	djnz	SFTLLP2
	dec	c
	jr	nz,SFTLLP1
	jr	XORSEED

;copy seed to [FAC1+1,FAC1+8]
;destroy: f,de,hl
GETSEED:
	push	bc
	ld	hl,SEED
	ld	de,FAC1+1
	ld	bc,0008h
	ldir
	pop	bc
	ret

SEED:
	db	01h, 23h, 45h, 67h, 89h, 0abh, 0cdh, 0efh


;PEEK() function
F_PEEK:
	call	SKIPSP
	cp	'@'
	push	af		;z-flag
	jr	nz,PEEKVRAM	;peek()
	inc	hl		;peek@()
PEEKVRAM:
	call	FNCNUM
	call	FTOI
	pop	af		;z-flag
	ld	a,(de)
	jr	z,PEEKEND	;peek@()
;peek()
	call	CHGVRAM
	ld	a,(de)
	ld	h,a
	call	CHGDRAM
	ld	a,h

PEEKEND:
	ld	hl,FAC1
	ld	(hl),02h
	inc	hl
	ld	(hl),a
	inc	hl
	ld	(hl),00h
	ret


;ATN() function
F_ATN:
	call	CHKNUM
	ld	a,(FAC1)
	cp	02h
	call	z,ITOS
	ld	hl,FAC1+2
	ld	a,(hl)		;sign
	push	af
	res	7,(hl)

	ld	hl,SQ2MNS1S
	call	SNGORDBL
	call	SETCMPF
	jr	c,ATN1		;|x|<sqrt(2)-1

	ld	hl,SQ2PLS1S
	call	SNGORDBL
	call	SETCMPF
	jr	c,ATN2		;|x|<sqrt(2)+1

;atan(|x|)=pi/2+atan(-1/|x|) if sqrt(2)+1<|x|
ATN3:
	call	CPYFAC
	call	SETMNS1
	call	DIVF		;x'=-1/|x|
	ld	hl,PIDIV2S
	call	SNGORDBL
	call	SETF2
	jr	ATNMAIN

;atan(|x|)=pi/4+atan((|x|-1)/(|x|+1)) if sqrt(2)-1<|x|<sqrt(2)+1
ATN2:
	call	PUSHF1		;|x|
	call	INCF1
	call	POPF2		;|x|
	call	PUSHF1		;1+|x}
	call	SETMNS1
	call	ADDF		;|x|-1
	call	POPF2		;1+|x|
	call	DIVF		;x'=(|x|-1)/(|x|+1)
	ld	hl,PIDIV2S
	call	SNGORDBL
	call	SETF2
	ld	hl,FAC2+1	;exponent
	dec	(hl)		;pi/4
	jr	ATNMAIN

;atan(|x|) if |x|<sqrt(2)-1
ATN1:
	ld	hl,ZEROD
	call	SETF2
	ld	a,05h
	ld	(FAC2),a

ATNMAIN:
	call	PUSHF2

;atan(x)=x-x^3/3+x^5/5-x^7/7+...=x(1-1/3*x^2(1-3/5*a^2(...(1-21/23*a^2))))

	call	PUSHF1		;x
	call	CPYFAC
	call	MULF		;x^2
	call	CPYFAC		;x^2
	call	SETPLS1		;y0

	ld	bc,0016h
	ld	a,(FAC2)
	cp	08h
	jr	nz,ATNLP
	ld	bc,0026h
ATNLP:
	call	PUSHF2		;x^2
	push	bc
	ld	hl,FAC1+2
	set	7,(hl)		;sign
	call	MULF		;-y*x^2
	call	CPYFAC
	pop	hl
	push	hl
	inc	l
	call	I2TOF
	call	EXFAC
	call	DIVF		;-y*x^2/(c+1)
	call	CPYFAC
	pop	hl
	dec	l
	push	hl
	call	I2TOF		;c-1
	call	MULF		;-y*x^2*(c-1)/(c+1)
	call	INCF1		;y=1-y*x^2*(c-1)/(c+1)
	pop	bc
	call	POPF2		;x^2
	dec	c
	jr	nz,ATNLP

	call	POPF2		;x
	call	MULF		;x*y
	call	POPF2
	call	ADDF

	pop	af		;sign
	ld	hl,FAC1+2
	xor	(hl)
	and	80h
	xor	(hl)
	ld	(hl),a
	ret


;sqrt(2)+1
SQ2PLS1S:
	db	05h, 82h, 1ah, 82h, 79h, 9ah
SQ2PLS1D:
	db	08h, 82h, 1ah, 82h, 79h, 99h, 0fch, 0efh, 32h

;sqrt(2)-1
SQ2MNS1S:
	db	05h, 7fh, 54h, 13h, 0cch, 0d0h
SQ2MNS1D:
	db	08h, 7fh, 54h, 13h, 0cch, 0cfh, 0e7h, 79h, 92h


;SGN() function
F_SGN:
	call	CHKNUM
	ld	a,(FAC1)
	cp	02h
	call	z,ITOS
	call	CHKSGNSD
	ret	z
	jp	c,SETMNS1
	jp	SETPLS1


;FRAC() function
F_FRAC:
	call	CHKNUM
FRAC:
	call	CPYFAC
	call	FIX
	jp	SUBF21


;FIX() function
;(round toward zero)
F_FIX:
	call	CHKNUM
FIX:
	ld	hl,FAC1
	ld	a,(hl)		;type
	cp	02h
	ret	z
	ld	e,a
	ld	d,00h
	add	hl,de		;FAC1+5or8

	dec	a		;4or7
	add	a,a		;*2
	add	a,a		;*4
	add	a,a		;*8
	add	a,80h		;a0h or b8h
	ld	b,a

	ld	a,(FAC1+1)	;exponent
	cp	81h
	jp	c,SETZERO
	sub	b
	ret	nc
FIXLP1:
	add	a,08h		;
	jr	c,FIXC
	ld	(hl),00h
	dec	hl
	jr	FIXLP1
FIXC:
	ld	b,a
	inc	b
	xor	a
FIXLP2:
	rra
	scf
	djnz	FIXLP2
	and	(hl)
	ld	(hl),a
	ret


;PAI() function
F_PAI:
	call	SKIPSP
	cp	'('
	jr	nz,PAINZ
	inc	hl
	call	NARGMO
	call	CHKRPAR
	ld	hl,PIS
	call	SNGORDBL
	jp	SETMULF
PAINZ:
	ld	hl,PIS
	jp	SETF1

;pi
PIS:
	db	05h, 82h, 49h, 0fh, 0dah, 0a2h
PID:
	db	08h, 82h, 49h, 0fh, 0dah, 0a2h, 21h, 68h, 0c2h
;pi*2
PI2S:
	db	05h, 83h, 49h, 0fh, 0dah, 0a2h
PI2D:
	db	08h, 83h, 49h, 0fh, 0dah, 0a2h, 21h, 68h, 0c2h
;pi/2
PIDIV2S:
	db	05h, 81h, 49h, 0fh, 0dah, 0a2h
PIDIV2D:
	db	08h, 81h, 49h, 0fh, 0dah, 0a2h, 21h, 68h, 0c2h


;RAD() function
F_RAD:
	call	CHKNUM
RAD:
	ld	hl,PIDIV180S
	call	SNGORDBL
	jp	SETMULF

;pi/180
PIDIV180S:
	db	05h, 7bh, 0eh, 0fah, 35h, 13h
PIDIV180D:
	db	08h, 7bh, 0eh, 0fah, 35h, 12h, 94h, 0e9h, 0c9h

;input: FAC1, hl=single data address
;output: hl=hl or hl+6
;destroy: af,de
SNGORDBL:
	ld	a,(FAC1)
	cp	08h
	ret	nz
	ld	de,0006h
	add	hl,de
	ret


;INP() function
F_INP:
	call	CHKNUM
	call	FTOI1
	ld	b,d		;=0
	ld	c,e
	in	l,(c)
	ld	h,d		;=0
	jp	SETI2


;CDBL() function
F_CDBL:
	call	CHKNUM
	ld	a,(FAC1)
	cp	02h
	jp	z,ITOD		;integer
	cp	05h
	jp	z,STOD		;single
	ret			;double


;CSNG() function
F_CSNG:
	call	CHKNUM
	ld	a,(FAC1)
	cp	02h
	jp	z,ITOS		;integer
	cp	08h
	ld	a,05h
	jp	z,DTOS		;double
	ret			;single


;CINT() function
F_CINT:
	call	CHKNUM
	ld	a,(FAC1)
	cp	02h
	jp	nz,SDTOI	;single or double
	ret			;integer


;DSKF() function
F_DSKF:
	jp	SNERR


;EOF() function
F_EOF:
	call	CHKNUM
	call	FTOI1
	ld	a,e
	or	a
	jp	z,FCERR
	call	GETBUFP
	jp	z,CFERR

	inc	hl
	ld	a,(hl)		;+1: to next block?
	ld	de,000dh
	or	a
	jr	nz,EOFNEXT
	inc	hl
	ld	a,(hl)		;+2: position
	inc	hl
	add	hl,de
	ld	e,a
	add	hl,de
	ld	a,(hl)		;+16+pos
	sub	1ah

;0->ffff, not0->0000
EOFEND
	cp	01h
	sbc	a,a
	ld	h,a
	ld	l,a
	jp	SETI2

EOFNEXT:
	add	hl,de
	ld	a,(hl)		;+14: record number(low)
	inc	hl
	and	(hl)		;+15: record number(high)
	inc	a		;0 if record number=ffff
	jr	EOFEND


;FPOS() function
F_FPOS:
;LOC() function
F_LOC:
;LOF() function
F_LOF:
	jp	SNERR


;POS function
F_POS:
	call	CHKNUM
	ld	a,(DSPXY)
	jp	SETI1


;FAC() function
F_FAC:
	call	CHKNUM
	call	FTOI1
	call	CHKSGN
	jp	c,FCERR
	ex	de,hl
	ld	a,(FAC1)
	cp	02h
	jr	nz,FAC
	ld	a,05h
	ld	(FAC1),a

;input: hl, (FAC1)=type
;output: FAC1
FAC:
	ld	a,h
	or	a
	jp	nz,OVERR
	ld	a,l
	cp	02h
	jp	c,SETPLS1
	push	hl
	dec	l
	call	FAC
	call	CPYFAC
	pop	hl
	call	I2TOF
	jp	MULF


;SUM() function
F_SUM:
	call	CHKNUM
	call	CHKSGN
	jp	c,FCERR
	call	FIX
	call	PUSHF1
	call	INCF1
	call	POPF2
	ld	hl,FAC1+1
	dec	(hl)
	jp	MULF


;FRE() function
F_FRE:
	call	CHKNUM

;only fre(0)?
	call	CHKSGN
	jp	nz,FCERR

	call	CHKFRE
	ex	de,hl
	jp	SETI2


;LPOS() function
F_LPOS:
	call	CHKNUM
	ld	a,(PRNPOS)	;printer x position
	jp	SETI1


;JOY() function
F_JOY:
	call	CHKNUM
	call	FTOI1
	ld	a,e
	cp	08h
	jp	nc,FCERR

	ld	hl,0e008h
	cp	04h
	jr	nc,JOYBTN	;4-7:button

;0-3:X/Y
	inc	a
	ld	b,a
	ld	a,01h
JOYLP1:
	rlca
	djnz	JOYLP1
	ld	(JOYBIT+1),a	;02,04,08,10

	call	CHGVRAM
	call	VBLNK

	ld	b,13h
JOYLP2:
	djnz	JOYLP2

;(10+b)*28 states after v-blank
JOYLP3:
	ld	a,(hl)
	inc	b
JOYBIT:
	and	02h		;02,04,08,10
	jp	z,JOYLP3

	dec	b
	ld	l,b
	ld	h,00h
JOYEND:
	call	CHGDRAM
	ld	(FAC1+1),hl
	ld	a,02h
	ld	(FAC1),a
	ret

JOYBTN:
	sub	03h
	ld	b,a
	ld	a,01h
JOYLP4:
	rlca
	djnz	JOYLP4
	ld	d,a		;02,04,08,10

	call	CHGVRAM
	call	VBLNK

	ld	bc,00ddh
	call	WAIT

;after about 14136 states
	ld	a,(hl)
	and	d		;0=push
	cp	01h
	sbc	a,a
	ld	h,a
	ld	l,a
	jr	JOYEND


;CHR$() function
F_CHR:
	call	CHKLPAR
	xor	a
	ld	(STRDSC1),a
CHRLP:
	call	INT1ARG
	call	ADDSTR1
	ld	hl,(PROGAD)
	ld	a,(hl)
	inc	hl
	cp	','
	jr	z,CHRLP

	dec	hl
	call	CHKRPAR
	jp	INKYEND

;STR$() function
F_STR:
	call	CHKNUM
	call	FTOA
	push	hl
	ld	bc,0101h
	xor	a
	cpir
	sub	c
	pop	hl
	call	MKSTRTMP
	jp	INKYEND


;HEX$() function
F_HEX:
	call	CHKNUM
	call	FTOI
	call	HEXMAIN
	jr	OCTEND

;convert 2-byte integer to hexadeciman string
;input: de
;output: (FAC3), a=length, hl=FAC3
;destroy: f,bc
HEXMAIN:
	ld	hl,FAC3
	ld	c,00h		;length
	ld	a,d
	call	CNVHEX2
	ld	a,e
	call	CNVHEX2
	jr	nz,HEXNZ	;if c>0
	ld	(hl),'0'
	inc	c
HEXNZ:
	ld	a,c
	ld	hl,FAC3
	ret

;input: a=data, c=length, hl=address
;output: c=length, z-flag(1:c=0)
;destroy: af,b
CNVHEX2:
	ld	b,a
	rrca
	rrca
	rrca
	rrca
	call	CNVHEX1
	ld	a,b
;	call	CNVHEX1
;	ret

CNVHEX1:
	and	0fh
	jr	nz,CNVHEXNZ
	inc	c
	dec	c
	ret	z
CNVHEXNZ:
	cp	0ah
	sbc	a,69h
	daa
	ld	(hl),a
	inc	hl
	inc	c
	ret


;OCT$() function
F_OCT:
	call	CHKNUM
	call	FTOI
	call	OCTMAIN
OCTEND:
	call	MKSTRTMP
	jp	INKYEND

;convert 2-byte integer to octal string
;input: de
;output: (FAC3), a=length, hl=FAC3
;destroy: f,bc,de
OCTMAIN:
	ld	hl,FAC3
	ld	bc,0600h	;c=length
	ex	de,hl
	xor	a
	add	hl,hl
	rla
OCTLP:
	or	a
	jr	nz,OCTNZ
	inc	c
	dec	c
	jr	z,OCTZ
OCTNZ:
	add	a,'0'
	ld	(de),a
	inc	de
	inc	c
OCTZ:
	xor	a
	add	hl,hl
	rla
	add	hl,hl
	rla
	add	hl,hl
	rla
	djnz	OCTLP
OCTSETLEN:
	ld	hl,FAC3
	ld	a,c
	or	a
	ret	nz
	inc	a
	ld	(hl),'0'
	ret


;BIN$() function
F_BIN:
	call	CHKNUM
	call	FTOI
	call	BINMAIN
	jr	OCTEND

;convert 2-byte integer to binary string
;input: de
;output: (FAC3), a=length, hl=FAC3
;destroy: f,bc,de
BINMAIN:
	ld	hl,FAC3
	ld	bc,1000h	;c=length
	ex	de,hl
BINLP:
	ld	a,'1'
	add	hl,hl
	jr	c,BINC
	dec	a		;'0'
	inc	c
	dec	c
	jr	z,BINZ
BINC:
	ld	(de),a
	inc	de
	inc	c
BINZ:
	djnz	BINLP
	jr	OCTSETLEN


;MKI$() function
F_MKI:
	call	CHKNUM
	ld	a,(FAC1)
	cp	02h
	call	nz,SDTOI
	ld	a,02h
MKISD:
	ld	hl,FAC1+1
	call	MKSTRTMP
	jp	INKYEND


;MKS$() function
F_MKS:
	call	CHKNUM
	ld	a,(FAC1)
	cp	02h
	call	z,ITOS
	ld	a,05h
	jr	MKISD


;MKD$() function
F_MKD:
	call	CHKNUM
	ld	a,(FAC1)
	cp	02h
	call	z,ITOD
	ld	a,(FAC1)
	cp	05h
	ld	a,00h
	ld	h,a
	ld	l,a
	call	z,ITOSEND
	ld	a,08h
	jr	MKISD


;SPACE$() function
F_SPAC:
	call	CHKNUM
	call	FTOI1
	ld	a,e
	or	a
	jp	z,INKYLEN

	push	af		;
	call	MKSTRTMP
	pop	bc		;
	ld	hl,(STRDSC1+1)
SPACLP:
	ld	(hl),' '
	inc	hl
	djnz	SPACLP
	jp	INKYEND


;ASC() function
F_ASC:
	call	STRARG2
	or	a
	jr	z,SETI1
	ld	a,(de)
	jr	SETI1


;LEN() function
F_LEN:
	call	CHKSTR
SETI1:
	ld	(FAC1+1),a
	xor	a
	ld	(FAC1+2),a
	ld	a,02h
	ld	(FAC1),a
	ret


;VAL() function
F_VAL:
	call	STRARG2
	ex	de,hl
	ld	c,a
	ld	b,00h
	add	hl,bc
	ld	a,(hl)
	push	af
	push	hl
	ld	(hl),00h
;	or	a
	sbc	hl,bc
	call	ATOF
	pop	hl
	pop	af
	ld	(hl),a
	ret


;CVS() function
F_CVS:
	call	STRARG2
	cp	05h
	jp	c,FCERR
	ld	(FAC1),a	;=5
	ex	de,hl
	ld	de,FAC1+1
	ld	bc,0005h
	ldir
	ret


;CVD() function
F_CVD:
	call	STRARG2
	cp	08h
	jp	c,FCERR
	ld	(FAC1),a	;=8
	ex	de,hl
	ld	de,FAC1+1
	ld	bc,0008h
	ldir
	ret


;CVI() function
F_CVI:
	call	STRARG2
	cp	02h
	jp	c,FCERR

	ld	hl,FAC1
	ld	(hl),a		;=2
	inc	hl
	ld	a,(de)
	ld	(hl),a
	inc	de
	inc	hl
	ld	a,(de)
	ld	(hl),a
	ret


;ERR functionn
F_ERR:
	ld	a,(ERRVAL)
	jp	SETI1


;ERL function
F_ERL:
	ld	hl,(ERLVAL)
	ld	a,05h
	ld	(FAC1),a
	ld	b,a		;b-bit7=sign
	ld	a,h
	jp	ITOSPLS


;CSRLIN function
F_CSRL:
	ld	a,(DSPXY+1)
	jp	SETI1


;STRPTR function
F_STRP:
	ld	hl,(STRPTR)
	jp	SETI2


;DTL function
F_DTL:
	ld	hl,0000h
	jp	SETI2


;LEFT$() function
F_LEFT:
	call	ARGSI1
	ld	c,00h
	jr	LRCHKRPAR


;RIGHT$() function
F_RGT:
	call	ARGSI1
	ld	d,a
	sub	e
	jr	nc,RGTNC
	xor	a
RGTNC:
	ld	c,a
	ld	e,d
	ld	a,d
LRCHKRPAR:
	ld	b,a		;
	push	hl
	ld	hl,(PROGAD)
	call	CHKRPAR
	pop	hl
	ld	a,b		;
	jr	MID


;MID$() function
F_MID:
	call	MIDPAR

;input: STRDSC1=string, a=string length, c=2nd parameter-1, e=3rd parameter
MID:
	call	OUTLEN
	ld	hl,(STRDSC1+1)
	ld	b,00h
	add	hl,bc		;+2nd parameter-1
	call	MKSTRTMP
	jp	INKYEND


;MID$() command
C_MID:
	push	hl		;
	call	CHKLPAR
	call	CHKVAR
	pop	hl		;
	ld	a,c
	cp	03h
	jp	nz,TMERR
	call	MIDPAR
	call	OUTLEN
	push	af		;output length
	ld	hl,(STRDSC1+1)
	ld	b,00h
	add	hl,bc
	push	hl		;destination string address

	ld	hl,(PROGAD)
	call	SKIPSP
	cp	EQ_
	jp	nz,SNERR
	inc	hl
	call	STRARG
	pop	hl		;destination string address
	pop	bc		;b=output length

	or	a
	ret	z
	ld	c,a
	ld	a,b
	or	a
	ret	z
MIDLP:
	ld	a,(de)
	ld	(hl),a
	inc	de
	inc	hl
	dec	c
	ret	z
	djnz	MIDLP
	ret


;get output length for mid$
;input: a=string length, c=2nd parameter-1, e=3rd parameter
;output: a=output length
;destroy: f
OUTLEN:
	sub	c
	jr	nc,OUTLENNC
	xor	a
OUTLENNC:
	cp	e
	ret	c
	ld	a,e
	ret


;get MID$() parameters
;output: STRDSC1, a=string length, c=2nd parameter-1, e=3rd parameter
MIDPAR:
	call	ARGSI1
	ld	d,a
	dec	e
	push	de		;d=length, e=2nd parameter-1
	inc	e
	jp	z,FCERR		;if 2nd parameter=0

	ld	e,0ffh
	ld	hl,(PROGAD)
	ld	a,(hl)
	cp	','
	call	z,ARGI1
	call	CHKRPAR
	pop	bc		;b=length, c=2nd parameter-1
	ld	a,b
	ret


;INKEY$ function
F_INKY:
	call	SKIPSP
	ld	(PROGAD),hl
	cp	'('
	jr	nz,INKY
	call	FNCNUM
	call	FTOI1
	add	a,e
	jr	z,INKY0
	dec	a
	jp	nz,FCERR

INKY1:
	call	GETC
;	cp	03h
;	jp	z,STOPSP
	jr	INKYMAIN

INKY0:
	call	GETCH
	push	af
	call	AUTORPT		;for next INKEY$/INKEY$(1)
	pop	af
;	cp	03h
;	jp	z,STOPSP
	or	a
	jr	z,INKYLEN

INKYMAIN:
	ld	b,a		;

;input: b=ascii
;output: FAC1,STRDSC1
MKSTRTMP1:
	ld	a,(SDSCCNT)
	or	a
	ld	hl,(TMPSTR)

	jr	z,MKSTRTMP1Z
	ld	hl,(FREEAD)
MKSTRTMP1Z:
	ld	(STRDSC1+1),hl
	ld	(hl),b
	inc	hl
	call	CHKSTK
	ld	(FREEAD),hl
	ld	a,01h

INKYLEN:
	ld	(STRDSC1),a
INKYEND:
	ld	hl,STRDSC1
	ld	(FAC1+1),hl
	ld	a,03h
	ld	(FAC1),a
	ret

;no parameter
INKY:
	call	GETCH
;	cp	03h
;	jp	z,STOPSP
INKYCMP:
	cp	00h
	ld	(INKYCMP+1),a
	jr	z,INKYZ
	or	a
	jr	nz,INKYMAIN
INKYZ:
	xor	a
	jr	INKYLEN


;INSTR() function
F_INST:
	call	CHKLPAR
	call	ARG
	jp	z,MOERR
	ld	a,(FAC1)
	cp	03h
	ld	de,0000h
	jr	z,INSTZ
	call	FTOI1
	ld	a,e
	or	a
	jp	z,FCERR
	dec	e
	ld	hl,(PROGAD)
	ld	a,(hl)
	cp	','
	jp	nz,SNERR
	inc	hl
	push	de
	call	ARG
	pop	de
INSTZ:
	push	de		;1st parameter-1
	call	STRARG2
	push	af		;2nd length
	push	de		;2nd address
	call	SKIPSP
	cp	','
	jp	nz,SNERR
	inc	hl
	call	STRARG
	push	af		;3rd length
	push	de		;3rd address
	call	SKIPSP
	call	CHKRPAR
	pop	hl		;3rd addres
	pop	bc		;b=3rd length
	pop	de		;2nd address
	pop	af		;2nd length
	ld	c,a
	ex	(sp),hl		;1st parameter-1
	ld	a,l
	add	hl,de		;start address
	pop	de		;3rd address
	push	bc		;c=2nd length
	sub	c		;1st parameter-1-2nd length
	jr	nc,INST0
	ld	c,a		;-(2nd length-1st parameter+1)
;hl=start address, de=3rd address, c=3rd length, b=2nd length-1st parameter+1
INSTLP1:
	push	hl
	push	de
	push	bc
INSTLP2:
	ld	a,c
	or	a
	jr	z,INSTNZ
	ld	a,(de)
	cp	(hl)
	jr	nz,INSTNZ
	inc	de
	inc	hl
	inc	c
	djnz	INSTLP2
;found
	pop	bc
	pop	de
	pop	hl
	pop	hl		;l=2nd length
	ld	a,l
	add	a,c
	inc	a
	jp	I1TOF

INSTNZ:
	pop	bc
	pop	de
	pop	hl
	inc	hl
	inc	c
	jr	nz,INSTLP1
INST0:
	pop	af
	jp	SETZEROI


;HEXCHR$() function
F_HEXC:
	call	CHKLPAR
	call	STRARG
	ld	b,a
	call	CHKRPAR

	xor	a
	ld	(STRDSC1),a

	srl	b
	jr	z,HEXCEND
	jp	c,SNERR

HEXCLP:
	ld	a,(de)
	inc	de
	call	HEX
	jp	nc,SNERR
	rlca
	rlca
	rlca
	rlca
	ld	c,a
	ld	a,(de)
	inc	de
	call	HEX
	jp	nc,SNERR
	or	c

	push	bc
	push	de
	call	ADDSTR1
	pop	de
	pop	bc
	djnz	HEXCLP

HEXCEND:
	jp	INKYEND

;'0'-'F'->00h-0fh
;input: a
;output: a, c-flag(1=ok)
HEX:
	sub	'0'
	cp	'9'-'0'+1
	ret	c
	sub	'A'-'0'
	cp	'F'-'A'+1
	ret	nc
	sub	0f6h		;+10, set c-flag
	ret


;MEM$() function
F_MEM:
	call	MEMPAR
	ex	de,hl
	call	MKSTRTMP
	jp	INKYEND


;MEM$() command
C_MEM:
	call	MEMPAR
	push	af		;2nd parameter
	push	de		;1st parameter
	call	SKIPSP
	cp	EQ_
	jp	nz,SNERR
	inc	hl
	call	STRARG		;a=length
	pop	hl		;1st parameter
	pop	bc		;b=2nd parameter
	ld	c,a		;length
	ld	a,b
	or	a
	ret	z
MEMLP:
	ld	a,c
	or	a
	jr	z,MEMZ
	dec	c
	ld	a,(de)
	inc	de
MEMZ:
	ld	(hl),a
	inc	hl
	djnz	MEMLP
	ret


;get MEM$() parameters
;output: de=1st parameter, a=2nd parameter, hl=program address
MEMPAR:
	call	CHKLPAR
	call	INTARG
	ld	a,(hl)
	cp	','
	jp	nz,SNERR
	push	de
	call	INT1INC		;e
	call	CHKRPAR
	ld	a,e
	pop	de
	ret


;SCRN$() function
F_SCRN:
	call	CHKLPAR
	call	GETXY
	ld	a,(hl)
	cp	','
	jp	nz,SNERR
	inc	hl
	ld	b,e
	push	bc		;c=x,b=y
	call	INT1ARG
	call	CHKRPAR
	ld	a,e

	push	af		;length
	call	MKSTRTMP
	pop	bc		;b=length
	pop	hl		;l=x,h=y
	ld	a,b
	or	a
	jr	z,SCRNEND
	call	XY2AD
	ld	de,(STRDSC1+1)
SCRNLP:
	call	CHGVRAM
	ld	c,(hl)
	inc	hl
	call	CHGDRAM
	ld	a,c
	call	DACN
	ld	(de),a
	inc	de
	djnz	SCRNLP
SCRNEND:
	jp	INKYEND


;VARPTR() function
F_VARP:
	call	CHKLPAR
	cp	'#'
	jr	z,VARPF
	sub	'A'
	cp	'Z'-'A'+1
	jp	nc,MOERR
	inc	hl
	call	GETTYPE
	call	GETVAR
	ex	de,hl
	call	SETI2
	ld	hl,(PROGAD)
	call	SKIPSP
	jp	CHKRPAR

;VARPTR(#n)
VARPF:
	inc	hl
	call	INT1ARG
	call	CHKRPAR
	ld	a,(FILES)
	cp	e		;e=file number
	jp	c,FCERR
	call	GETBUFP
	jp	SETI2


;STRING$() function
F_STRG:
	call	CHKLPAR
	call	INT1ARG
	push	af		;1st parameter
	ld	a,(hl)
	cp	','
	jp	nz,SNERR
	inc	hl
	call	ARG
	jp	z,MOERR
	ld	a,(FAC1)
	cp	03h
	jr	z,STRGSTR

;2nd parameter=numeric
STRGNUM:
	call	FTOI1
	ld	hl,(PROGAD)
	call	CHKRPAR
	pop	af		;1st parameter
	or	a
	jp	z,INKYLEN
	push	de		;2nd parameter
	call	MKSTRTMP
	pop	de		;1st parameter
	ld	a,(STRDSC1)
	ld	b,a
	ld	hl,(STRDSC1+1)
STRGLP1:
	ld	(hl),e
	inc	hl
	djnz	STRGLP1
	jp	INKYEND

;2nd parameter=string
STRGSTR:
	call	STRARG2
	ld	(STRGLP2+1),a	;2nd parameter length
	call	CHKRPAR
	pop	bc		;b=1st parameter
	ld	a,(STRGLP2+1)	;string length
	push	bc		;b=1st parameter
	call	MULINT1
	ld	a,h
	or	a
	jp	nz,LSERR
	ld	a,l
	or	a
	jp	z,INKYLEN
	push	de		;2nd parameter address
	ld	hl,SDSCCNT
	inc	(hl)		;for protection of 2nd parameter
	call	MKSTRTMP
	ld	hl,SDSCCNT
	dec	(hl)
	pop	de		;2nd parameter address
	pop	bc		;b=1st parameter
	ld	a,b		;1st parameter
	ld	hl,(STRDSC1+1)
	ex	de,hl
STRGLP2:
	ld	bc,0000h	;2nd parameter length
	push	hl
	ldir
	pop	hl
	dec	a
	jr	nz,STRGLP2
	jp	INKYEND


;TIME$ command
C_TIME:
	call	SKIPSP
	cp	EQ_
	jp	nz,SNERR
	inc	hl
	call	STRARG
	cp	8
	jp	nz,FCERR

	ex	de,hl
	call	ATOF
	push	hl		;string address
	call	FTOI1
	ld	a,e
	cp	24
	jp	nc,FCERR

	ld	c,00h
	sub	12
	jr	c,TIMEC
	ld	e,a
	inc	c
TIMEC:
	ld	a,c
	ld	(AMPM),a
	ld	a,e
	ld	b,60
	call	MULINT1		;60*hours
	ex	(sp),hl		;push hour*60, pop string address
	ld	a,(hl)
	cp	':'
	jp	nz,SNERR
	inc	hl
	call	ATOF
	push	hl		;string address
	call	FTOI1
	ld	a,e
	cp	60
	jp	nc,FCERR
	pop	hl		;string address
	ex	(sp),hl		;push sring address, pop hours*60
	add	hl,de		;hours*60+minutes
	ld	de,60
	call	MULINT2		;(hours*60+minutes)*60
	ex	(sp),hl		;push (hour*60+minutes)*60, pop string address
	ld	a,(hl)
	cp	':'
	jp	nz,SNERR

	inc	hl
	inc	hl
	inc	hl
	ld	a,(hl)
	push	af		;
	ld	(hl),00h	;termination
	dec	hl
	dec	hl
	call	ATOF
	pop	af		;
	ld	(hl),a

	call	FTOI1
	ld	a,e
	cp	60
	jp	nc,FCERR
	pop	hl		;(hours*60+minues)*60
	add	hl,de		;(hours*60+minutes)*60+seconds
	ex	de,hl
	ld	hl,60*60*12	;counter2 initial value
	or	a
	sbc	hl,de
	ex	de,hl

	call	CHGVRAM
	ld	hl,0e006h
	ld	(hl),e
	ld	(hl),d
	call	CHGDRAM
	ret


;TIME$ function
F_TIME:
	call	RDTIMER
	ld	hl,60*60*12	;counter2 initial value
	or	a
	sbc	hl,de

	ld	de,FAC3
	ld	a,(AMPM)
	or	a
	jr	z,TIMEAM
	ld	bc,60*60*2
	add	hl,bc
TIMEAM:
	push	af		;(AMPM)
	ld	bc,60*60*10
	call	DIVTIM
	dec	de
	ld	a,(de)
	pop	bc		;(AMPM)
	add	a,b
	ld	(de),a
	inc	de

	ld	bc,60*60
	call	DIVTIM
	ld	a,':'
	ld	(de),a
	inc	de

	ld	bc,60*10
	call	DIVTIM
	ld	bc,60
	call	DIVTIM
	ld	a,':'
	ld	(de),a
	inc	de

	ld	bc,10
	call	DIVTIM
	ld	a,l
	add	a,'0'
	ld	(de),a

	ld	a,08
	ld	hl,FAC3
	call	MKSTRTMP
	jp	INKYEND


;hl/de for TIME$
;input: hl=dividend, bc=devisor, de=address
;output: hl=remainder, de=next address
;destroy: af
DIVTIM:
	xor	a
DIVTLP:
	sbc	hl,bc
	inc	a
	jr	nc,DIVTLP
	add	hl,bc
	add	a,'0'-1
	ld	(de),a
	inc	de
	ret


;FN() function
F_FN:
	call	CHKVAR
	ld	a,c
	ld	(FNTYP+1),a
	or	a		;reset z-flag
	set	6,c		;fn mark
	call	SRCHVAR
	jp	nc,UFERR
	ld	(PROGAD),hl

	ex	de,hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	ex	de,hl
	ld	a,(hl)
	cp	'('
	jr	z,FNPAR

;difinition has no parameters
FNNOPAR:
	cp	EQ_
	jp	nz,SNERR
	inc	hl
	ld	de,(PROGAD)
	push	de
	call	ARG
	pop	hl
	ld	(PROGAD),hl
	ld	a,(FNTYP+1)
	ld	c,a
	jp	CNVTYP1

;with parameters
FNPAR:
	push	af		;'('

	call	SKIPSPINC
	push	hl		;fn parameter definition

	ld	hl,(PROGAD)
	call	SKIPSP
	cp	'('
	jp	nz,SNERR

FNLP1:
	inc	hl
	call	ARG		;caller
	ex	(sp),hl

	call	CHKGETV		;fn parameter definition
	ld	a,(FAC1)	;caller
	cp	03h
	jr	nz,FNNUM

;string
FNSTR:
	ld	a,c
	cp	03h
	jp	nz,TMERR
	pop	bc		;caller address

	call	PUSHF1		;dummy, type=string
	push	de		;formal parameter value address

	push	bc		;caller address
	ld	hl,STRDSC1
	ld	b,03h
FNLP2:
	ld	a,(de)
	ld	c,(hl)
	ld	(hl),a
	ld	a,c
	ld	(de),a
	inc	de
	inc	hl
	djnz	FNLP2

	call	COPYSTR
	jr	FNCHKCM

;number
FNNUM:
	ld	a,c
	cp	03h
	jp	z,TMERR
	push	de		;formal parameter value address
	call	CNVTYP1
	pop	de		;formal parameter value address

	push	de		;formal parameter value address
	ld	hl,FAC2
	ld	(hl),c		;type
	inc	hl
	ex	de,hl
	ld	b,00h
	ldir
	pop	de		;formal parameter value address

	pop	bc		;caller address

	call	PUSHF2		;formal parameter original value
	push	de		;formal parameter value address

	push	bc		;caller
	ld	hl,FAC1
	ld	c,(hl)
	inc	hl
	ld	b,00h
	ldir

FNCHKCM:
	pop	hl		;caller
	ld	a,(hl)
	cp	','
	jr	nz,FNRSLT
	push	hl		;caller
	ld	hl,(PROGAD)	;fn parameter definition
	call	SKIPSP
	cp	','
	jp	nz,SNERR
	call	SKIPSPINC
	ex	(sp),hl
	jr	FNLP1

FNRSLT:
	ld	de,(PROGAD)	;fn parameter definition
	call	CHKRPAR
	push	hl		;caller
	ex	de,hl		;fn parameter definition
	call	SKIPSP
	call	CHKRPAR
	call	SKIPSP
	cp	EQ_
	jp	nz,SNERR
	inc	hl
	call	ARG		;result
	pop	hl		;caller
	ld	(PROGAD),hl
FNTYP:
	ld	c,05h		;type
	call	CNVTYP1

FNLP3:
	pop	de		;formal parameter value address
	ld	a,d
	cp	'('
	ret	z
	call	POPF2
	ld	a,(FAC2)
	cp	03h
	jr	z,FNSTRBK
	ld	c,a
	ld	b,00h
	ld	hl,FAC2+1
	ldir
	jr	FNLP3

;string
FNSTRBK:
	ld	a,(STRDSC1)
	ld	hl,(STRDSC1+1)
	push	af
	push	hl

	call	BACKSTR
	ld	hl,STRDSC1
	ld	bc,0003h
	ldir

	pop	hl
	pop	af
	ld	(STRDSC1),a
	ld	(STRDSC1+1),hl
	jr	FNLP3


;USR function
F_USR:
	call	SKIPSP
	ld	bc,0000h
	cp	0ah+1
	jr	nc,USR0
	inc	hl
	dec	a
	add	a,a
	ld	c,a
USR0:
	ex	de,hl
	ld	hl,USR0AD
	add	hl,bc
	ld	c,(hl)
	inc	hl
	ld	b,(hl)
	push	bc
	ex	de,hl
	call	CHKLPAR
	call	ARG
	jp	z,MOERR
	call	CHKRPAR
	ld	a,(FAC1)
	ld	hl,FAC1+1
	ld	de,(FAC1+1)
	ret


;ATTR$() function
F_ATTR:
	jp	SNERR


;CHARACTER$() function
F_CHAR:
	call	CHKLPAR
	call	GETXY
	call	CHKRPAR
	ld	l,c
	ld	h,e
	call	XY2AD
	call	CHGVRAM
	ld	b,(hl)
	call	CHGDRAM
	ld	a,b
	call	DACN
	jp	INKYMAIN


;EQV operator
O_EQV:
	call	F12TOI2
	ld	a,h
	xor	d
	cpl
	ld	h,a
	ld	a,l
	xor	e
	cpl
	ld	l,a
	jr	SETI2

;IMP operator
O_IMP:
	call	F12TOI2
	ld	a,h
	cpl
	or	d
	ld	h,a
	ld	a,l
	cpl
	or	e
	ld	l,a
	jr	SETI2

;XOR operator
O_XOR:
	call	F12TOI2
	ld	a,h
	xor	d
	ld	h,a
	ld	a,l
	xor	e
	ld	l,a
	jr	SETI2

;OR operator
O_OR:
	call	F12TOI2
	ld	a,h
	or	d
	ld	h,a
	ld	a,l
	or	e
	ld	l,a
	jr	SETI2

;AND operator
O_AND:
	call	F12TOI2
	ld	a,h
	and	d
	ld	h,a
	ld	a,l
	and	e
	ld	l,a
	jr	SETI2

;NOT operator
O_NOT:
	call	CHKNUM
	call	EXFAC
	call	FTOI2
	ld	a,d
	cpl
	ld	h,a
	ld	a,e
	cpl
	ld	l,a
;	jr	SETI2

;input: hl=integer (signed)
;output: FAC1
;destroy: hl
SETI2:
	ld	a,02h
	ld	(FAC1),a
	ld	(FAC1+1),hl
	ld	hl,0000h
	ld	(FAC1+3),hl
	ld	(FAC1+5),hl
	ld	(FAC1+7),hl
	ret

F12TOI2:
	ld	a,(FAC1)
	cp	03h
	jp	z,TMERR
	call	FTOI2
	push	de		;
	call	EXFAC
	ld	a,(FAC1)
	cp	03h
	jp	z,TMERR
	call	FTOI2
	pop	hl		;
	ret


;convert float to 1-byte integer [0,255]
;input: FAC1
;output: de, a(0=ok)
;destroy: f,bc,hl
FTOI1:
	call	FTOI2
	ld	a,d
	or	a
	ret	z
	jp	FCERR


;convert float to 2-byte integer for USR() [-32768,32767]
;input: FAC1
;output: de
;destroy: af,bc,hl
FTOI2:
	call	FTOI
	ld	a,d
	or	e
	ret	z		;de=0 (for negative value rounding)
	ld	a,(FAC1+2)	;sign(float) or higher byte(integer)
	xor	d
	ret	p
	jp	OVERR


;convert float to 2-byte integer [-65535,65535]
;input: FAC1
;output: de
;destroy: af,bc,hl
FTOI:
	ld	de,(FAC1+1)
	ld	a,(FAC1)
	cp	02h
	ret	z

	ld	de,0000h
	ld	a,(FAC1+1)	;exponent
	or	a
	ret	z
	sub	91h
	jp	nc,OVERR

	neg
	ld	b,a
	ld	hl,(FAC1+2)
	ld	c,l		;sign
	ld	l,h
	ld	h,c
	set	7,h
	ld	a,(FAC1+4)	;fraction
	dec	b
	jr	z,FTOIZ
FTOILP:
	srl	h
	rr	l
	rra
	djnz	FTOILP

FTOIZ:
	rlca
	jr	nc,FTOINC
	inc	hl
	ld	a,h
	or	l
	jp	z,OVERR

FTOINC:
	ex	de,hl
	bit	7,c		;sign
	ret	z

;	neg
	sbc	hl,de
	ex	de,hl
	ret


O_GTLT:
O_LTGT:
	call	CMPSTRNUM
	jr	nz,SETTRUE
	jr	SETFALSE

O_EQLT:
O_LTEQ:
	call	CMPSTRNUM
	jr	c,SETTRUE
	jr	z,SETTRUE
	jr	SETFALSE

O_EQGT:
O_GTEQ:
	call	CMPSTRNUM
	jr	nc,SETTRUE
	jr	SETFALSE

O_EQ:
	call	CMPSTRNUM
	jr	z,SETTRUE
	jr	SETFALSE

O_GT:
	call	CMPSTRNUM
	jr	z,SETFALSE
	jr	nc,SETTRUE
	jr	SETFALSE

O_LT:
	call	CMPSTRNUM
	jr	c,SETTRUE
;	jr	SETFALSE

SETZEROI:
SETFALSE:
	ld	hl,ZEROI
SETF1:
	ld	de,FAC1
SETF:
	ld	b,00h
	ld	c,(hl)
	inc	c
	ldir
	ret

SETTRUE:
SETMNS1I:
	ld	hl,MNSONEI
	jr	SETF1

SETZEROD:
	ld	hl,ZEROD
	jr	SETF1


SETZERO:
	ld	a,(FAC1)
	ld	hl,ZEROI+1
	cp	02h
	jr	z,SETF1TYP
	ld	hl,ZEROD+1
	jr	SETF1TYP

SETMNS1:
	ld	a,(FAC1)	;type
	ld	hl,MNSONEI+1
	cp	02h
	jr	z,SETF1TYP
	ld	hl,MNSONED+1
	jr	SETF1TYP

SETPLS1:
	ld	a,(FAC1)	;type
	ld	hl,PLSONEI+1
	cp	02h
	jr	z,SETF1TYP
	ld	hl,PLSONED+1

;input: a=type, hl=data address(no type)
SETF1TYP:
	ld	de,FAC1+1
	ld	b,00h
	ld	c,a
	ldir
	ret

SETF2:
	ld	de,FAC2
	jr	SETF

;integer 0
ZEROI:
	db	02h, 00h, 00h
;double 0
ZEROD:
	db	08h, 00h, 00h, 00h, 00h, 00h, 00h, 00h, 00h

;integer -1
MNSONEI:
	db	02h, 0ffh, 0ffh
;double -1
MNSONED:
	db	08h, 81h, 80h, 00h, 00h, 00h, 00h, 00h, 00h

;integer +1
PLSONEI:
	db	02h, 01h, 00h


;+ operator
O_PLS:
	ld	a,(FAC1)
	cp	03h
	ld	a,(FAC2)
	jr	z,PLSSTR
	cp	03h
	jp	nz,ADDF
	jp	TMERR

PLSSTR:
	cp	03h
	jp	z,ADDSTR
	jp	TMERR


;- operator
O_MNS:
	call	CHKNUM2
	jp	SUBF


;MOD operator
O_MOD:
	call	CHKNUM2
	call	PUSHF1
	call	PUSHF2
	call	DIVF
	call	FIX
	call	POPF2
	call	MULF
	call	CPYFAC
	call	POPF1
	jp	SUBF


;yen mark operator
O_YEN:
	call	CHKNUM2
	call	DIVF
	jp	FIX


;/ operator
O_DIV:
	call	CHKNUM2
	jp	DIVF


;* operator
O_MUL:
	call	CHKNUM2
	jp	MULF

;^ operator
O_POW:
	call	CHKNUM2
	call	CNVTYP2

	call	PUSHF1
	call	PUSHF2
	call	FRAC
	ld	a,(FAC1+1)	;exponent
	or	a
	jr	nz,POWNZ

;integral number
	call	POPF1
	call	PUSHF1
	call	FRAC
	ld	a,(FAC1+1)	;exponent
	or	a

POWNZ:
	push	af		;z-flag=1 if integral^integral
	pop	bc		;z-flag
	call	POPF2
	call	POPF1
	push	bc		;z-flag

	ld	hl,(FAC2+1)
	ld	a,h		;sign
	rlca
	jr	nc,POWNC	;y>0
	pop	bc		;cancel stack
	or	a		;reset z-flag
	push	af		;z-flag=0

POWNC:
	ld	a,l		;exponent
	or	a
	jr	z,POWONE	;x^0
	call	CHKSGNSD
	jr	z,POWZERO
	jr	c,POWNEG

;(positive)^y
	call	POWPOS

POWEND:
	pop	af		;z-flag
	ret	nz

;round to the nearest integral number
	call	CPYFAC
	call	INT		;round toward minus infinity
	call	PUSHF1
	call	SUBF		;(-1,0]
	ld	a,(FAC1+1)
	ld	b,a
	call	POPF1
	ld	a,b		;exponent
;	or	a
;	ret	z		;+-0
;	cp	80h
;	ret	c		;(-0.5,0)
	rlca
	ret	nc		;(0.5,0]
	jp	INCF1		;(-1.0,-0.5]

;x^y=exp(ylog(x))
POWPOS:
	call	PUSHF2		;y
	call	LOG		;log(x)
	call	POPF2		;y
	call	MULF		;ylog(x)
	jp	EXP

;x^0=1
POWONE:
	call	SETPLS1
	jr	POWEND

;0^y=0 (y>0)
POWZERO:
	ld	a,h		;sign
	rlca
	jr	nc,POWEND	;0^(positive)
	jp	DIV0ERR		;0^(negative)

;(negative)^y
POWNEG:
	ld	a,(FAC2)	;5 or 8
	add	a,a		;*2
	add	a,a		;*4
	add	a,a		;*8
	add	a,80h-8		;a0h or b8h
	cp	l		;exponent
	jp	c,FCERR

	call	PUSHF1
	call	PUSHF2
	call	POPF1

	call	INT
	call	CMPF
	jp	nz,FCERR	;y=not integer
	call	FTOINT
	ld	hl,FAC1
	ld	e,(hl)
	ld	d,00h
	add	hl,de		;hl=FAC1+5or8
	ld	b,(hl)		;even/odd

	call	POPF1
	push	bc		;
	call	ABS
	call	POWPOS
	pop	af		;
	rrca
	call	c,NEGABSNZ	;y=odd
	jr	POWEND

;FAC1=-abs(FAC1) (float)
;destroy: af,hl
NEGABS:
	ld	a,(FAC1+1)	;exponent
	or	a
	ret	z

;FAC1=-abs(FAC1) without checking zero (float)
;destroy: f,hl
NEGABSNZ:
	ld	hl,FAC1+2
	set	7,(hl)		;sign
	ret


;convert FAC1 to string
;input: FAC1
;output: FAC3, hl=FAC3
;destoy: FAC1,FAC2,af,bc,de
FTOA:
	ld	a,(FAC1)
	cp	02h
	jp	z,ITOA

	ld	b,a		;05 or 08
	ld	hl,FAC3
	ld	(hl),' '
	ld	a,(FAC1+1)	;exponent
	or	a
	jp	z,FTOAZERO	;FAC1=0
	ld	a,(FAC1+2)
	rlca
	jr	nc,FTOAPLS	;FAC1>0
	ld	(hl),'-'
FTOAPLS:
	or	a
	rra
	ld	(FAC1+2),a	;bit7=sign
	ld	hl,E8S
	ld	a,b		;05 or 08
	rrca
	jr	c,FTOAC1	;if single
	ld	hl,E16D
FTOAC1:
	call	SETCMPF
	jp	nc,LARGE	;FAC1 >= 1e8
	ld	hl,EM2S
	call	SNGORDBL
	call	SETCMPF
	jp	c,SMALL

FTOA2:
	ld	b,09h
	ld	a,(FAC1)
	rrca
	jr	c,FTOALP1	;if single
	ld	b,11h
FTOALP1:
	push	bc
	ld	hl,E7S
	ld	a,(FAC1)	;05or 08
	rrca
	jr	c,FTOAC3	;if single
	ld	hl,E15D
FTOAC3:
	call	SETCMPF
	jr	nc,FTOAOK
	call	MUL10
	pop	bc
	djnz	FTOALP1
	push	bc
FTOAOK:
	ld	hl,FAC1
	ld	a,(hl)		;05 or 08
	rrca			;
	inc	hl
	ld	a,(hl)		;exponent
	jr	c,FTOAC4	;if single
	sub	0afh		;exponent-(80h+8*3))+1
	ld	b,a
	ld	a,(FAC1+8)
	jr	FTOALP2
FTOAC4:
	sub	97h		;exponent-(80h+8*6)+1
	ld	b,a
	ld	a,(FAC1+5)

FTOALP2:
	rlca
	djnz	FTOALP2
	call	c,INCF1		;round up
	call	FTOINT

	ld	hl,INTE7
	ld	a,(FAC1)	;05 or 08
	rrca
	jr	c,FTOAC5	;if single
	ld	hl,INTE15
FTOAC5:
	ld	de,FAC3+1
	pop	bc
	ld	c,b		;
	dec	b
FTOALP3:
	dec	b
	jp	m,FTOADOT
	call	DIVINT
	add	a,'0'
	ld	(de),a
	inc	de
	jr	FTOALP3

FTOADOT:
	call	CHKINT0
	jr	z,FTOAEND
	ld	a,'.'
	ld	(de),a
	inc	de
	ld	a,c		;
	or	a
FTOALP4:
	call	nz,DIVINT
	add	a,'0'
	ld	(de),a
	inc	de
	call	CHKINT0
	jr	nz,FTOALP4

FTOAEND:
	ld	(de),a		;a=0
	ld	hl,FAC3
	ret

SMALL:
	ld	bc,002dh	;c='-'
SMALLLP:
	inc	b
	push	bc
	call	MUL10
	ld	hl,E0S
	call	SNGORDBL
	call	SETCMPF
	pop	bc
	jr	c,SMALLLP
	jr	LARSMA

LARGE:
	ld	bc,002bh	;c='+'
LARGELP:
	inc	b
	push	bc
	call	DIV10
	ld	hl,E2S
	call	SNGORDBL
	call	SETCMPF
	pop	bc
	jr	nc,LARGELP

LARSMA:
	push	bc
	call	FTOA2

LARLP2:
	inc	hl
	ld	a,(hl)
	or	a
	jr	nz,LARLP2

	ld	a,(FAC1)
	and	01h
	add	a,'D'		;'D' or 'E'
	ld	(hl),a

	inc	hl
	pop	bc		;
	ld	a,b
	ld	(hl),c
	inc	hl
	ld	bc,2f0ah	;b='0'-1, c=10
LARLP3:
	inc	b
	sub	c
	jr	nc,LARLP3
	add	a,c
	ld	(hl),b

FTOAZERO:
	inc	hl
	add	a,'0'
	ld	(hl),a
	inc	hl
	ex	de,hl
	xor	a
	jr	FTOAEND


;convert float to 4or7-byte integer (>=0)
;(round toward zero)
;input: FAC1
;output: FAC1 (integer)
;destroy: af,bc,d,hl
FTOINT:
	ld	hl,FAC1
	ld	a,(hl)
	ld	d,a		;;
	add	a,a		;*2
	add	a,a		;*4
	add	a,a		;*8
	add	a,78h		;81h+1fh or 81h+37h
	inc	hl		;exponent
	sub	(hl)		;
	ld	c,a
	inc	hl
	set	7,(hl)		;FAC1+2
	ret	z		;

FTOINTLP1:
	ld	hl,FAC1+2
	ld	b,d		;;
	or	a
FTOINTLP2:
	rr	(hl)
	inc	hl
	djnz	FTOINTLP2

	dec	c
	jr	nz,FTOINTLP1
	ret


;convert 4or7-byte integer to float (>=0)
;input: FAC1 (integer)
;output: FAC1
;destroy: af,bc,de,hl
INTTOF:
	call	CHKINT0
	jp	z,SETZERO

	ld	hl,FAC1
	ld	a,(hl)		;type
	ld	e,a
	ld	d,00h
	add	hl,de		;FAC1+5or8

	dec	a
	ld	b,a		;4 or 7
	dec	b		;3 or 6
	add	a,a
	add	a,a
	add	a,a
	ld	c,a		;20h or 38h
	ld	a,(FAC1+2)
INTTOFLP1:
	add	a,80h
	jr	c,INTTOFEND
	push	hl
	push	bc
	or	a
INTTOFLP2:
	rl	(hl)
	dec	hl
	djnz	INTTOFLP2
	rla
	pop	bc
	pop	hl
	dec	c
	jr	nz,INTTOFLP1

INTTOFEND:
	ld	(FAC1+2),a

	ld	a,c
	add	a,80h
	ld	(FAC1+1),a
	ret


;a=FAC1(int4or7)/(hl)<=9, FAC1%=(hl), hl+=4
;input: FAC1(integer), hl=division factor (4or7-byte int) address
;output: a=FAC1/(hl4or7), FAC1=FAC1%(hl4or7), hl=hl+4or7, a=[0,9]
;destroy: f,FAC2
DIVINT:
	push	bc
	push	de
;copy (hl)->(FAC2+2)
	ld	de,FAC2+2
	ld	b,00h
	ld	a,(FAC1)
	dec	a
	ld	c,a		;4 or 7
	ldir

	push	hl

	dec	de
	ld	hl,FAC1-FAC2
	add	hl,de
	ex	de,hl

	ld	b,a		;4 or 7
	or	a

;	ld	c,00h
DIVINTLP1:
	push	hl		;hl=FAC2+5 or FAC2+8
	push	de		;de=FAC1+5 or FAC2+8
	push	bc

;FAC1=FAC1-FAC2
DIVINTLP2:
	ld	a,(de)
	sbc	a,(hl)
	ld	(de),a
	dec	de
	dec	hl
	djnz	DIVINTLP2

	pop	bc
	pop	de		;de=FAC1+5 or FAC1+8
	pop	hl		;hl=FAC2+5 or FAC1+8

	inc	c
	jr	nc,DIVINTLP1
	dec	c

;FAC1=FAC1+FAC2
	or	a
DIVINTLP3:
	ld	a,(de)
	adc	a,(hl)
	ld	(de),a
	dec	de
	dec	hl
	djnz	DIVINTLP3

	ld	a,c		;result
	pop	hl
	pop	de
	pop	bc
	ret


;subtract 4or7-byte integer from 4or7-byte integer
;input: FAC1(integer),FAC2(integer)
;output: FAC1(integer),f
;destroy: af,b,de,hl
SUBINT:
	ld	de,FAC1
	ld	a,(de)		;type
	ld	b,a
	dec	b		;4 or 7
	add	a,e
	ld	e,a		;no carry
	ld	hl,FAC2-FAC1
	add	hl,de		;de=FAC1+5or8, hl=FAC2+5or8
SUBINTLP:
	ld	a,(de)
	sbc	a,(hl)
	ld	(de),a
	dec	de
	dec	hl
	djnz	SUBINTLP
	ret


;negate 4or7-byte integer
;input: FAC2(integer)
;output: FAC2(integer), c-flag, z-flag
;destroy: af,c,hl
NEGINT2:
	ld	a,(FAC2)	;type
	ld	hl,FAC2+5
	sub	05h
	ld	c,a
	jr	z,NEGINT2Z	;single

;double
	ld	hl,FAC2+8
	xor	a
	ld	c,a
	sub	(hl)
	ld	(hl),a
	dec	hl
	ld	a,c		;=0
	sbc	a,(hl)
	ld	(hl),a
	dec	hl
	ld	a,c		;=0
	sbc	a,(hl)
	ld	(hl),a
	dec	hl
	ld	a,c		;=0

NEGINT2Z:
	sbc	a,(hl)
	ld	(hl),a
	dec	hl
	ld	a,c		;=0
	sbc	a,(hl)
	ld	(hl),a
	dec	hl
	ld	a,c		;=0
	sbc	a,(hl)
	ld	(hl),a
	dec	hl
	ld	a,c		;=0
	sbc	a,(hl)
	ld	(hl),a
	ret


;compare 4or7-byte integer and 4or7-byte integer
;input: FAC1(integer), FAC2(integer)
;output: c-flag,z-flag
;destroy: af,b,de,hl
CMPINT:
	ld	a,(FAC1)
	ld	b,a
	dec	b		;4 or 7
	ld	de,FAC1+2
	ld	hl,FAC2+2
CMPINTLP:
	ld	a,(de)
	cp	(hl)
	ret	nz
	inc	de
	inc	hl
	djnz	CMPINTLP
	ret


;shift left arithmetic for 4or7-byte integer
;input: FAC1(integer)
;output: FAC1,c-flag
;destroy: af,hl
SLAINT:
	ld	hl,FAC1+5
	ld	a,(FAC1)	;type
	cp	05h
	jr	z,SLAINTZ	;4-byte integer
;7-byte integer
	ld	hl,FAC1+8
	sla	(hl)
	dec	hl
	rl	(hl)
	dec	hl
	rl	(hl)
	dec	hl
SLAINTZ:
	rl	(hl)
	dec	hl
	rl	(hl)
	dec	hl
	rl	(hl)
	dec	hl
	rl	(hl)
	ret


;4or7-byte integer=0?
;output: a(4or7 bytes OR),z-flag
;destroy: af,b
CHKINT0:
	push	hl
	ld	hl,FAC1
	ld	b,(hl)
	dec	b		;4 or 7
	inc	hl
	inc	hl
	xor	a
CHKINT0LP:
	or	(hl)
	inc	hl
	djnz	CHKINT0LP
	pop	hl
	ret


;convert integer to string [-32768,+32767]
;input: FAC1
;output: (FAC3), hl=FAC3
;destroy: af,bc,de
ITOA:
	ld	bc,FAC3
	ld	de,(FAC1+1)
	ld	a,d
	rlca
	ld	a,' '
	jr	nc,ITOAPLS
	xor	a
	sub	e
	ld	e,a
	sbc	a,d
	sub	e
	ld	d,a
	ld	a,'-'
ITOAPLS:
	ld	(bc),a
	inc	bc
;	jr	I2TOA

;convert integer to string [0,+65535]
;input: bc=FAC3, de=value
;output: (FAC3), hl=FAC3
;destroy: af,bc,de
I2TOA:
	ld	hl,DIGIT
	xor	a
	push	af		;counter
ITOALP1:
	ld	a,(hl)
	or	a
	jr	z,ITOAEND
	inc	hl
	push	hl		;DIGIT
	ld	h,(hl)
	ld	l,a

	ex	de,hl
	xor	a
ITOALP2:
	sbc	hl,de		;c-flag=0
	jr	c,ITOAC
	inc	a
	jr	ITOALP2
ITOAC:
	add	hl,de
	ex	de,hl
	pop	hl		;DIGIT
	inc	hl

	or	a
	jr	nz,ITOAOK
	ex	(sp),hl		;h=counter
	inc	h
	dec	h
	ex	(sp),hl
	jr	z,ITOALP1

ITOAOK:
	add	a,'0'
	ld	(bc),a
	inc	bc
	ex	(sp),hl		;h=counter
	inc	h
	ex	(sp),hl
	jr	ITOALP1

ITOAEND:
	pop	af
	ld	a,e
ITOAZ:
	add	a,'0'
	ld	(bc),a
	inc	bc
	xor	a
	ld	(bc),a
	ld	hl,FAC3
	ret

DIGIT:
	dw	10000, 1000, 100, 10
	db	00h


;9999999999999999.5
E16D:
	db	08h, 0b6h, 0eh, 1bh, 0c9h, 0bfh, 03h, 0ffh, 0feh

;999999999999999.95
E15D:
;	db	08h, 0b2h, 63h, 5fh, 0a9h, 31h, 9fh, 0ffh, 0fdh
	db	08h, 0b2h, 63h, 5fh, 0a9h, 31h, 9fh, 0ffh, 0fch

;99999999.5
E8S:
	db	05h, 9bh, 3eh, 0bch, 1fh, 0f0h

;9999999.95
E7S:
	db	05h, 98h, 18h, 96h, 7fh, 0f3h

;9.99999995
E2S:
	db	05h, 84h, 1fh, 0ffh, 0ffh, 0f3h

;9.9999999999999995
E2D:
	db	08h, 84h, 1fh, 0ffh, 0ffh, 0ffh, 0ffh, 0ffh, 0feh

;0.999999995
E0S:
	db	05h, 80h, 7fh, 0ffh, 0ffh, 0ebh

;0.99999999999999995
E0D:
	db	08h, 80h, 7fh, 0ffh, 0ffh, 0ffh, 0ffh, 0ffh, 0fch

;0.00999999995
EM2S:
	db	05h, 7ah, 23h, 0d7h, 0ah, 30h

;0.0099999999999999995
EM2D:
	db	08h, 7ah, 23h, 0d7h, 0ah, 3dh, 70h, 0a3h, 0d5h


INTE15:
	db	03h, 8dh, 7eh,0a4h,0c6h, 80h, 00h	;10^15
	db	00h, 5ah,0f3h, 10h, 7ah, 40h, 00h	;10^14
	db	00h, 09h, 18h, 4eh, 72h,0a0h, 00h	;10^13
	db	00h, 00h,0e8h,0d4h,0a5h, 10h, 00h	;10^12
	db	00h, 00h, 17h, 48h, 76h,0e8h, 00h	;10^11
	db	00h, 00h, 02h, 54h, 0bh,0e4h, 00h	;10^10
	db	00h, 00h, 00h, 3bh, 9ah,0cah, 00h	;10^9
	db	00h, 00h, 00h, 05h,0f5h,0e1h, 00h	;10^8
	db	00h, 00h, 00h, 00h, 98h, 96h, 80h	;10^7
	db	00h, 00h, 00h, 00h, 0fh, 42h, 40h	;10^6
	db	00h, 00h, 00h, 00h, 01h, 86h,0a0h	;10^5
	db	00h, 00h, 00h, 00h, 00h, 27h, 10h	;10^4
	db	00h, 00h, 00h, 00h, 00h, 03h,0e8h	;10^3
	db	00h, 00h, 00h, 00h, 00h, 00h, 64h	;10^2
	db	00h, 00h, 00h, 00h, 00h, 00h, 0ah	;10^1
	db	00h, 00h, 00h, 00h, 00h, 00h, 01h	;10^0

INTE7:
	db	00h, 98h, 96h, 80h	;10^7
	db	00h, 0fh, 42h, 40h	;10^6
	db	00h, 01h, 86h,0a0h	;10^5
	db	00h, 00h, 27h, 10h	;10^4
	db	00h, 00h, 03h,0e8h	;10^3
	db	00h, 00h, 00h, 64h	;10^2
	db	00h, 00h, 00h, 0ah	;10^1
	db	00h, 00h, 00h, 01h	;10^0


;0.1
EM1S:
	db	05h, 7dh, 4ch, 0cch, 0cch, 0cdh

;0.1
EM1D:
	db	08h, 7dh, 4ch, 0cch, 0cch, 0cch, 0cch, 0cch, 0cdh

;10
TENS:
	db	05h, 84h, 20h, 00h, 00h, 00h

TEND:
	db	08h, 84h, 20h, 00h, 00h, 00h, 00h, 00h, 00h


;analyze an argument
;input: hl=program address
;output: hl=next address, FAC1=numerical value or pointer to string descriptor
;        z-flag(1=no argument)
;destroy: FAC2,af,bc,de
ARG:
	xor	a
	ld	e,a
	push	de		;e=dummy operator
	dec	a
	ld	(FAC1),a	;ff=unknown
	ld	(OPRTR),a
ARGLP1:
	push	hl
	call	CHKFRE
	pop	hl
ARGLP2:
	call	SKIPSP
	ld	b,a		;
	inc	hl
	ld	(PROGAD),hl

;command, function, operator
	or	a
	jp	m,ARGCMD
	jr	z,ARGTAIL

;previous argument without operator?
	ld	a,(OPRTR)
	or	a
	jr	z,ARGTAIL
;[A-Z]
	ld	a,b		;
	sub	'A'
	cp	'Z'-'A'+1
	jr	c,ARGVAR	;'A'-'Z'
;number
	ld	a,b		;
	cp	18h+1
	jr	c,ARGNUM	;01h-18h
;others
	cp	22h		;double quotation mark
	jr	z,ARGSTR
	cp	'('
	jr	z,PARL
ARGTAIL:
	dec	hl
	ld	(PROGAD),hl
	ld	d,00h		;operator
	jp	OPR

PARL:
	call	CHKCLN
	jp	z,MOERR
	call	ARG
	ld	a,(FAC1)
	inc	a
	jp	z,SNERR
	call	CHKRPAR
	jr	ARGLP1

ARGNUM:
	dec	hl
	call	ICODENUM
	xor	a
	ld	(OPRTR),a
	jp	ARGLP2

ARGVAR:
	call	GETTYPE
	jr	z,ARGARR	;array
	call	SRCHVAR
	jr	nc,ARGVC	;not found
;found
	ld	(PROGAD),hl
ARGVNC:
	ld	a,c
	cp	03h
	jr	z,ARGVSTR

;numerical variable
ARGVNUM:
	xor	a
	ld	(OPRTR),a
	ld	hl,FAC1
	ld	(hl),c
	inc	hl
	ex	de,hl
	ld	b,a		;=0
	ldir
	jp	ARGNEXT2

;array
ARGARR:
	call	GETARR
	jr	ARGVNC

;not found
ARGVC:
	ld	de,ZEROD+1
	jr	ARGVNC

;string variable
ARGVSTR:
	call	ARGSCHK
	ld	a,(de)
	ld	(STRDSC1),a
	inc	de
	ld	a,(de)
	ld	l,a
	inc	de
	ld	a,(de)
	ld	h,a
	ld	de,(STRPTR)
	add	hl,de
	ld	(STRDSC1+1),hl
	xor	a
	ld	(OPRTR),a
	jp	ARGNEXT2

ARGSTR:
	call	ARGSCHK
	xor	a
	ld	(OPRTR),a

	ld	d,h		;
	ld	e,l

	ld	b,a		;=0
ARGSLP:
	ld	a,(hl)
	or	a
	jr	z,ARGSOK
	inc	hl
	cp	22h		;double quotation mark
	jr	z,ARGSOK
	inc	b
	jp	z,LSERR
	jr	ARGSLP
ARGSOK:
	ld	a,b
	ld	(STRDSC1),a
	ld	(STRDSC1+1),de
	jp	ARGLP2

;destroy: af,bc
ARGSCHK:
	ld	a,03h
	ld	(FAC1),a
	ld	bc,STRDSC1
	ld	(FAC1+1),bc
	ret

;function
FNC:
	ld	a,(OPRTR)
	or	a
	jp	z,ARGTAIL

	ld	a,02h
	ld	(FAC1),a
	ld	a,(hl)
	or	a
	jp	p,SNERR
	cp	0cdh+1
	jp	nc,SNERR
	inc	hl
	ld	(PROGAD),hl
	call	CALLFNC
FNCRTN:
	ld	hl,(PROGAD)
CLRSTRD:
	ld	a,(FAC1)
	sub	03h
	jr	z,CLRSTRDZ
	xor	a
	ld	(STRDSC1),a
CLRSTRDZ:
	ld	(OPRTR),a	;=0
	jp	ARGLP1

CALLFNC:
	cp	RND_
	jr	z,SKIPARG
	cp	PEEK_
	jr	z,SKIPARG
	cp	PAI_
	jr	z,SKIPARG
	cp	CHR_
	jr	z,SKIPARG
	cp	CVI_+1
	jr	nc,SKIPARG

;INT()...SQR(), ATN()...JOY(), STR$()...CVI()
;INT()...SQR(), ATN()...FIX(), RAD()...JOY(), STR$()...CVI()
	push	af
	call	CHKLPAR
	call	ARG
	jp	z,MOERR
	call	CHKRPAR
	pop	af

SKIPARG:
	ld	de,FNCTBL
	jp	JPTBL

ARGCMD:
	cp	INPUT_
	jp	z,F_INPT
	inc	a
	jr	z,FNC		;ff
	inc	a
	cp	EQV_+2
	jp	c,ARGTAIL	;80-e7,fe
;e8-fd
	ld	d,b		;operator

;input: d=operator
OPR:
	ld	a,(OPRTR)
	inc	a
	cp	02h
	jr	nc,OPROPR	;not 00 nor ff
OPR1:
	ld	a,(FAC1)
	inc	a
	jr	nz,NOTHEAD

;first +, for numeric and string
	ld	hl,FAC1
	ld	(hl),02h
	xor	a
	inc	hl
	ld	(hl),a		;FAC1+1
	inc	hl
	ld	(hl),a		;FAC1+2

	ld	a,d		;operator
	cp	PLUS_
	jr	z,ARGNEXT2
	cp	MINUS_
	jr	z,NOTHEAD
	cp	NOT_
	jr	z,NOTHEAD
	or	a
	jp	nz,SNERR
	dec	a
	ld	(FAC1),a	;ff=unkown
NOTHEAD:
	ld	a,d		;operator
	call	GETPRIO
	ld	e,a
	pop	bc		;c=previous operator
	push	bc
	ld	a,c
	or	a
	jr	z,ARGHEAD

	call	GETPRIO		;previous operator
	cp	e		;priority
	jr	nc,CALLOP
	ld	a,d		;operator
ARGNEXT:
	ld	(OPRTR),a
	call	PUSHF1
	ld	a,(FAC1)
	cp	03h
	call	z,COPYSTR
ARGNEXT2:
	ld	hl,(PROGAD)
	jp	ARGLP1

OPROPR:
	ld	a,d		;operator
	cp	NOT_
	jr	z,ARGNEXT
	cp	PLUS_
	jr	z,ARGNEXT2
	cp	MINUS_
	jp	nz,SNERR
OPRMNS:
	call	SETMNS1I
	ld	a,ASTRSK_
	jr	ARGNEXT

CALLOP:
	call	CPYFAC
	call	POPF1
	ld	a,(FAC2)
	cp	03h
	jr	nz,CALLOPEND

;string
	push	bc		;c=previous operator
	ld	a,(STRDSC1)
	ld	hl,(STRDSC1+1)
	call	BACKSTR
	ld	(STRDSC4),a
	ld	(STRDSC4+1),hl
	pop	bc

CALLOPEND:
	push	de		;;d=following operator
	ld	a,c
	ld	de,OPRTBL-(EQV_-80h)*2
	call	JPTBL
	xor	a
	ld	(OPRTR),a
	pop	de		;;d=following operator
	jp	OPR1

ARGHEAD:
	add	a,d		;d=0?
	jr	nz,ARGNEXT
	pop	hl		;cancel dummy operator
	ld	hl,(PROGAD)
	ld	a,(FAC1)
	inc	a
	ret

;input: a=operator
;output: a=priority
;destroy: f,hl
GETPRIO:
	or	a
	ret	z
	sub	EQV_
	ld	hl,PRIO
	add	a,l
	ld	l,a
	jr	nc,PRIONC
	inc	h
PRIONC:
	ld	a,(hl)
	ret

;operator priority:
; eqv
; imp
; xor
; or
; and
; not
; >< <> =< <= => >= =  >  <
; +-
; mod
; yen mark
; / *
; ^
PRIO:
;operator priority table
	db	1, 2, 3, 4, 5, 6, 7, 7
	db	7, 7, 7, 7, 7, 7, 7, 8
	db	8, 9,10,11,11,12


;increment hl and get a 1-byte integer argument [0,255]
INT1INC:ds	INT1ARG-INT1INC-1
	inc	hl
;	jp	INT1ARG

;get a 1-byte integer argument [0,255]
;input: hl=program address
;output: FAC1,a,de=integer, hl=next address
;destroy: FAC2,f,bc
_INT1ARG:ds	INT1ARG-_INT1ARG
	org	INT1ARG

	call	INT2ARG
	ld	a,d
	or	a
	ld	a,e
	ret	z
	jp	FCERR


;get two 1-byte integer arguments [0,255]
;input: hl=program address
;output: FAC1, bc=1st, de=2nd, hl=next address
;destroy: af
INT1ARG2:
	call	INT1ARG
	ld	a,(hl)
	cp	','
	jp	nz,SNERR
	push	de
	call	INT1INC
	pop	bc
	ret


;get a 2-byte integer argument [-32768,32767]
;input: hl=program address
;output: FAC1,de=integer, hl=next address
;destroy: FAC2,af,bc
INT2ARG:
	call	NARGMO
	push	hl
	call	FTOI2
	pop	hl
	ret


;get a 2-byte integer argument [-65535,65535]
;input: hl=program address
;output: FAC1,de=integer, hl=next address
;destroy: FAC2,af,bc
INTARG:
	call	NARGMO
	push	hl
	call	FTOI
	pop	hl
	ret


;get a numerical argument
;input: hl=program address
;output: FAC1, hl=next address, z-flag(1=other than numerical value)
;destroy: af,bc,de,FAC2
NUMARG:
	push	hl
	call	ARG
	pop	de
	ret	z		;no value
	ld	a,(FAC1)
	sub	03h
	ret	nz		;numerical value
	ex	de,hl
	ld	(PROGAD),hl
	ret


;get a numerical argument and check result (MISSING OPERAND)
;input: hl=program address-1
;output: FAC1, hl=next address
;destroy: af,bc,de,FAC2
NARGMOINC:
	inc	hl

;get a numerical argument and check result (MISSING OPERAND)
;input: hl=program address
;output: FAC1, hl=next address
;destroy: af,bc,de,FAC2
NARGMO:
	call	ARG
	ld	a,(FAC1)
	cp	03h
	jp	z,TMERR
	inc	a
	ret	nz
	jp	MOERR


;get a numerical argument for function
;input: hl=program address
;output: FAC1
;destroy: FAC2,af,bc,de,hl
FNCNUM:
	call	CHKLPAR
	call	ARG
	jp	z,MOERR
	call	CHKRPAR
;	jr	CHKNUM

;check numeric or not
;destroy: af
CHKNUM:
	ld	a,(FAC1)	;type
CHKNUMTYP:
	cp	03h
	jp	z,TMERR
	inc	a
	ret	nz
	jp	SNERR


;check numeric or not for FAC1 and FAC2
;destroy: af
CHKNUM2:
	call	CHKNUM
	ld	a,(FAC2)
	jr	CHKNUMTYP


;get a string argument (when no value, no error)
;input: hl=program address
;output: a=length, de=string address, hl=next address, z-flag(1=string)
;destroy: f,bc,FAC1,FAC2
SARGNOVAL:
	call	ARG
	jr	nz,STRARG2	;string or number
	or	0ffh		;reset z-flag
	ld	a,00h		;length=0
	ret			;no value


;get a string argument
;input: hl=program address
;output: a=length, de=string address, hl=next address, z-flag(1=ok)
;destroy: f,bc,FAC1,FAC2
STRARG:
	call	ARG
	jp	z,MOERR
STRARG2:
	ex	de,hl
	call	CHKSTR
;get string address
	inc	hl
	ld	c,(hl)
	inc	hl
	ld	h,(hl)
	ld	l,c
	ex	de,hl
	ret


;check string argument or not
;output: a=length, hl=string descriptor address
;destroy: f
CHKSTR:
	ld	a,(FAC1)
	cp	03h
	jr	nz,CHKSERR
	ld	hl,(FAC1+1)
	ld	a,(hl)
	ret

CHKSERR:
	inc	a
	jp	nz,TMERR
	jp	SNERR


;get x,y parameter
;input: hl=program address
;output: bc=x, de=y, hl=next address
;destroy: af
GETXY:
	call	INT1ARG2
	ld	a,c
	cp	WIDTH
	jp	nc,FCERR
	ld	a,e
	cp	HEIGHT
	ret	c
	jp	FCERR


;get string,integer argument for LEFT$(),RIGHT$(),MID$()
;input: hl=program address
;output: STRDSC1, a=length, FAC1,de=integer
;destroy: FAC2,f,bc
ARGSI1:
	call	CHKLPAR
	call	ARG
	jp	z,MOERR
	ex	de,hl
	call	CHKSTR
	ex	de,hl
	push	af		;length
	ld	a,(hl)
	cp	','
	jp	nz,SNERR
	call	ARGI1
	pop	af		;length
	ret

;protect string descriptor and get integer argument
;input: hl=program address
;output: FAC1,de=integer, hl=next address
;destroy: FAC2,af,bc
ARGI1:
	push	hl		;program address
	call	COPYSTR
	pop	hl
	call	INT1INC
	jp	BACKSTR


;check integer or not
;input: FAC1, FAC2
;output: z-flag(1=integer and integer),
;        FAC1,FAC2(if z-flag=0, convert integer to single)
;destroy: af,b,hl
CHKINT:
	ld	a,(FAC1)
	cp	02h
	ld	a,(FAC2)
	jr	nz,CHKINTNZ

	cp	02h
	ret	z
	call	ITOS
	or	0ffh		;reset z-flag
	ret

CHKINTNZ:
	cp	02h
	ret	nz
	call	ITOS2
CHKINTEND:
	or	0ffh		;reset z-flag
	ret


;compare FAC2 and FAC1 (string or number)
;input: FAC1,FAC2
;output: c-flag, z-flag
;destroy: af,bc,de,hl
CMPSTRNUM:
	ld	a,(FAC1)
	cp	03h
	jr	z,CMPSTR
	ld	a,(FAC2)
	cp	03h
	jr	nz,CMPF
	jp	TMERR

;set FAC2 and compare
;input: FAC1,hl=float address
;output: c-flag, z-flag, FAC2
;destroy: af,bc,de,hl
SETCMPF:
	call    SETF2
;	jp	CMPF

;compare FAC2 and FAC1 (number)
;input: FAC1,FAC2
;output: c-flag, z-flag
;destroy: af,bc,de,hl
CMPF:
	call	CHKINT
	jr	z,CMPI		;integer
	call	CNVTYP2SD

;single-single or double-double
CMPFSD:
	ld	de,FAC1+2
	ld	hl,FAC2+2
	or	h		;h>0, reset z-flag
	ld	a,(hl)
	rlca
	ld	a,(de)
	jr	c,CMPFNEG
	rlca
	ret	c

CMPFPOS:
	dec	e		;dec de
	dec	l		;dec hl
	dec	l		;dec hl
	ld	b,(hl)		;FAC2 type
CMPFLP:
	inc	l		;inc hl
	ld	a,(de)
	cp	(hl)
	ret	nz
	inc	de
	djnz	CMPFLP
	ret

CMPFNEG:
	rlca
	ret	nc

;FAC1<0, FAC2<0
	ex	de,hl
	jr	CMPFPOS


;compare FAC2 and FAC1 (signed integer)
;input: FAC1,FAC2
;output: c-flag, z-flag
;destroy: af,de,hl
CMPI:
	ld	hl,(FAC1+1)
	ld	de,(FAC2+1)
	or	a
	sbc	hl,de
	ld	a,h
	rlca
	ret	po		;if v-flag=0
	ccf
	ret

;compare FAC2 and FAC1 (string)
CMPSTR:
	ld	a,(FAC2)
	cp	03h
	jp	nz,TMERR
	ld	hl,STRDSC4
	ld	b,(hl)		;length
	ld	(hl),00h
	ld	a,(STRDSC1)
	cp	b
	push	af		;flag
	jr	c,CMPSNC
	ld	a,b
CMPSNC:
	or	a
	jr	z,CMPSZ

	ld	de,(STRDSC1+1)
	ld	hl,(STRDSC4+1)
	ld	b,a
CMPSLP:
	ld	a,(de)
	cp	(hl)
	jr	nz,CMPSNZ
	inc	de
	inc	hl
	djnz	CMPSLP
CMPSZ:
	pop	af		;flag
	ret

CMPSNZ:
	pop	hl		;cancel stack
	ret


;FAC1 = FAC1 + FAC2 (integer)
;input: FAC1,FAC2
;output: FAC1
;destroy: FAC2,af,bc,de,hl
ADDI:
	ld	hl,(FAC1+1)
	ld	de,(FAC2+1)
	or	a
	adc	hl,de
	jp	pe,ADDIOVER
	ld	(FAC1+1),hl
	ret

ADDIOVER:
	call	CNVTYP2
	jr	ADDFSD

;FAC1=FAC2-FAC1
SUBF21:
	call	CHKINT
	jr	z,SUBI21
	call	CNVTYP2SD
	ld	hl,FAC1+2
	jr	CHGSGNADD

;FAC1 = FAC2 - FAC1 (integer)
SUBI21:
	ld	hl,(FAC2+1)
	ld	de,(FAC1+1)
	jr	SUBIMAIN

;FAC1 = FAC1 - FAC2 (integer)
;input: FAC1,FAC2
;output: FAC1
;destroy: FAC2,af,bc,de,hl
SUBI:
	ld	hl,(FAC1+1)
	ld	de,(FAC2+1)
SUBIMAIN:
	or	a
	sbc	hl,de
	jp	pe,SUBFOVER
	ld	(FAC1+1),hl
	ret

SUBFOVER:
	call	CNVTYP2
	jr	SUBFSD

;FAC1 = FAC1 - FAC2
;input: FAC1,FAC2
;output: FAC1
;destroy: FAC2,af,b,de,hl
SUBF:
	call	CHKINT
	jr	z,SUBI
	call	CNVTYP2

;single or double
SUBFSD:
	ld	hl,FAC2+2
CHGSGNADD:
	ld	a,(hl)
	xor	80h
	ld	(hl),a
	jp	ADDFSD

;+1
PLSONES:
	db	05h, 81h, 00h, 00h, 00h, 00h
PLSONED:
	db	08h, 81h, 00h, 00h, 00h, 00h, 00h, 00h, 00h


;FAC1=FAC1+1(single or double)
;destroy: FAC2,af,bc,de,hl
INCF1:
	ld	hl,PLSONES
	call	SNGORDBL
;	jr	SETADDF

;set FAC2 and add
;input: FAC1, hl=float address
SETADDF:
	call    SETF2
;	jr	ADDF


;FAC1 = FAC1 + FAC2
;input: FAC1,FAC2
;output: FAC1
;destroy: FAC2,af,bc,de,hl
ADDF:
	call	CHKINT
	jr	z,ADDI
	call	CNVTYP2SD

;single or double
ADDFSD:
	ld	hl,(FAC2+1)	;l=(FAC2+1),h=(FAC2+2)
	ld	a,l
	or	a
	ret	z		;FAC2=0

	push	hl
	set	7,h
	ld	(FAC2+1),hl
	ld	a,l

	ld	hl,(FAC1+1)	;l=(FAC1+1),h=(FAC1+2)
	push	hl
	set	7,h
	ld	(FAC1+1),hl

	sub	l
	jr	c,ADD1GT2	;|FAC1| > |FAC2|
	jr	nz,ADD1LE2

	call	CMPFSD
	jr	c,ADD1GT2	;|FAC1| > |FAC2|

ADD1LE2:
	call	EXFAC
	pop	de
	pop	hl

	ld	a,e
	or	a
	jr	nz,ADDCHKSGN
	ld	(FAC1+1),hl
	ret

ADD1GT2:
	pop	hl		;l=(FAC1+1),h=(FAC1+2)
	pop	de		;e=(FAC2+1),d=(FAC2+2)

ADDCHKSGN:
	ld	a,l
	sub	e
	ld	b,a		;exponent difference (FAC1-FAC2)

	ld	a,h
	push	af		;sign

	xor	d
	rlca
	push	af		;c-flag=0:same sign

	call	c,NEGINT2

SAMESGN:
	ld	de,FAC2
	ld	a,(de)
	add	a,e		;no carry
	ld	e,a		;de=FAC2+5 or FAC2+8
	ld	c,00h		;for round-up

ADDFLP1:
	ld	a,b		;exponent difference
	sub	08h
	jr	c,EXPLT8
	ld	b,a		;exponent difference
	ld	a,(de)
	ld	c,a		;for round-up

	ld	hl,(FAC2+6)
	ld	(FAC2+7),hl
	ld	hl,(FAC2+4)
	ld	(FAC2+5),hl
	ld	hl,(FAC2+2)
	ld	(FAC2+3),hl

	pop	af		;same sign?
	push	af
	sbc	a,a		;00h or ffh
	ld	(FAC2+2),a

	jp	ADDFLP1

EXPLT8:
	add	a,08h
	jr	z,SAMEEXP

	pop	af		;same sign?
ADDFLP2:
	push	af		;same sign?
	ld	hl,FAC2+2
	ld	a,(FAC2)	;type
	bit	0,a
	jr	nz,ADDFNZ	;single

	rr	(hl)
	inc	l
	rr	(hl)
	inc	l
	rr	(hl)
	inc	l
ADDFNZ:
	rr	(hl)
	inc	l
	rr	(hl)
	inc	l
	rr	(hl)
	inc	l
	rr	(hl)
	inc	l

	rr	c		;for round-up
	pop	af		;same sign?
	djnz	ADDFLP2
	push	af		;same sign?

SAMEEXP:
	ld	hl,FAC1
	ld	a,(hl)
	ld	b,a
	dec	b
	add	a,l		;no carry
	ld	l,a		;FAC1+5 or FAC1+8
;	or	a
ADDFLP3:
	ld	a,(de)		;FAC2+5 or FAC2+8...
	adc	a,(hl)
	ld	(hl),a
	dec	e		;dec de
	dec	l		;dec hl
	djnz	ADDFLP3

	rla
	pop	de		;e-bit0=same sign?
	xor	e
	rrca
	jr	nc,SEARCH1

;	ld	hl,FAC1+1
	inc	(hl)
	jp	z,OVERR

;	scf			;c-flag=1
	ld	a,(FAC1)
	dec	a
	ld	b,a
ADDFLP4:
	inc	l		;inc hl
	rr	(hl)
	djnz	ADDFLP4
	rr	c		;
	jp	CHKROUND

SEARCH1:
;	ld	hl,FAC1+1
	dec	l		;dec hl
	ld	a,(hl)		;type
	add	a,l		;no carry
	ld	l,a		;hl=FAC1+5 or FAC1+8

ADDFLP5:
	ld	a,(FAC1+2)
	rlca
	jr	c,CHKROUND
	ld	d,h		;FAC1+5 or FAC1+8
	ld	e,l
	ld	a,(FAC1)	;type
	dec	a
	ld	b,a		;4 or 7
	sla	c
ADDFLP6:
	rl	(hl)
	dec	l		;dec hl
	djnz	ADDFLP6
	dec	(hl)		;exponent
	ex	de,hl		;hl=FAC1+5 or FAC1+8
	jr	nz,ADDFLP5
	jp	MULFZERO	;pop af/jp SETZERO

CHKROUND:
	rl	c		;for round-up
	jr	nc,ADDSGN

;round up
	ld	a,(FAC1)
	dec	a
	ld	b,a
ADDFLP7:
	inc	(hl)
	jr	nz,ADDSGN
	dec	l		;dec hl
	djnz	ADDFLP7

;	ld	hl,FAC1+1
	inc	(hl)
	jp	z,OVERR
;	ld	(FAC1+2),80h

ADDSGN:
	ld	hl,FAC1+2
	pop	af		;sign
	rlca
	ld	a,(hl)
	rla
	rrca
	ld	(hl),a
	ret


;hl=a*b
;input: a,b
;output: hl
;destroy: af,bc
MULINT1:
	ld	hl,0000h
	ld	c,l		;=0
	jr	MULI1NZ
MULI1C:
	add	hl,bc
MULI1NZ:
	srl	b
	rr	c
	add	a,a
	jr	c,MULI1C
	jr	nz,MULI1NZ
	ret


;dehl=de*hl
;input: de,hl
;output: dehl
;destroy: af,bc
MULINT2:
	ld	a,h
	ld	c,l
	ld	hl,0000h
	ld	b,10h
MULI2LP:
	rra
	rr	c
	jr	nc,MULI2NC
	add	hl,de
MULI2NC:
	rr	h
	rr	l
	djnz	MULI2LP
	rra
	rr	c
	ld	e,c
	ld	d,a
	ex	de,hl
	ret


;FAC1 = FAC1 * 0.1
;input: FAC1
;output: FAC1
;destroy: FAC2,af,bc,de,hl
DIV10:
	ld	hl,EM1S
	ld	a,(FAC1)	;05 or 08
	rrca
	jr	c,SETMULF	;if single
	ld	hl,EM1D
	jr	SETMULF

;FAC1 = FAC1 * 10
;input: FAC1
;output: FAC1
;destroy: FAC2,af,bc,de,hl
MUL10:
	ld	hl,TENS
	ld	a,(FAC1)	;05 or 08
	rrca
	jr	c,SETMULF	;if single
	ld	hl,TEND
;	jr	SETMULF

;set FAC2 and multiply
;input: FAC1, hl=float address
SETMULF:
	call	SETF2
;	jr	MULF

;FAC1 = FAC1 * FAC2
;input: FAC1,FAC2
;output: FAC1
;destroy: FAC2,af,bc,de,hl
MULF:
	call	CNVTYP2
	call	MULDIVSGN
	jp	z,SETZERO	;FAC2=0
	ret	c		;FAC1=0

;initial result={0(4or7bytes),FAC1(4or7bytes)}
	push	af		;sign
	ld	a,(FAC1)
	dec	a
	ld	b,a		;4 or 7
	ld	hl,MULWRK
MULFLP1:
	ld	(hl),00h
	inc	hl
	djnz	MULFLP1

	ld	(TMP),hl	;MULWRK+4or7
	ex	de,hl
	ld	hl,FAC1+2
;	ld	b,00h
	ld	c,a		;4 or 7
	ldir

	ld	c,a		;4 or 7
	add	a,a
	add	a,a
	add	a,a		;*8
;	or	a
	ld	b,a

MULFLP2:
	push	bc

;shift result and FAC1
	push	af		;c-flag
	ld	a,c		;4 or 7
	add	a,a		;8 or 14
	ld	b,a
	pop	af		;c-flag
	ld	hl,MULWRK
MULFLP3:
	rr	(hl)		;(MULWRK)...(MULWRK+7or13)
	inc	hl
	djnz	MULFLP3
	jr	nc,MULFNC1

;add
	ld	de,FAC2
	ld	a,c		;4 or 7
	inc	a
	add	a,e		;no carry
	ld	e,a		;de=FAC2+5 or FAC2+8

	add	a,MULWRK-FAC2-2	;no carry
;	or	a
	ld	l,a		;hl=MULWRK+3or6
	ld	h,d

	ld	b,c		;4 or 7
MULFLP4:
	ld	a,(de)
	adc	a,(hl)
	ld	(hl),a
	dec	de
	dec	hl
	djnz	MULFLP4
MULFNC1:
	pop	bc
	djnz	MULFLP2

	ld	hl,FAC1+1
	dec	(hl)		;;
	jr	nc,MULFNC2
	inc	(hl)

;shift result
	ld	hl,MULWRK
	ld	b,c		;4 or 7
	inc	b		;5 or 8
MULFLP5:
	rr	(hl)		;(MULWRK)...(MULWRK+4or7)
	inc	hl
	djnz	MULFLP5

MULFNC2:
;round to the nearest even
	ld	hl,(TMP)	;MULWRK+4or7
	ld	a,(hl)
	adc	a,a		;check (MULWRK+4or7)-bit7
	jr	nc,MULFEND

	ld	b,c		;4 or 7
	dec	b		;3 or 6
MULFLP6:
	inc	hl
	or	(hl)		;(MULWRK+5or8)...(MULWRK+7or13)
	djnz	MULFLP6

	ld	hl,(TMP)
	dec	hl		;MULWRK+3or6

	jr	nz,ROUNDUP
	ld	a,(hl)
	rrca
	jr	nc,MULFEND

ROUNDUP:
	ld	b,c		;4 or 7
MULFLP7:
	inc	(hl)
	jr	nz,MULFEND
	dec	hl
	djnz	MULFLP7

	ld	a,80h
	ld	(MULWRK+1),a
	ld	hl,FAC1+1
	inc	(hl)
	jp	z,OVERR

MULFEND:
	ld	b,00h
	ld	de,FAC1+2
	ld	hl,MULWRK
	ldir
;exponent
	ld	hl,FAC1+1
	ld	a,(FAC2+1)
	add	a,(hl)
	jr	c,MULFC
	sub	80h		;;not 81h
	jr	nc,MULEXP

MULFZERO:
	pop	af		;cancel
	jp	SETZERO

MULFC:
	sub	80h		;;not 81h
MULFCHKOV:
	jp	nc,OVERR
MULEXP:
	ld	(hl),a		;FAC1+1
	jr	z,MULFZERO
	inc	l		;inc hl,FAC1+2
	pop	af		;sign
	rlca
	ld	a,(hl)
	rla
	rrca
	ld	(hl),a
	ret


;check signs and set sign bits of FAC1/2 for MULF/DIVF
;output: a-bit7(xor of FAC1,2 sign), c-flag(1:FAC1=0) z-flag(1:FAC2=0)
;destroy: f,hl
MULDIVSGN:
	ld	hl,FAC2+1
	ld	a,(hl)		;FAC2+1
	or	a
	ret	z
	ld	l,FAC1+1-(FAC1+1)/256*256
	ld	a,(hl)		;FAC1+1
	cp	01h
	ret	c
	inc	l
	ld	a,(hl)		;FAC1+2
	set	7,(hl)
	ld	l,FAC2+2-(FAC2+2)/256*256
	xor	(hl)		;FAC2+2, clear c-flag
	set	7,(hl)
	dec	h		;clear z-flag
	ret


;FAC1 = FAC1 / FAC2
;input: FAC1,FAC2
;output: FAC1
;destroy: FAC2,af,bc,de,hl
DIVF:
	call	CNVTYP2
	call	MULDIVSGN
	jp	z,DIV0ERR	;FAC2=0
	ret	c		;FAC1=0

	push	af		;sign
	xor	a
	ld	l,a
	ld	h,a
	ld	e,a
	ld	d,a
	ld	c,a
	ld	b,a
	ld	(TMP),a

	exx			;
	push	hl
	push	de
	push	bc

	ld	a,(FAC1)
	dec	a		;4 or 7
	add	a,a		;*2
	add	a,a		;*4
	add	a,a		;*8
	ld	c,a		;32 or 56

	call	CMPINT
	jr	nc,DIVFLP
	ld	hl,FAC1+1
	dec	(hl)		;exponent

	call	SLAINT

DIVFLP:
	jr	c,DIVFC1
	call	CMPINT
	jr	c,DIVFC2
DIVFC1:
	call	SUBINT
	or	a		;reset c-flag
DIVFC2:
	ccf
	exx			;

	ld	a,(FAC1)	;type=05 or 08
	bit	3,a
	jr	z,DIVZ		;05=single
	adc	hl,hl
	rl	e
DIVZ:
	rl	d
	rl	c
	rl	b
	ld	a,(TMP)
	rla
	ld	(TMP),a
	exx		;

	call	SLAINT
	dec	c
	jr	nz,DIVFLP

	rla			;store c-flag
	ld	c,a
	call	CMPINT
	ld	a,c
	pop	bc
	pop	de
	pop	hl
	exx

	rra
	jr	c,DIVFUP	;SLAINT
	rlca
	jr	c,DIVFNOUP	;CMPINT

;round to the nearest even
	jr	nz,DIVFUP	;CMPINT
	ld	a,l
	rrca
	jr	nc,DIVFNOUP

;round up
DIVFUP:
	inc	hl
	ld	a,h
	or	l
	jr	nz,DIVFNOUP
	inc	de
	ld	a,d
	or	e
	jr	nz,DIVFNOUP
	inc	bc
	ld	a,b
	or	c
	jr	nz,DIVFNOUP
	ld	a,(TMP)
	inc	a
	ld	(TMP),a
	jr	nz,DIVFNOUP
	ld	hl,FAC1+1
	inc	(hl)		;exponent
	jp	z,OVERR
	ld	hl,0000h
;	ld	de,0000h
;	ld	bc,0000h
	ld	a,80h
	ld	(TMP),a

DIVFNOUP:
	ld	a,l
	ld	(FAC1+8),a
	ld	a,h
	ld	(FAC1+7),a
	ld	a,e
	ld	(FAC1+6),a

	ld	hl,FAC1+5
	ld	(hl),d
	dec	hl
	ld	(hl),c
	dec	hl
	ld	(hl),b
	dec	hl
	ld	a,(TMP)
	ld	(hl),a
	dec	hl

	ld	a,(FAC2+1)	;exponent
	ld	b,a
	ld	a,(hl)		;FC1+1,exponent
	sub	b
	jr	c,DIVF1LT2

;FAC1>=FAC2
	add	a,81h
	ccf
	jp	MULFCHKOV

;FAC1<FAC2
DIVF1LT2:
	jp	p,MULFZERO
	add	a,81h
	jp	MULEXP


;check sign
;input: FAC1
;output: z-flag(1:FAC1=0), c-flag(1:FAC1<0)
;destroy: af,hl
CHKSGN:
	ld	a,(FAC1)
	cp	02h
	jr	nz,CHKSGNSD
	ld	hl,(FAC1+1)
	ld	a,h
	or	l
	ret	z
	ld	a,h
	rlca
	ret

;check sign of single/double
;input: FAC1
;output: z-flag(1:FAC1=0), c-flag(1:FAC1<0)
;destroy: af
CHKSGNSD:
	ld	a,(FAC1+1)	;exponent
	or	a
	ret	z
	ld	a,(FAC1+2)	;sign
	rlca
	ret


;convert type of FAC1
;input: FAC1, c=type
;output: FAC1
;destroy: af,b,de,hl
CNVTYP1:
	ld	a,(FAC1)
	cp	c
	ret	z
	cp	02h
	ld	a,c
	jp	z,ITOSD
	cp	02h
	jp	z,SDTOI
	cp	05h
	jp	z,DTOS
	jp	STOD


;convert types of FAC1 and FAC2 to single-single or double-double
;input: FAC1, FAC2
;output: FAC1, FAC2
;destroy: af,b,hl
CNVTYP2:
	ld	a,(FAC1)
	cp	02h
	call	z,ITOS
	ld	a,(FAC2)
	cp	02h
	jr	nz,CNVTYP2SD
	call	ITOS2

;FAC1,FAC2=single or double
CNVTYP2SD:
	ld	a,(FAC1)
	ld	hl,FAC2
	cp	(hl)
	ret	z
	jr	nc,STOD2
;	jr	c,STOD


;convert single to double
;input: FAC1
;output: FAC1
;destroy: af,b,hl
STOD:
	ld	a,08h
STODTYP:
	ld	(FAC1),a
	xor	a
	ld	h,a
	ld	l,a
	jp	ITOSEND


;convert single to double (FAC2)
;input: FAC2
;output: FAC2
;destroy: af,b,hl
STOD2:
	ld	a,08h
	ld	(FAC2),a
	xor	a
	ld	h,a
	ld	l,a
	jp	ITOS2END


;convert double to single
;input: FAC1
;output: FAC1
;destroy: af,b,hl
DTOS:
;round to the nearest even
	ld	hl,(FAC1+7)
	ld	a,(FAC1+6)
	or	a
	rla
	jr	nc,DTOSEND
	or	h	
	or	l
	ld	hl,FAC1+5
	ld	b,03h
	jr	nz,DTOSLP
	ld	a,(hl)
	rrca
	jr	nc,DTOSEND
DTOSLP:
	inc	(hl)
	jr	nz,DTOSEND
	dec	hl
	djnz	DTOSLP
	ld	a,(hl)		;sign
	inc	a
	xor	(hl)
	and	7fh
	xor	(hl)
	ld	(hl),a
	and	7fh
	jr	nz,DTOSEND
	dec	hl
	inc	(hl)		;exponent
	jp	z,OVERR

DTOSEND:
	ld	a,05h
	jp	STODTYP


;convert 1-byte signed integer to single float
;input: a=integer (signed)
;output: FAC1
;destroy: af,bc,hl
I1TOF:
	ld	l,a
	rlca
	sbc	a,a
	ld	h,a
;	jr	I2TOF

;convert 2-byte signed integer to single float
;input: hl=integer (signed)
;output: FAC1
;destroy: af,b,hl
I2TOF:
	ld	(FAC1+1),hl
	jr	ITOS


;convert integer to double
;input: FAC1
;output: FAC1
;destroy: af,b,hl
ITOD:
	ld	a,08h
	db	21h		;ld hl,

;convert integer to single
;input: FAC1
;output: FAC1
;destroy: af,b,hl
ITOS:
	ld	a,05h
ITOSD:
	ld	(FAC1),a
	ld	hl,(FAC1+1)
	ld	b,h		;bit7=sign
	ld	a,h
	rlca
	jr	nc,ITOSPLS	;FAC1>=0
ITOSMNS:
	xor	a
	sub	l
	ld	l,a
	sbc	a,h
	sub	l
	ld	h,a
ITOSPLS:
;	ld	a,h
	or	l
	jr	z,ITOSZERO

	ld	a,h
	or	a
	jr	nz,ITOSNZ
;<256
	ld	h,l
	ld	l,a		;=0
	ld	a,89h
	jr	ITOSLP

;>=256
ITOSNZ:
	ld	a,91h
ITOSLP:
	dec	a
	add	hl,hl
	jr	nc,ITOSLP

	ld	(FAC1+1),a
	ld	a,b		;bit7=sign
	rlca
	rr	h
	ld	a,l
	rra
	ld	l,h
	ld	h,a
ITOSZERO:
	ld	(FAC1+2),hl
	xor	a
	ld	h,a
	ld	l,a
	ld	(FAC1+4),hl
ITOSEND:
	ld	(FAC1+6),hl
	ld	(FAC1+8),a
	ret


;convert integer to single (FAC2)
;input: FAC2
;output: FAC2
;destroy: af,b,hl
ITOS2:
	ld	a,05h
	ld	(FAC2),a
	ld	hl,(FAC2+1)
	ld	b,h		;bit7=sign
	ld	a,h
	rlca
	jr	nc,ITOS2PLS	;FAC2>=0
ITOS2MNS:
	xor	a
	sub	l
	ld	l,a
	sbc	a,h
	sub	l
	ld	h,a
ITOS2PLS:
;	ld	a,h
	or	l
	jr	z,ITOS2ZERO

	ld	a,h
	or	a
	jr	nz,ITOS2NZ
;<256
	ld	h,l
	ld	l,a		;=0
	ld	a,89h
	jr	ITOS2LP

;>=256
ITOS2NZ:
	ld	a,91h
ITOS2LP:
	dec	a
	add	hl,hl
	jr	nc,ITOS2LP

	ld	(FAC2+1),a
	ld	a,b		;bit7=sign
	rlca
	rr	h
	ld	a,l
	rra
	ld	l,h
	ld	h,a
ITOS2ZERO:
	ld	(FAC2+2),hl
	xor	a
	ld	h,a
	ld	l,a
	ld	(FAC2+4),hl
ITOS2END:
	ld	(FAC2+6),hl
	ld	(FAC2+8),a
	ret


;convert single/double to integer
;input: FAC1
;output: FAC1
;destroy: af,b,de,hl
SDTOI:
	push	bc
;	call	FTOI2
	call	FTOI
	pop	bc
	ld	(FAC1+1),de
	ld	a,02h
	ld	(FAC1),a
	ret


;convert string to number
;input: hl=address
;output: FAC1, hl=next address
;destroy: FAC2, af,bc,de
ATOF:
	call	SKIPSP
	call	ATOICODE
	push	hl
	ld	hl,FAC1
	call	ICODENUM
	pop	hl
	ret


;convert intermediae code to number
;input: hl=address
;output: FAC1, hl=next address
;destroy: af,bc,de
ICODENUM:
	ld	a,(hl)
	inc	hl
	dec	a
	cp	0ah
	jr	c,ICODE1DIG	;0-9
	jr	z,ICODEINT	;line number
	cp	12h-1
	jr	z,ICODEINT	;integer
	cp	0fh-1
	jr	z,ICODEINT	;hex
	cp	0eh-1
	jr	z,ICODEINT	;binary
	cp	0dh-1
	jr	z,ICODEINT	;octal
	cp	15h-1
	jr	z,ICODEFLOAT	;single
	cp	18h-1
	jr	z,ICODEFLOAT	;double
	jp	SNERR

ICODE1DIG:
	ld	d,00h
	ld	e,a
ICODE02:
	ld	(FAC1+1),de
	ld	a,02h
	ld	(FAC1),a
	ret

ICODEINT:
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
	jr	ICODE02

ICODEFLOAT:
	sub	10h-1
	ld	de,FAC1
	ld	(de),a		;05 or 08
	inc	de
	ld	b,00h
	ld	c,a
	ldir
	cp	08h
	ret	z		;double

;single
	xor	a
	ld	(de),a
	inc	de
	ld	(de),a
	inc	de
	ld	(de),a
	ret


;convert number string to intermediate code
;input: hl=address
;output: FAC1, a=length, hl=next address
;destroy: FAC2, f,bc,de
ATOICODE:
	ld	a,(hl)
	cp	'&'
	jp	z,AICOBH

	ld	a,12h		;initial value=integer
	ld	(AICTYP+1),a

	call	ATOIF
	ld	a,c		;before-dot counter
	ld	(BFRDCNT+1),a
	ld	d,h		;
	ld	e,l
	call	SKIPSP
	ld	c,00h		;after-dot counter
	cp	'.'
	ex	de,hl
	jr	nz,CHKEXP
	ex	de,hl

;after dot
	inc	hl
;	ld	b,00h		;sign
AICLP1:
	call	CTOF
	jr	nc,BFRDCNT
	inc	c
	jr	AICLP1
BFRDCNT:
	ld	a,00h		;before-dot count
	add	a,c		;total digits
	cp	0ah
	ld	a,15h
	jr	c,AICNC
	ld	a,18h
AICNC:
	ld	(AICTYP+1),a

CHKEXP:
	ld	d,h
	ld	e,l
	call	SKIPSP
	call	CHK1011
	jr	nz,CHKEXPNZ1
	inc	hl
	ld	a,(hl)
CHKEXPNZ1:
	ld	b,18h
	cp	'D'
	jr	z,CHKEXPZ
	ld	b,15h
	cp	'E'
	jr	nz,NOEXP
CHKEXPZ:
	call	SKIPSPINC
	call	CHK1011
	jr	nz,CHKEXPNZ2
	inc	hl
	ld	a,(hl)
CHKEXPNZ2:
	push	hl		;
	cp	'+'
	jr	z,AICEXP
	cp	'-'
	jr	z,AICEXP
	call	CHKFIG
	jr	c,AICEXP
;with E/D, but not exponent
	pop	af		;cancel stack
	or	a		;reset z-flag

;no exponent
NOEXP:
	ex	de,hl
	push	af		;z-flag=0: no E/D
	ex	de,hl		;address
	call	PUSHF1
	xor	a
	sub	c
	call	I1TOF
	jr	AICE0		;E0/D0

AICEXP:
	pop	hl		;
	xor	a		;set z-flag
	push	af		;z-flag=1: with E/D
;	inc	hl
	ld	a,b
	ld	(AICTYP+1),a
	ex	de,hl
	call	PUSHF1
	ex	de,hl
	push	bc		;c=after-dot counter
	call	ATOIF
	pop	bc		;c=after-dot counter
	push	hl		;address
	call	CPYFAC
	ld	a,c		;c=after-dot counter
	call	I1TOF
	call	SUBF21
	pop	de		;address

AICE0:
	call	CPYFAC
	call	POPF1
	push	de		;address

	ld	hl,FAC2+2
	ld	a,(hl)
	push	af		;exponent sign
	res	7,(hl)

AICLP2:
	ld	a,(FAC2+1)
	or	a
	jr	z,AICLP2END	;E0/D0

	ld	a,(FAC1+1)
	or	a
	jr	z,AICLP2END	;=0

	call	PUSHF1
	call	SETMNS1
	call	ADDF
	call	CPYFAC
	call	POPF1

	pop	bc		;b=exponent sign
	push	bc
	call	PUSHF2
	ld	a,b
	rlca
	push	af
	call	nc,MUL10
	pop	af
	call	c,DIV10
	call	POPF2
	jr	AICLP2

AICLP2END:
	pop	af		;cancel exponent sign
	pop	hl		;address
	pop	af		;z-flag: with E/D?
	jr	z,AICTYP	;with exponent

;no exponent
	ld	d,h
	ld	e,l
	call	SKIPSP
	ld	b,15h
	cp	'!'
	jr	z,AICTYPB
	ld	b,18h
	cp	'#'
	jr	z,AICTYPB
	ex	de,hl
	ld	a,(AICTYP+1)
	cp	12h		;initial value(integer)?
	jr	nz,AICFLOAT	;float
	ld	a,(FAC1+1)	;exponent
	cp	90h
	jr	nc,AICFLOAT2	;>=32768

;integer,<=32767
	ex	de,hl
	ld	a,(hl)
	cp	'%'
	ex	de,hl
	jr	nz,AICNZ
	ex	de,hl
	inc	hl
AICNZ:
	push	hl
	call	SDTOI
	ld	hl,(FAC1+1)
	ld	a,h
	or	a
	ld	a,l
	pop	hl
	jr	nz,AICTYP
	cp	10
	jr	nc,AICTYP

;integer (0-9): 01h-0ah
	inc	a
	ld	(FAC1),a
	ld	a,01h
	ret

;!#
AICTYPB:
	inc	hl
	ld	a,b
	jr	AICSETTP

;>=32768 -> single or double
AICFLOAT2:
	ld	a,(FAC1+6)
	ld	de,(FAC1+7)
	or	d
	or	e
	ld	a,15h
	jr	z,AICZ
	ld	a,18h
AICZ:
	ld	(AICTYP+1),a

;float
AICFLOAT:
	ld	a,(hl)
	cp	'%'
	jp	z,SNERR
;	jr	AICTYP

;integer (10-32767): 12h
;single: 15h
;double: 18h
AICTYP:
	ld	a,00h		;type
AICSETTP:
	ld	(FAC1),a
	sub	0fh		;length
	cp	06h
	ret	nz		;integer or double

;single
	push	hl
	call	DTOS
	pop	hl
	ld	a,15h
	ld	(FAC1),a
	ld	a,06h
	ret


;convert string to float (integer part)
;input: hl=address
;output: FAC1, hl=next address, c=digits
;destroy: FAC2,af,bc,de
ATOIF:
	push	hl
	call	SETZEROD
	pop	hl
	call	SKIPSP
	ld	c,00h
	or	a
	ret	z

	ld	b,c		;=0, sign for CTOF
	inc	hl
	cp	'+'
	jr	z,ATOIFZ
	cp	PLUS_
	jr	z,ATOIFZ
	inc	b
	cp	'-'
	jr	z,ATOIFZ
	cp	MINUS_
	jr	z,ATOIFZ
	dec	b
	dec	hl
ATOIFZ:
	ld	d,h
	ld	e,l
ATOIFLP:
	push	de
	call	CTOF
	pop	de
	ex	de,hl
	ret	nc
	ld	h,d
	ld	l,e
	ld	a,(FAC1+1)	;exponent
	or	a
	jr	z,ATOIFLP
	inc	c
	jr	ATOIFLP


;FAC1 = FAC1 * 10 + [0-9]
;input: FAC1, hl=address, b=sign(0=plus, 1=minus)
;output: FAC1, c-flag(1=OK), hl=next address
;destroy: FAC2,af,de
CTOF:
	ld	d,h
	ld	e,l
	call	SKIPSP
	or	a
	ret	z		;c-flag=0

	sub	'0'
	cp	'9'-'0'+1
	ex	de,hl
	ret	nc
	ex	de,hl
	inc	hl
	push	hl

	push	bc
	push	af
	call	ABS
	call	MUL10
	call	CPYFAC
	pop	af
	call	I1TOF
	call	ADDF
	pop	bc

	ld	a,b
	rrca
	call	c,NEGABS

	pop	hl
	scf
	ret


;&O,&B,&H
AICOBH:
	ld	de,0000h
	call	SKIPSPINC
	call	CHK1011
	jr	z,AICOBH
	ld	b,0fh		;type=0f
	cp	'H'
	jr	z,AICHEX
	dec	b		;type=0e
	cp	'O'
	jr	z,AICOCT
	dec	b		;type=0d
	cp	'B'
	jp	nz,SNERR

;&Bxxxxxxxxxxxxxxxx
AICBIN:
	ld	c,11h
AICBINLP:
	call	SKIPSPINC
	sub	'0'
	cp	'1'-'0'+1
	jr	nc,OBHEND
	dec	c
	jp	z,OVERR
	ex	de,hl
	add	hl,hl		;*2
	add	a,l		;no carry
	ld	l,a
	ex	de,hl
	jr	AICBINLP

;&Oxxxxxx
AICOCT:
	ld	c,07h
AICOCTLP:
	call	SKIPSPINC
	sub	'0'
	cp	'7'-'0'+1
	jr	nc,OBHEND
	dec	c
	jp	z,OVERR
	ex	de,hl
	add	hl,hl		;*2
	add	hl,hl		;*4
	add	hl,hl		;*8
	jp	c,OVERR
	add	a,l		;no carry
	ld	l,a
	ex	de,hl
	jr	AICOCTLP

;&Hxxxx
AICHEX:
	ld	c,05h
AICHEXLP:
	call	SKIPSPINC
	call	CHK1011
	jr	z,AICHEXLP
	call	HEX
	jr	nc,OBHEND

AICHEX0F:
	dec	c
	jp	z,OVERR
	ex	de,hl
	add	hl,hl		;*2
	add	hl,hl		;*4
	add	hl,hl		;*8
	add	hl,hl		;*16
	add	a,l		;no carry
	ld	l,a
	ex	de,hl
	jr	AICHEXLP

OBHEND:
	ld	a,b
	ld	(FAC1),a	;type
	ld	(FAC1+1),de	;value
	ld	a,03h		;length
	ret


;print string
;input: hl=address
;destroy: none
PUTS:
	push	af
	push	hl

PUTSLP:
	ld	a,(hl)
	or	a
	jr	z,PUTSEND
	call	PUTC
	inc	hl
	jr	PUTSLP

PUTSEND:
	pop	hl
	pop	af
	ret


;print string and new line
;input: hl=address
;destroy: af
PUTSNL:
	call	PUTS
;	jr	PUTNL

;put new line
;destroy: af
PUTNL:
	ld	a,0dh
	call	PUTDEV
	ld	a,(SELPRN)
	or	a
	jr	z,PRTNL		;CRT
	jr	PUTNLPRN	;printer


;put new line to printer if necessary
;destroy: af
PUTNLXPRN:
	ld	a,(PRNPOS)
	or	a
	ret	z

;put new line for printer
;destroy: af
PUTNLPRN:
	xor	a
	ld	(PRNPOS),a	;=0
	ld	a,0dh
	call	PRINTER
	ld	a,0ah
	jp	PRINTER


;put new line if necessary
;destroy: af
PUTNLX:
	ld	a,0dh
	call	PUTDEV
	ld	a,(SELPRN)
	or	a
	jr	nz,PUTNLXPRN	;printer
;	jr	PRTNLX

;set (SELPRN) and put new line in CRT if necessry
PRTNLX:
	xor	a
	ld	(SELPRN),a
	ld	a,(DSPXY)
	push	hl
	ld	hl,XMIN
	cp	(hl)
	pop	hl
	ret	z

;print new line in CRT
PRTNL:
	push	hl
	ld	hl,(DSPXY)
	call	CTLCR
	pop	hl
	ld	a,(PLOTON+1)
	or	a
	ret	z
	jp	PUTNLPRN


;put a character
;input: a=ASCII code, (SELPRN)=0:CRT, 1:printer
;destroy: af
PUTC:
	call	PUTDEV
	push	af
	ld	a,(SELPRN)
	or	a
	jp	nz,PUTCNZ	;printer
	pop	af
;	jp	PRTC		;CRT

;print a character in CRT (00-1f: control code)
;input: a=ASCII code
;destroy: af
PRTC:
	cp	20h
	jp	c,PRTCTL
;	jr      PRTCH

;print a character in CRT (no control code)
;input: a=ASCII code
;destroy: af
PRTCH:
	push	hl
	ld	h,a
PLOTON:
	ld	a,00h		;0=plot off, not0=plot on
	or	a
	ld	a,h
	call	nz,PUTPRNCODE	;plot on
	ld	a,h
	call	ADCN
	ld	h,a
	call	CHGVRAM
	ld	a,h
	ld	hl,(CSRAD)
	ld	(hl),a
	set	3,h		;attribute
	ld	a,(ATTR)
	ld	(hl),a
	call	CHGDRAM
	call	CTLRGT
PRTCEND:
	pop	hl
	ret


;put a character (no control code)
;input: a=ASCII code, (SELPRN)=0:CRT, 1:printer
PUTCH:
	call	PUTDEV
	push	af
	ld	a,(SELPRN)
	or	a
	jp	nz,PUTCNZ
	pop	af
	jr	PRTCH

CTLTBL:
	dw	CTLEND, CTLEND, CTLEND, CTLEND, CTLEND, CTLCLL, CTLEND, CTLBEL
	dw	CTLDEL, CTLEND, CTLEND, CTLHOM, CTLCLS, CTLCR,  CTLSCU, CTLSCD
	dw	CTLCS1, CTLCS0, CTLINS, CTLHIR, CTLEND, CTLALP, CTLKAT, CTLGRP
	dw	CTLEND, CTLEND, CTLERA, CTLEND, CTLRGT, CTLLFT, CTLUP,  CTLDWN

;control code
;destroy: af
PRTCTL:
	push	hl
	push	de
	ld	h,a
	ld	a,(PLOTON+1)
	or	a
	ld	a,h
	call	nz,PLOTONCTL	;plot on
	ld	a,h
	ld	hl,(DSPXY)
	ld	de,CTLTBL
	call	JPTBL
	pop	de
	pop	hl
CTLEND:
	ret

;plot on (control code)
PLOTONCTL:
	ld	l,15h
	cp	0bh		;home
	jr	z,PLOTONCTLEND
	inc	l
	cp	0ch		;clr
	jr	z,PLOTONCTLEND
	ld	l,' '
	cp	10h		;face
	ret	c
	cp	11h+1		;face
	ret	nc
PLOTONCTLEND:
	ld	a,l
	jp	PUTPRN


;clear from cursor position to line end(05)
CTLCLL:
	ld	a,l
	jp	DELLIN2

;bell(07)
CTLBEL:
	push	bc
	call	BELL
	pop	bc
	ret

;del(08)
CTLDEL:
	ld	hl,(DSPXY)
	call	CHKLINE2
	ld	hl,DELSTR
	jr	nz,CTLDELNZ	;connected to previous character
	inc	hl
CTLDELNZ:
	call	PUTS
	call	CHKIPOS2
	call	CHGVRAM
	ld	de,(CSRAD)
DELLP:
	call	NEXTPOS		;de=left address, hl=right xy
	call	CHKLINE2
	jr	z,DELZ
	push	hl		;right xy
	call	XY2AD		;de=left address, hl=right address
	ld	a,(hl)
	ld	(de),a
	set	3,h
	set	3,d
	ld	a,(hl)
	ld	(de),a
	res	3,h
	ex	de,hl
	pop	hl		;de=right address, hl=right xy
	jr	DELLP
DELZ:
	xor	a
	ld	(de),a
	set	3,d
	ld	a,(ATTR)
	ld	(de),a
	call	CHGDRAM
	ret

DELSTR:
	db	1dh, ' ', 1dh, 00h

;get next position
;input: h=y, l=x
;output: h=new y, l=new x
;destroy: af
NEXTPOS:
	inc	l
	ld	a,(XMAX)
	cp	l
	ret	nc
	ld	a,(XMIN)
	ld	l,a
	inc	h
	ret

;input: h=y, l=x
;output: h=y, l=x
;destroy: af
PREVPOS:
	ld	a,(XMIN)
	cp	l
	dec	l
	ret	c
	ld	a,(XMAX)
	ld	l,a
	dec	h
	ret

;home(0b)
CTLHOM:
	ld	hl,(XMIN)	;l=(XMIN),h=(YMIN)
	jr	SETCSR

;cls(0c)
CTLCLS:
	jp	CLSMAIN

;right(1c)
CTLRGT:
	ld	hl,(CSRAD)
	inc	hl
;next line?
	ld	(CSRAD),hl
	ld	hl,DSPXY
	inc	(hl)
	ld	a,(XMAX)
	cp	(hl)
	ret	nc

	inc	hl
	ld	a,(hl)		;y
	ld	hl,(YMIN)	;l=(YMIN)
	ld	h,a
	ld	a,(YMAX)
	sub	l
	jr	CTLCR2		;cut line connection if YMIN=YMAX

;CR(0d)
CTLCR:
	ld	a,(ATTR)
	and	7fh
	ld	(ATTR),a
	xor	a
CTLCR2:
	push	de
	call	SETLINE		;0=cut line connection, 1-255=connect
	pop	de
	ld	hl,(DSPXY)
	ld	a,(XMIN)
	ld	l,a
	inc	h
	ld	a,(YMAX)
	cp	h
	jr	nc,SETCSR

;scroll
	ld	hl,(XMIN)	;h=(YMIN)
	ld	l,a
CNSL1SCR:
	ld	a,0ffh		;1 screen?
	or	a
	push	af
	call	nz,SCRLUP
	pop	af
	call	z,SCRLUP49
	ld	h,l
	call	DELLIN
	ld	a,(XMIN)
	ld	l,a

;set cursor position
;input: h=y, l=x
;output: (DSPXY)=xy, (CSRAD)=hl=VRAM address
;destroy: af
SETCSR:
	ld	(DSPXY),hl
	call	XY2AD
	ld	(CSRAD),hl
	ret


;scroll up(0e)
CTLSCU:
	ld	a,(CNSL1SCR+1)
	or	a
	jp	z,SCRLUP50
	ld	l,h		;y
	ld	a,(YMIN)
	ld	h,a
	call	SCRLUP
	ld	h,l		;y
	jp	DELLIN

;scroll down(0f)
CTLSCD:
	ld	a,(CNSL1SCR+1)
	or	a
	jp	z,SCRLDWN50
	ld	l,h		;y
	ld	a,(YMAX)
	ld	h,a
	call	SCRLDWN
	ld	h,l		;y
	jp	DELLIN

;set character set=1(10)
CTLCS1:
	ld	hl,ATTR
	set	7,(hl)
	ret

;set character set=0(11)
CTLCS0:
	ld	hl,ATTR
	res	7,(hl)
	ret

;hiragana(13)
CTLHIR:
	call	CTLCS1
	jr	CTLKANA

;alphanumeric(15)
CTLALP:
	call	CTLCS0
	ld	hl,005ah
	jr	SETINPMOD

;katakana(16)
CTLKAT:
	call	CTLCS0
CTLKANA:
	ld	hl,0143h
	jr	SETINPMOD

;graph(17)
CTLGRP:
	ld	hl,02ffh
;input: h=mode,l=cursor character (display code
SETINPMOD:
	ld	a,h
	ld	(KANAF),a
	ld	a,l
	ld	(FLSDT),a
	ret

;erase to console end(1a)
CTLERA:
	ld	a,(YMAX)
	ld	hl,(DSPXY)
CTLERALP:
	push	af
	call	DELLIN
	pop	af
	inc	h
	cp	h
	jr	nc,CTLERALP
	ret

;left(1d)
CTLLFT:
	ld	a,(XMIN)
	cp	l
	jr	z,CTLLFTZ
	ld	a,l
	or	a
	jr	z,CTLLFTZ
	dec	l
	ld	a,(XMAX)
	cp	l
	jp	nc,SETCSR
	ld	l,a
	jp	SETCSR

;previous line?
CTLLFTZ:
	ld	a,(YMIN)
	cp	h
	ret	z
	ld	a,h
	or	a
	ret	z

	dec	h
	ld	a,(XMAX)
	ld	l,a
	jp	SETCSR

;up(1e)
CTLUP:
	ld	a,(YMIN)
	cp	h
	jr	nz,CTLUPNZ
	ld	a,(CNSL1SCR+1)
	or	a
	ret	nz
	jp	SCRLDWN50

CTLUPNZ:
	ld	a,h
	or	a
	ret	z
	dec	h
	dec	h
;	jr	CTLDWN

;down(1f)
CTLDWN:
	inc	h
	ld	a,(YMAX)
	cp	h
	jp	nc,SETCSR
	ld	h,a
	push	hl
	call	SETCSR
	pop	hl
	jp	CTLSCU

;inst(12)
CTLINS:
	push	bc
	ld	hl,(DSPXY)
	ld	d,00h
	ld	a,(ATTR)
	ld	e,a
	call	NEXTPOS
	call	CHKLINE2
	jr	z,INSSCRL
	ld	hl,(DSPXY)
	call	CHGVRAM
INSLP:
	push	hl
	call	XY2AD
	ld	a,(hl)
	set	3,h
	ld	b,(hl)
	ld	(hl),e
	res	3,h
	ld	(hl),d
	ld	d,a
	ld	e,b
	pop	hl
	call	NEXTPOS
	call	CHKLINE2
	jr	nz,INSLP

	call	CHGDRAM
	ld	a,d
	or	a
	jr	nz,INSSCRL
	pop	bc
	ret

INSSCRL:
	push	hl		;;
	ld	a,(YMAX)
	cp	h
	jr	c,INSSCRLU
	ld	l,h
	ld	h,a
	jr	z,NOSCRLD
	call	SCRLDWN

INSSCRLEND:
	ld	c,l		;
	dec	l
	ld	a,0ffh		;connect

INSSCRLEND2:
	ld	h,l
	push	de
	call	SETLINE
	pop	de
	ld	h,c		;
	call	DELLIN

	pop	hl		;;
	call	XY2AD

	call	CHGVRAM
	ld	(hl),d
	set	3,h
	ld	(hl),e
	call	CHGDRAM
	pop	bc
	ret

INSSCRLU:
	pop	hl		;;
	dec	h
	push	hl		;;

	ld	l,a		;(YMAX)
	ld	a,1eh		;up
;	call	PRTC
	call	PUTC
	ld	a,(YMIN)
	ld	h,a
	sub	l		;(YMIN)-(YMAX)
	ld	c,l		;(YMAX)
	jr	z,NOSCRLU
	call	SCRLUP
	jr	INSSCRLEND

NOSCRLD:
	ld	a,0ffh
	ld	(INPTXY+1),a	;scroll out
	jr	INSSCRLEND

NOSCRLU:
;	ld	a,0ffh
	dec	a
	ld	(INPTXY+1),a
	xor	a		;not connect
	jr	INSSCRLEND2


;printer (continued from PUTC)
PUTCNZ:
	pop	af
;	jr	PUTPRN

;convert to printer code and print
;input: a
;destroy: af
PUTPRN:
	push	af
	call	PRINTER
	jr	c,PUTPRNERR
	pop	af
	cp	20h
	jr	nc,PUTPRNNC
	cp	11h
	ret	c
	cp	17h
	ret	nc
PUTPRNNC:
	ld	a,(PRNPOS)
	inc	a
	ld	(PRNPOS),a
	ret

;time out
PUTPRNERR:
	xor	a
	ld	(PLOTON+1),a
	jp	DOERR

;put data to printer
;input: a=ASCII code
;output: c-flag(1=time out)
;destroy: af
PRINTER:
	push	hl
	push	bc
	push	af
	ld	a,(PRNMOD)
	or	a
	jr	nz,PRNERR	;graphic mode
	ld	b,10h
	ld	hl,0000h
PRNLP1:
	in	a,(0feh)
	rrca
	jr	nc,PRNNC	;wait for #RDA=0
;	and	02h
;	jr	nz,PRNERR	;if SNS=1
	dec	hl
	ld	a,h
	or	l
	jr	nz,PRNLP1
	djnz	PRNLP1
PRNERR:
;time out
	pop	af
	pop	bc
	pop	hl
	scf
	ret

PRNNC:
	pop	af
	out	(0ffh),a
	ld	a,80h
	out	(0feh),a	;set RDP=1
	ld	hl,1000h
PRNLP2:
	in	a,(0feh)
	rrca
	jr	c,PRNEND	;wait for #RDA=1
	dec	hl
	ld	a,h
	or	l
	jr	nz,PRNLP2
;time out
	xor	a
	scf
	jr	PRNEND2

PRNEND:
	xor	a
PRNEND2:
	out	(0feh),a	;set RDP=0
	pop	bc
	pop	hl
	ret

PUTPRNCODE:
	call	PRNCODE
	jr	PUTPRN


;input: a=ASCII code
;output: a=printer code
;destroy: f
PRNCODE:
	cp	11h
	jr	c,PRNCODEEND	;00-10
	cp	17h
	ret	c		;11-16
	cp	20h
	jr	c,PRNCODEEND	;17-1f
	cp	60h
	ret	c		;00-5f
	cp	70h
	jr	c,PRNCODEEND	;60-6f
	cp	0c1h
	ret	c
	cp	0cfh
	ret	z
	cp	0d7h
	ret	z
	cp	0ffh
	ret	z
PRNCODEEND:
	ld	a,'.'
	ret


;set printer
;destroy: af
SETPRN:
	call	CHKTMODE
	ld	a,01h
	ld	(SELPRN),a
	ret


;check if printer mode is text (error if graphic mode)
;destroy: af
CHKTMODE:
	ld	a,(PRNMOD)
	or	a
	ret	z		;text mode
	jp	BPERR		;graphic mode


;check if printer mode is graphic (error if text mode)
;destroy: af
CHKGMODE:
	ld	a,(PRNMOD)
	or	a
	ret	nz
	jp	BPERR		;text mode


;check connection to previous character
;input: h=y, l=x
;output: z-flag(1=not connect), a(0=not connect)
;destroy: f
CHKLINE2:
	ld	a,(XMIN)
	cp	l
	jr	nc,CHKLINE3
	inc	a		;reset z-flag
	ret

;check connection to previous line
;input: h=y
;output: z-flag(1=not connect), a(0=not connect)
;destroy: f
CHKLINE3:
	push	hl
	dec	h
	db	3eh		;ld a,

;check connection to next line
;input: h=y
;output: z-flag(1=not connect), a(0=not connect)
;destroy: f
CHKLINE4:
	push	hl
	call	CALCLINE
	and	(hl)
	pop	hl
	ret

;cut connection before cursor line
;output: hl=line status address
;destroy: af,de
CUTLINE:
	ld	a,(DSPXY+1)

;cut connection to previous line
;input: a=y
;output: hl=line status address
;destroy: af,de
CUTLINE2:
	or	a
	ret	z
	ld	h,a
	dec	h
	xor	a
;	jp	SETLINE

;set line connection status (connection to next line)
;input: h=y, a=data(0=not connect)
;output: hl=line status address
;destroy: f,de
SETLINE:
	ld	d,a
	call	CALCLINE
	inc	d
	dec	d
	jr	z,SETLINEZ
	or	(hl)
	ld	(hl),a
	ld	a,d
	ret

SETLINEZ:
	cpl
	and	(hl)
	ld	(hl),a
	ld	a,d
	ret


;calculate line status address and bit (connection to next line)
;input: h=y
;output: hl=line status address, a=bit
;destroy: f
CALCLINE:
	push	bc
	ld	a,h
	ld	hl,LINEST-1
CALCLLP1:
	inc	hl
	sub	08h
	jr	nc,CALCLLP1
	add	a,09h
	ld	b,a
	ld	a,80h
CALCLLP2:
	rlca
	djnz	CALCLLP2
	pop	bc
	ret


;scroll up
;input: h=first line, l=last line
;destroy: af
SCRLUP:
	push	hl
	push	de
	push	bc

	ld	a,l
	sub	h
	jr	c,SCRLUEND
	jr	z,SCRLUEND
	ld	c,a		;last-first

	ld	a,(XMIN)
	ld	l,a
	ld	a,(XMAX)
	sub	l
	inc	a
	ld	b,a		;(XMAX)-(XMIN)+1

	exx
	push	hl
	push	de
	push	bc
	exx

	push	hl
	push	bc
	ld	b,c
SCRLULP1:
	inc	h
	ld	c,h		;
	call	CHKLINE4
	dec	h
	call	SETLINE
	ld	h,c		;

	ld	a,(INPTXY+1)
	cp	h
	jr	nz,SCRLUNZ
	dec	a
	ld	(INPTXY+1),a
SCRLUNZ:
	djnz	SCRLULP1

	xor	a
	call	SETLINE
	pop	bc
	pop	hl

	call	XY2AD		;x=(XMIN), y=first line
	call	CHGVRAM
	ex	de,hl

;display code address
	push	de

;attribute address
	exx
	pop	de
	set	3,d
	exx

SCRLULP2:
;display code address
	ld	hl,WIDTH
	add	hl,de
	push	hl		;
	push	bc

;attribute address
	exx
	ld	hl,WIDTH
	add	hl,de
	push	hl		;;
	exx

SCRLULP3:
	ldi			;display code
	inc	c
	exx
	ldi			;attribute
	exx
	djnz	SCRLULP3

;attribute address
	exx
	pop	de		;;
	exx

;display code address
	pop	bc
	pop	de		;
	dec	c
	jr	nz,SCRLULP2

	call	CHGDRAM

	exx
	pop	bc
	pop	de
	pop	hl
	exx

SCRLUEND:
	pop	bc
	pop	de
	pop	hl
	ret


;scroll down
;input: h=last line, l=first line
;destroy: af
SCRLDWN:
	push	hl
	push	de
	push	bc

	ld	a,h
	sub	l
	jr	c,SCRLDEND
	jr	z,SCRLDEND
	ld	c,a		;last-first

	ld	a,(XMIN)
	ld	l,a
	ld	a,(XMAX)
	sub	l
	inc	a
	ld	b,a		;(XMAX)-(XMIN)+1

	exx
	push	hl
	push	de
	push	bc
	exx

	push	hl
	push	bc
	ld	b,c
SCRLDLP1:
	dec	h
	ld	c,h		;
	call	CHKLINE4
	inc	h
	call	SETLINE
	ld	h,c		;

	ld	a,(INPTXY+1)
	cp	h
	jr	nz,SCRLDNZ
	dec	a
	ld	(INPTXY+1),a
SCRLDNZ:
	djnz	SCRLDLP1

	xor	a
	call	SETLINE

	pop	bc
	pop	hl

	call	XY2AD
	call	CHGVRAM
	ex	de,hl

;display code address
	push	de

;attribute address
	exx
	pop	de
	set	3,d
	exx

SCRLDLP2:
;display code address
	ld	hl,0-WIDTH
	add	hl,de
	push	hl		;
	push	bc

;attribute address
	exx
	ld	hl,0-WIDTH
	add	hl,de
	push	hl		;;
	exx

SCRLDLP3:
	ldi			;display code
	inc	c
	exx
	ldi			;attribute
	exx
	djnz	SCRLDLP3

;attribute address
	exx
	pop	de		;;
	exx

;display code address
	pop	bc
	pop	de		;
	dec	c
	jr	nz,SCRLDLP2

	call	CHGDRAM

	exx
	pop	bc
	pop	de
	pop	hl
	exx

SCRLDEND:
	pop	bc
	pop	de
	pop	hl
	ret


;scroll up circulary (50 lines)
;destroy: af
SCRLUP50:
	push	hl
	push	de
	push	bc
	call	CPY050
	ld	hl,HEIGHT*2
	call	SCRLUP
	pop	bc
	pop	de
	pop	hl
	ret

;scroll up circulary (49 lines)
;input: l=y
;destroy: af
SCRLUP49:
	push	hl
	push	de
	push	bc

	push	hl
	call	CPY050
	pop	hl

	ld	h,00h
	call	SCRLUP

	ld	h,l
	inc	h
	ld	l,HEIGHT*2
	call	SCRLUP

	pop	bc
	pop	de
	pop	hl
	ret

;copy y=0 line to y=50 line
CPY050:
	call	CHGVRAM
	ld	hl,0d000h	;y=0
	ld	de,0d000h+WIDTH*HEIGHT*2	;y=50
	ld	bc,WIDTH
	ldir
	ld	hl,0d800h
	ld	de,0d800h+WIDTH*HEIGHT*2
	ld	bc,WIDTH
	ldir
	call	CHGDRAM
	ret

;scroll down circulary (50 lines)
;destroy: af
SCRLDWN50:
	push	hl
	push	de
	push	bc
	ld	hl,HEIGHT*2*100h
	call	SCRLDWN
	call	CPY500
	pop	bc
	pop	de
	pop	hl
	ret

;scroll down circulary (49 lines)
;input: l=y
;destroy: af
SCRLDWN49:
	push	hl
	push	de
	push	bc

	ld	h,HEIGHT*2
	call	SCRLDWN

	ld	a,l
	or	a
	jr	z,SCRLD49END
	ld	h,l
	dec	h
	ld	l,00h
	call	SCRLDWN

	push	hl
	call	CPY500
	pop	hl

SCRLD49END:
	pop	bc
	pop	de
	pop	hl
	ret

;copy y=50 line to y=0 line
CPY500:
	call	CHGVRAM
	ld	hl,0d000h+WIDTH*HEIGHT*2	;y=50
	ld	de,0d000h	;y=0
	ld	bc,WIDTH
	ldir
	ld	hl,0d800h+WIDTH*HEIGHT*2
	ld	de,0d800h
	ld	bc,WIDTH
	ldir
	call	CHGDRAM
	ret


;delete a line
;input: h=y
;destroy: af
DELLIN:
	ld	a,(XMIN)

;input: a=x, h=y
DELLIN2:
	push	hl
	push	bc
	ld	l,a
	ld	a,(XMAX)
	sub	l
	inc	a
	ld	b,a		;(XMAX)-(XMIN)+1

	call	CHGVRAM
	call	XY2AD

	ld	a,(ATTR)
CLRLLP:
	ld	(hl),00h
	set	3,h		;attribute
	ld	(hl),a
	res	3,h
	inc	hl
	djnz	CLRLLP
	call	CHGDRAM
	pop	bc
	pop	hl
	ret


;clear screen (rectangle area)
;destroy: af,hl
CLSMAIN:
	push	de
	push	bc

	ld	hl,(XMIN)	;l=(XMIN),h=(YMIN)
	ld	(DSPXY),hl
	ld	a,(XMAX)
	sub	l
	inc	a
	ld	c,a		;(XMAX)-(XMIN)+1
	ld	a,(YMAX)
	sub	h
	inc	a
	ld	b,a		;(YMAX)-(YMIN)+1
	call	XY2AD
	ld	(CSRAD),hl

	call	CHGVRAM
;	call	VBLNK

	xor	a
	push	hl
	push	bc
	call	SETRECT
	pop	bc
	pop	hl
	set	3,h		;attribute
	ld	a,(ATTR)
	push	bc
	call	SETRECT
	pop	bc

	call	CHGDRAM

	ld	a,(YMIN)
	ld	h,a
CLSLP:
	ld	c,h		;
	xor	a
	call	SETLINE
	ld	h,c		;
	inc	h
	djnz	CLSLP

	pop	bc
	pop	de
	ret


;set VRAM rectangle area
;input: a=data, b=height, c=width, hl=VRAM address
;destroy: f,bc,de,hl
SETRECT:
	push	bc
	push	hl
	ld	b,00h
	ld	(hl),a
	dec	c
	jr	z,RECTZ
	ld	d,h
	ld	e,l
	inc	de
	ldir
RECTZ:
	pop	hl
	ld	c,WIDTH		;b=0
	add	hl,bc
	pop	bc
	djnz	SETRECT
	ret


;DIM command
C_DIM:
	call	CHKVAR
	jr	nz,NOTARR
	push	hl		;1st character address
	call	SRCHARR
	pop	hl		;1st character address
	jp	c,DDERR

	ld	a,c		;type
	exx
	push	hl
	push	de
	push	bc
	ld	h,00h		;element bytes
	ld	l,a		;type
	exx

	ld	(TMP),hl	;1st character address
	ld	hl,(PROGAD)	;address of '('
	inc	hl
	push	de		;array address
	push	de		;array address
	push	bc		;b=name length, c=type

	ld	c,00h		;dimension counter
DIMLP:
	push	bc		;c=dimensions
	call	INT2ARG
	pop	bc		;c=dimensions
	bit	7,d
	jp	nz,FCERR
	inc	c
	jp	z,OMERR		;over 255 dimensions

	ld	a,(BASE+1)
	or	a
	jr	nz,DIMNZ	;option base 1
	inc	de		;option base 0
DIMNZ:
	push	de
	exx
	pop	de
	call	MULINT2
	ld	a,d
	or	e
	jp	nz,OMERR
	exx

	ex	de,hl
	pop	de		;d=name length, e=type
	ex	(sp),hl		;push size, pop array address
	push	hl		;array address
	push	de		;d=name length, e=type

	ld	hl,(PROGAD)
	ld	a,(hl)
	inc	hl
	cp	','
	jr	z,DIMLP
	cp	')'
	jp	nz,SNERR
	ld	(PROGAD),hl
	call	MAKEARR

	exx
	pop	bc
	pop	de
	pop	hl
	exx

DIMNEXT:
	ld	hl,(PROGAD)
	call	SKIPSP
	cp	','
	ret	nz
	inc	hl
	jr	C_DIM

;normal variable
NOTARR:
	call	SRCHVAR
	call	nc,MAKEVAR
	jr	DIMNEXT


;screen input
;output: hl=INPBUF, c-flag(1=stop)
;destroy: af,bc,de
INPTSCR:
	ld	hl,(BUFP)
	ld	a,h
	or	l
	jr	nz,INPT1DEV
	ld	hl,(DSPXY)
	ld	a,(XMIN)
	ld	l,a
	ld	(INPTXY),hl
	ld	(INPTX),a

	call	SCREDIT
	ret	c
	ld	a,0ffh
	jr	COPYIBF


;one line input for CMT device
INPT1DEV:
	ld	hl,INPBUF
	ld	c,MAXINP
INPT1DLP:
	call	GETDEV
	jr	c,INPT1DC
	cp	0dh
	jr	z,INPT1DC
	ld	(hl),a
	inc	hl
	dec	c
	jr	nz,INPT1DLP
	jp	BOERR

INPT1DC:
	ld	(hl),00h
;	call	CLRBUFP
	ld	hl,INPBUF
	or	a		;reset c-flag
	ret

INPTX:
	db	00h
INPTXY:
	db	00h, 00h

;one line input for keyboard
;output: hl=INPBUF, c-flag(1=stop)
;destroy: af,bc,de
INPT1:
	ld	hl,(BUFP)
	ld	a,h
	or	l
	jr	nz,INPT1DEV

	ld	hl,(DSPXY)
	ld	(INPTXY),hl
	ld	a,l
	ld	(INPTX),a

	call	SCREDIT
	ret	c
	ld	a,(INPTX)
;	jr	COPYIBF

;copy characters to input buffer from VRAM
;input: a=length, h=start y, l=start x
;output: hl=INPBUF, c-flag=0
;destroy: af,bc,de
COPYIBF:
	push	af
	xor	a
	ld	(TMPATTR+1),a
	ld	de,INPBUF
	pop	af
	or	a
	jr	z,CPIBFZ3
	ld	c,a
	ld	b,MAXINP
	call	CHGVRAM

CPIBFLP1:
	push	hl		;x,y
	call	XY2AD
	ld	a,(hl)
	call	ALPKANA
	ld	a,00h
	jr	nc,TMPATTR

	set	3,h		;attribute
	ld	a,(hl)
	res	3,h
	and	80h
	rlca			;00h or 01h
TMPATTR:
	cp	00h		;00h or 01h
	ld	(TMPATTR+1),a
	jr	z,CPIBFZ1
	xor	11h		;11h or 10h
	ld	(de),a
	inc	de
	dec	b
	jp	z,BOERR

CPIBFZ1:
	ld	a,(hl)
	call	DACN
	call	ABCN
	ld	(de),a
	inc	de
	pop	hl		;x,y
	call	NEXTPOS
	call	CHKLINE2
	jr	z,CPIBFZ2
	dec	c
	jr	z,CPIBFZ2
	djnz	CPIBFLP1

CPIBFZ2:
	call	CHGDRAM
	ld	a,(TMPATTR+1)
	or	a
	jr	z,CPIBFZ3
	ld	a,11h
	ld	(de),a
	inc	de
	dec	b
	jp	z,BOERR
CPIBFZ3:
	call	PREVPOS
	call	SETCSR
	call	PUTNL

CPIBFLP2:
	dec	de
	ld	a,(de)
	cp	' '
	jr	z,CPIBFLP2
	inc	de
	xor	a
	ld	(de),a
	ld	hl,INPBUF
	ret


;screen edit
;output: h=start y, l=start x, c-flag(1=stop)
;destroy: af,bc,de
SCREDIT:
	call	CUTLINE
SEDLP:
	call	GETC
	call	SEDMAIN
	jr	SEDLP

;screen edit main routine
SEDMAIN:
	cp	20h
	jr	c,SEDCTL

INSMODE:
	ld	h,00h
	inc	h
	dec	h
	call	nz,INSERT

SEDPRTC:
	ld	h,a		;
	ld	a,(KANAF)	;0=alphanum, 1=kana, 2=graph
	or	a
	ld	a,h		;
	jr	nz,SEDPRTC2	;kana mode, graph mode

;alphanumeric mode
	cp	'A'
	jp	c,SEDPRTC2
	cp	'Z'+1
	jr	nc,SEDPRTC2
;A-Z
	ld	a,(MODKEY+1)
	rrca
	ld	a,(ATTR)
	push	af		;
	ld	a,h
	call	nc,CTLCS1	;shift
;	call	PRTC
	call	PUTC
	pop	af		;
	ld	(ATTR),a
	jr	CHKIPOS2

SEDPRTC2:
;	call	PRTC
	call	PUTC
CHKIPOS2:
	ld	hl,(DSPXY)
;check cursor position for input command
;input: h=y, l=x
;destroy: af
CHKIPOS:
	ld	a,(INPTXY+1)	;y
	cp	h
	ret	nz

	ld	a,(INPTX)
	cp	l
	jr	nc,IPOSNC
	ld	a,l
	ld	(INPTX),a
IPOSNC:
	ld	a,(INPTXY)	;x
	cp	l
	ret	c
	ld	(INPTXY),hl
	ret

NOINSTBL:
	db	02h, 03h, 04h, 05h, 06h, 0bh, 0ch, 0dh, 0eh, 0fh
	db	12h, 1ah, 1ch, 1dh, 1eh, 1fh

;control code
SEDCTL:
	ld	hl,NOINSTBL
	ld	bc,0010h
	cpir
	jr	nz,SEDCTLNZ
	ld	hl,INSMODE+1
	ld	(hl),b		;=0
SEDCTLNZ:
	ld	h,a		;
	cp	16h
	jr	z,SEDKANAKEY	;kana key
	ld	a,(KANAF)
	cp	02h		;graph mode?
	ld	a,h		;
	jr	nz,SEDCTL2	;alphanumeric mode, kana mode
;graph mode
	call	BACN
	jr	nz,SEDCTL2
;reversed characters
	call	PUTCH
	jr	CHKIPOS2

;kana key
;shift+kana=hiragana for emulator
SEDKANAKEY:
	call	CHKMODK
	rrca
	ld	a,h		;
	jr	c,SEDCTL2	;no shift
	ld	a,13h		;with shift
;	jr	SEDCTL2

;a=control character
SEDCTL2:
	ld	hl,(DSPXY)
	cp	01h
	jr	z,SEDINS
	cp	02h
	jp	z,SEDCTLB
	cp	03h
	jr	z,SEDSTOP
	cp	04h
	jr	z,SEDINICON
	cp	06h
	jp	z,SEDCTLF
	cp	09h
	jp	z,SEDTAB
	cp	0ah
	jp	z,SEDCTLJ
	cp	0dh
	jp	z,SEDRET
	cp	14h
	jp	z,SEDSETTAB
	cp	18h
	jp	z,SEDCONN
	cp	19h
	jp	z,SEDCLRTAB
;shift+up/down:roll down/roll up
	jp	SEDPRTC

;ctrl-a
SEDINS:
	ld	hl,INSMODE+1
	ld	a,(hl)
	cpl
	ld	(hl),a
	ret

;ctrl-c
SEDSTOP:
	pop	af		;cancel return address
STOPLP:
	inc	h
	call	CHKLINE3
	jr	nz,STOPLP
	dec	h
	call	SETCSR
	call	PUTNL

	xor	a
	ld	(INPBUF),a
	scf
	ret

;ctrl-d
SEDINICON:
;color
	ld	a,70h		;fore=white, back=black
	ld	(ATTR),a
	ld	(FLAATR+1),a

;plot off
	xor	a
	ld	(PLOTON+1),a

;console area size
	ld	h,a		;=0
	ld	l,a		;=0
	ld	(XMIN),hl	;XMIN, YMIN
	ld	hl,WIDTH-1+256*(HEIGHT-1)
	ld	(XMAX),hl	;XMAX, YMAX

;tab setting
	ld	hl,TABTBL
	ld	(hl),a		;=0
	inc	a
	ld	b,04h
SEDINILP:
	inc	hl
	ld	(hl),a		;=1
	djnz	SEDINILP

;1 screen
	ld	(CNSL1SCR+1),a	;>0

;set character set=0
	jp	CTLCS0

;ctrl-b
SEDCTLB:
	ld	a,h
	or	l
	jp	z,SETCSR	;x=0,y=0
	ex	de,hl
	ld	hl,(XMIN)
;	or	a
	sbc	hl,de
	ex	de,hl
	jp	z,SETCSR	;x=xmin,y=ymin

	call	CHKCTLB
	jr	nc,SEDCTLB
CTLBLP:
	call	CHKCTLB
	jr	c,CTLBLP
	call	NEXTPOS
	call	CHKIPOS
	jp	SETCSR

CHKCTLB:
	call	PREVPOS
	ld	d,h
	ld	e,l
	ld	a,(YMIN)
	inc	h
	jr	CTLBC

;ctrl-f
SEDCTLF:
	call	CHKCTLF
	jr	c,SEDCTLF
CTLFLP:
	call	CHKCTLF
	jr	nc,CTLFLP
	ex	de,hl
	call	SETCSR
	jp	CHKIPOS2

;check next position
;input: hl=yx
;output: hl=next yx, de=old yx, c-flag(1=alphabet, numeral, kana)
;destroy: af,b
CHKCTLF:
	call	NEXTPOS
	ld	d,h
	ld	e,l
	ld	a,(YMAX)
	cp	h
	jr	c,CTLFC
CTLBC:
	ex	de,hl
	push	hl
	call	XY2AD
	call	CHGVRAM
	ld	a,(hl)
;	call	DACN
	ld	h,a		;
	call	CHGDRAM
	ld	a,h		;
	call	DACN
	pop	hl
	cp	0c0h
	ret	nc
	cp	081h
	ccf
	ret	c
	jp	ALPNUM

CTLFC:
	ld	l,WIDTH-1
	ld	h,a		;(YMAX)
	pop	af		;cancel return address
	jp	SETCSR

;ctrl-i
SEDTAB:
	ex	de,hl
SEDTABLP:
	ld	a,e
	ld	hl,(XMAX)
	cp	l
	jr	nc,SEDTABEND
	inc	e
	ld	l,e
	call	CALCTAB
	and	(hl)
	jr	z,SEDTABLP
	ld	l,e

SEDTABEND:
	ld	h,d
	call	SETCSR
	jp	CHKIPOS2

;ctrl-j
SEDCTLJ:
	ld	hl,(DSPXY)
	ld	a,WIDTH
	sub	l
	ld	b,a
CTLJLP:
	ld	a,12h
;	call	PRTC
	call	PUTC
	djnz	CTLJLP
	xor	a
	jp	SETLINE

;ctrl-m(CR)
SEDRET:
	pop	af		;cancel return address
	push	hl		;h=y, l=x

	ld	a,h
	ld	hl,(INPTXY)
	cp	h
	jr	nz,SETLENMAX	;if other line

	call	CHKLINE3
	jr	nz,SETLENMAX	;if connected to previous line
	call	CHKLINE4
	jr	nz,SETLENMAX	;if connected to next line

	ld	a,(INPTX)	;end position
	sub	l		;(INPTXY)
	jr	SETLEN

SETLENMAX:
	ld	a,0ffh
SETLEN:
	ld	(INPTX),a	;length
	pop	hl		;h=y, l=x
SRETLP:
	call	CHKLINE2
	push	af
	call	PREVPOS
	pop	af
	jr	nz,SRETLP

	call	NEXTPOS
	ld	de,(INPTXY)	;initial position
	ld	a,h
	cp	d
	jr	nz,SRETNZ	;if different line
	ex	de,hl		;if same line
SRETNZ:
	or	a		;reset c-flag
	ret


;ctrl-t
SEDSETTAB:
	call	CALCTAB
	or	(hl)
	ld	(hl),a
	ret

;ctrl-x
;connect next line
SEDCONN:
	ld	a,01h
	jp	SETLINE

;ctrl-y
SEDCLRTAB:
	call	CALCTAB
	cpl
	and	(hl)
	ld	(hl),a
	ret


;calculate tab setting address and bit
;input: l=x
;output: hl=line status address, a=bit
;destroy: f,b
CALCTAB:
	ld	a,l
	ld	hl,TABTBL-1
CALCTLP1:
	inc	hl
	sub	08h
	jr	nc,CALCTLP1
	add	a,09h
	ld	b,a
	ld	a,80h
CALCTLP2:
	rlca
	djnz	CALCTLP2
	ret

TABTBL:
	db	00h, 01h, 01h, 01h, 01h

;insert a character
;output: hl=VRAM address
;destroy: f,bc,de
INSERT:
	push	af
	ld	a,12h
;	call	PRTC
	call	PUTC
	pop	af
	ret


;get data from device (CMT, sequential read)
;output: a=data, c-flag(1=no data)
;destroy: f,de
GETDEV:
	push	hl
	push	bc
	ld	hl,(BUFP)
	inc	hl
	ld	a,(hl)		;+1: to next block?
	or	a
	jr	z,GETDZ

;next block
	push	hl
	call	RDNEXT
	pop	hl
	jr	c,GETDEND2	;no next block

GETDZ:
	ld	d,h
	ld	e,l
	inc	hl
	ld	c,(hl)		;+2: position
	inc	(hl)		;z-flag
	ld	b,00h
	add	hl,bc		;+2+pos
	ld	c,0eh
	add	hl,bc		;+16+pos
	ld	a,(hl)		;[+16,+271]: buffer
	jr	nz,GETDEND1	;not end
	ld	a,01h
	ld	(de),a		;+1: to next block?
	ld	a,(hl)
GETDEND1:
;	call	ABCN
	cp	1ah
	scf
	jr	z,GETDEND2
	or	a
GETDEND2:
	pop	bc
	pop	hl
	ret


;read next block
;c-flag(1=no next block)
;destroy: af,bc,hl
RDNEXT:
	ld	hl,(BUFP)
	ld	bc,000eh
	add	hl,bc		;record number
	ld	a,(hl)		;+14: record number(low)
	inc	hl
	and	(hl)		;+15: record number(high)
	inc	a
	scf
	ret	z		;if record number=ffff

	dec	hl		;+14: record number
	ld	(DTADR),hl
	ld	hl,0102h
	ld	(SIZE),hl
	call	RDD
	ld	hl,(BUFP)
	dec	a
	jr	z,RDNEXTERR
	dec	a
	jr	z,RDNEXTBRK
	inc	hl
	xor	a
	ld	(hl),a		;+1: to next block?
	ret

RDNEXTERR:
	call	FCLOSE2
	jp	TRERR

RDNEXTBRK:
	call	FCLOSE2
	jp	STOPSP


;put a character to device (CMT, sequential write)
;continued from PUTC
;input: a=data
;destroy: none(file buffer pointer=0) / af(file buffer pointer<>0)
PUTDEV:
	push	hl
	push	af
	ld	hl,(BUFP)
	ld	a,h
	or	l
	jr	nz,PUTDMAIN
	pop	af
	pop	hl
	ret

PUTDMAIN:
	pop	af
	push	de
	push	bc
	push	af		;data

	ld	a,(hl)		;+0: file open mode
	cp	02h
	jp	nz,BMERR

	inc	hl
	ld	a,(hl)		;+1: to next block?
	or	a
	jr	z,PUTDDATA

	push	hl
	ld	de,000dh
	ld	(hl),d		;+1: to next block?=0
	add	hl,de
	ld	(DTADR),hl
	ld	(hl),d		;+14: record number(low)
	inc	hl
	ld	(hl),d		;+15: record number(high)
	ld	hl,0102h
	ld	(SIZE),hl
	call	WRD
	jr	c,PUTDBRK
	ld	hl,(BUFP)
	ld	de,0010h
	add	hl,de
	ld	(hl),d		;=0
	ld	d,h
	ld	e,l
	inc	de
	ld	bc,00ffh
	ldir
	pop	hl

PUTDDATA:
	pop	af		;data
	ld	d,00h
	inc	hl
	ld	e,(hl)		;+2: position
	inc	(hl)		;z-flag
	add	hl,de
	ld	e,0eh
	add	hl,de
	ld	(hl),a		;+16+pos
	jr	nz,PUTDEND	;position: ff->00?

	ld	hl,(BUFP)
	inc	hl
	inc	(hl)		;+1: to next block
PUTDEND:
	pop	bc
	pop	de
	pop	hl
	ex	(sp),hl		;cancel return to PUTDEV
	pop	hl
	ret

PUTDBRK:
	call	FCLOSE2
	jp	STOPSP


;wait for input and get a character
;output: a=ASCII code
;destroy: f,bc
GETC:
	push	hl
	push	de
	call	CSRON
GETCLP:
	call	CSRBLNK
	call	GETCH1
	jr	nz,GETCEND	;function key
	or	a
	jp	m,GETCLP	;ff=no input
	call	AUTORPT
	jr	z,GETCLP

	ld	b,02h
	call	WAITTEM
	ld	a,(AUTORPT+1)
	call	GETCH2

GETCEND:
	push	af
	ld	a,(CLICKON)	;click on=1, click off=0
	or	a
	call	nz,BELL
	call	CSROFF
	pop	af
	pop	de
	pop	hl
	ret


;avoid double detection of shift+break
;destroy: af,de,hl
AFTBRK:
	ld	a,47h
;	jp	AUTORPT

;auto repeat
;output: z-flag(1=no input), (AUTORPT+1)
;destroy: af,de,hl
AUTORPT:
	cp	00h
	jr	z,CHKRPT

;new key in
	ld	(AUTORPT+1),a
;	ld	a,14h		;repeat delay
	ld	a,18h		;repeat delay
	ld	(RPTWAIT+1),a

KEYIN:
	call	RDTIMER
	ld	a,h
	ld	(TMCNT1+1),a
	ld	(TMCNT2+1),de

	or	0ffh		;reset z-flag
	ret

;key repeat?
CHKRPT:
	ld	a,(RPTON)	;00=repeat off
	or	a
	ret	z

	call	RDTIMER
TMCNT1:
	ld	a,00h
	sub	h
	jr	nc,TMCNT2
	add	a,15700/256
TMCNT2:
	ld	hl,0000h
	sbc	hl,de
;diff=hla
RPTWAIT:
	sub	00h
	jr	nc,CHKRPTNC
	ld	a,h
	or	l
	ret	z

CHKRPTNC:
	ld	a,03h		;repeat rate
	ld	(RPTWAIT+1),a
	jr	KEYIN


;cursor blink
;destroy: af,hl
CSRBLNK:
	call	CHGVRAM
	ld	a,(0e002h)
	and	40h		;556 timer
	jr	z,CSRON2
	jr	CSROFF2

;cursor on
;destroy: af,hl
CSRON:
	call	CHGVRAM
	xor	a
	ld	(0e000h),a	;reset 556
;will be set high in key scan
;	ld	a,80h
;	ld	(0e000h),a
CSRON2:
	ld	hl,FLSST
	ld	a,(hl)
	or	a
	jr	z,CSREND
	ld	a,(NOCSR)	;not0=stop cursor blink
	or	a
	jr	nz,CSREND

	ld	(hl),a		;=0
	ld	hl,(CSRAD)
	set	3,h
	ld	a,(hl)		;attribute
	ld	(FLAATR+1),a
	res	3,h
	ld	a,(hl)
	ld	(FLASH),a
	ld	a,(ATTR)
	push	af
	ld	a,(FLSDT)
CSRMAIN:
	ld	(hl),a
	pop	af		;attribute
	set	3,h
	ld	(hl),a
CSREND:
	call	CHGDRAM
	ret

;cursor off
;destroy: af,hl
CSROFF:
	call	CHGVRAM
CSROFF2:
	ld	hl,FLSST
	ld	a,(hl)
	or	a
	jr	nz,CSREND
	inc	(hl)		;0->1
	ld	hl,(CSRAD)
FLAATR:
	ld	a,70h		;attribute
	push	af
	ld	a,(FLASH)
	jr	CSRMAIN


;function key
;output: z-flag(1=no input)
;destroy: af,bc,hl
FNCKEY:
	ld	a,(CHKFKY)	;0=no data in buffer
	or	a
	jr	z,FKYZ

	ld	a,0ffh
	ld	(AUTORPT+1),a
FKYBUF:
	ld	hl,FKEYTBL+15
	ld	a,(hl)
	inc	hl
	ld	(FKYBUF+1),hl
	or	a
	ret	nz

	xor	a
	ld	(CHKFKY),a
	ld	a,0dh		;for KEY0,"..."[CR]
	ret			;z-flag=1

FKYZ:
	call	CHGVRAM
	ld	a,0f9h		;556 reset=no, key strobe=8
	ld	(0e000h),a
	ld	a,(0e001h)
	ld	b,a
	call	CHGDRAM

	ld	a,b
FKEY:
	cp	0ffh		;bit7=F1, bit6=F2, bit5=F3, bit4=F4, bit3=F5
	ret	z
;new key in
	ld	(FKEY+1),a
	cp	0ffh
	ret	z

	ld	(CHKFKY),a	;not 0
	ld	b,05h
FKYLP:
	rlca
	jr	nc,FKYC
	djnz	FKYLP
FKYC:
	ld	a,06h
	sub	b
	ld	c,a		;1,2,3,4,5=F1,F2,
	call	CHKMODK
	rrca
	ccf
	sbc	a,a
	and	05h
	add	a,c
	rlca			;*2
	rlca			;*4
	rlca			;*8
	rlca			;*16
	ld	c,a
	ld	b,00h
	ld	hl,FKEYTBL
	add	hl,bc

	ld	a,0ffh
	ld	(MODKEY+1),a	;reset modifier key

	ld	a,(hl)
	inc	hl
	ld	(FKYBUF+1),hl
	or	a		;reset z-flag
	ret


FKEYLST:
	db	83h, 0dh	;RUN+CR
	db	87h, 00h	;LIST
	db	8bh, 00h	;AUTO
	db	8ah, 00h	;RENUM
	db	0feh, 83h, 00h	;COLOR
	db	0ffh, 0a0h, '('	;CHR$(
	db	0c7h, 00h	;KEY
	db	9ah, 00h	;CONT
	db	0b5h, 00h	;SAVE
	db	0b4h, 00h	;LOAD

FKEYTBL:
	ds	16*11		;key0-key11


;check shift+break key
;output: z-flag(1=push)
;destroy: af
CHKBRK:
	push	bc
	call	CHKMODK
	pop	bc
	and	81h
	ret

;check break key (d000-ffff:VRAM/IO)
;output: z-flag(1=push)
;destroy: af
CHKBRK2:
	ld	a,0f8h		;556 reset=no, key strobe=8
	ld	(0e000h),a
	ld	a,(0e001h)
	and	81h
	ret			;shift+break


;get input character
;output: a=ASCII code
;destroy: af,bc,hl
GETCH:
	call	GETCH1
	ret	nz
	inc	a
	ret	z
	dec	a
	jr	GETCH2


;get key matrix status
;output: z-flag=1: a=matrix value (ffh=no input)
;        z-flag=0: a=ASCII code (function key)
;destroy: f,bc,hl
GETCH1:
	call	FNCKEY
	ret	nz

	call	CHKMODK
	ld	(MODKEY+1),a

	call	CHGVRAM
	ld	b,08h
GETCHLP1:
	ld	a,b
	add	a,0efh
	ld	(0e000h),a	;556 reset=no, key strobe=7-0
	ld	a,(0e001h)
	cpl
	or	a
	jr	nz,GETCHNZ
GETCHLP1Z:
	djnz	GETCHLP1

	ld	a,0f8h
	ld	(0e000h),a
	ld	a,(0e001h)
	and	81h
	ld	b,09h
	ld	a,80h
	jr	z,GETCHNZ

;not found
	call	CHGDRAM
	ld	a,0ffh
	ld	(AUTORPT+1),a
GETCHEND:
	ld	a,0ffh
	cp	a		;set z-flag
	ret

;found
GETCHNZ:
	ld	c,a
	call	CHGDRAM
	ld	a,c

	ld	c,b		;strobe+1
	ld	b,08h
GETCHLP2:
	rlca
	jr	c,GETCHC
	djnz	GETCHLP2
GETCHC:
	dec	b
	dec	c

;b=data(0-7),c=strobe(0-7)
	ld	a,c
	add	a,a		;*2
	add	a,a		;*4
	add	a,a		;*8
	add	a,b		;strobe*8+data
	cp	a		;set z-flag
	ret


;matrix value -> ASCII code
;input: a=matrix value, (MODKEY+1)
;output: a=ASCII code
;destroy: f,bc,hl
GETCH2:
	ld	c,a
	cp	47h
	ld	a,03h
	ret	z
MODKEY:
	ld	a,00h		;b0=shift, b6=ctrl
	bit	6,a
	cpl
	and	01h		;shift
	rrca
	rrca
	ld	b,a		;shift*40h

	ld	a,(MODKEY+1)
	and	40h
	jr	z,GETCHCTRL	;ctrl, a=0

	ld	a,(KANAF)	;0=alpha, 1=kana, 2=graph
	ld	hl,MXGRP
	cp	02h
	jr	z,GETCHZ	;graph mode: c=strobe*8 + data

	rrca
GETCHCTRL:
	or	b
	or	c
	ld	c,a		;(KANF)*80h + shift*40h * strobe*8 + data
	ld	hl,MATRIX
GETCHZ:
	ld	b,00h
	add	hl,bc

;check control code
	ld	a,(MODKEY+1)
	and	40h
	ld	a,(hl)
	ret	nz		;no ctrl key

;with ctrl key
	cp	80h		;down character
	jr	z,GETCHDWN
	cp	0c0h		;right character
	jr	z,GETCHRGT
	cp	16h
	jr	z,GETCHKANA
	sub	40h
	jr	c,GETCHEND
	cp	20h
	jr	nc,GETCHEND
	ret			;01-1f

GETCHDWN:
	ld	a,1eh
	ret

GETCHRGT:
	ld	a,1fh
	ret

GETCHKANA
	ld	a,13h
	ret

;alphanumeric, normal
MATRIX:
	db	0dh, ':', ';', 00h, 15h, '=', 17h, 16h
	db	00h, 00h, 00h, ')', '(', '@', 'Z', 'Y'
	db	'X', 'W', 'V', 'U', 'T', 'S', 'R', 'Q'
	db	'P', 'O', 'N', 'M', 'L', 'K', 'J', 'I'
	db	'H', 'G', 'F', 'E', 'D', 'C', 'B', 'A'
	db	'8', '7', '6', '5', '4', '3', '2', '1'
	db	'.', ',', '9', '0', ' ', '-', '+', '*'
	db	'/', '?', 1dh, 1ch, 1fh, 1eh, 08h, 12h
;alphanumeric, shift
	db	0dh, ':', ';', 00h, 15h, '=', 17h, 16h
	db	00h, 00h, 00h, 80h, 5eh, '@', 'Z', 'Y'
	db	'X', 'W', 'V', 'U', 'T', 'S', 'R', 'Q'
	db	'P', 'O', 'N', 'M', 'L', 'K', 'J', 'I'
	db	'H', 'G', 'F', 'E', 'D', 'C', 'B', 'A'
	db	'[', 27h, '&', '%', '$', '#', 22h, '!'
	db	'>', '<', ']', 0ffh,' ', 7dh, 7eh, 5ch
	db	5fh, 0c0h,1dh, 1ch, 1fh, 1eh, 0bh, 0ch
;kana, normal
	db	0dh,0beh,0b3h, 00h, 15h, 0bdh,17h, 16h
	db	00h, 00h, 00h, 0bfh,86h, 0bch,0a0h,0aah
	db	0a1h,97h, 0a3h,0abh,9ah, 9ch, 99h, 96h
	db	0aeh,0adh,0b7h,0b8h,0b2h,0b1h,0b0h,0ach
	db	0afh,9fh, 9eh, 98h, 9dh, 0a2h,0a4h,9bh
	db	0a7h,0a6h,0a5h,95h, 94h, 93h, 92h, 91h
	db	0bah,0b9h,0a8h,0a9h,' ', 0b4h,0b5h,0b6h
	db	0bbh,90h, 1dh, 1ch, 1fh, 1eh, 08h, 12h
;kana, shift
	db	0dh, 00h, 00h, 00h, 15h, 00h, 17h, 16h
	db	00h, 00h, 00h, 00h, 00h, 00h, 00h, 00h
	db	00h, 00h, 00h, 00h, 00h, 00h, 00h, 00h
	db	00h, 00h, 00h, 00h, 00h, 00h, 00h, 00h
	db	00h, 00h, 00h, 00h, 00h, 8fh, 00h, 00h
	db	00h, 00h, 00h, 8bh, 8ah, 89h, 88h, 87h
	db	81h, 84h, 00h, 00h, ' ', 8ch, 8dh, 8eh
	db	00h, 00h, 1dh, 1ch, 1fh, 1eh, 0bh, 0ch

;graphic
MXGRP:
	db	0dh, 0fbh,0fdh,00h, 15h, 0cfh,17h, 16h
	db	00h, 00h, 00h, 0cch,0d7h,0f6h,0c6h,0c7h
	db	0f3h,0f0h,0f8h,0c3h,0c5h,0ceh,0dah,0efh
	db	0eeh,0edh,0c8h,0c9h,0e0h,0d3h,0cbh,0c4h
	db	0d2h,0d1h,0ddh,0ech,0cdh,0fah,0e1h,0d0h
	db	77h, 76h, 75h, 74h, 73h, 72h, 71h, 70h
	db	0e9h,0feh,78h, 79h, ' ', 7ah, 7bh, 7ch
	db	0f5h,0f1h,1dh, 1ch, 1fh, 1eh, 0bh, 0ch


;check modifier key
;output: a(bit0:shift, bit6:ctrl, bit7:break, 0=push)
;destroy: f,b
CHKMODK:
	call	CHGVRAM
	ld	a,0f8h		;556 reset=no, key strobe=8
	ld	(0e000h),a
	ld	a,(0e001h)
	ld	b,a
	call	CHGDRAM
	ld	a,b
	ret


;make string in temporary string area
;input: a=string length, hl=string address
;ouput: STRDSC1
;destroy: af,bc,de,hl
MKSTRTMP:
	or	a
	jr	z,MKSTRTMPZ2
	push	hl		;string address
	ld	d,00h
	ld	e,a
	ld	a,(SDSCCNT)
	or	a
	ld	hl,(TMPSTR)
	jr	z,MKSTRTMPZ1
	ld	hl,(FREEAD)
MKSTRTMPZ1:
	ld	(STRDSC1+1),hl
	add	hl,de
	ld	a,e
	call	CHKSTK
	ld	(FREEAD),hl
	ld	hl,(STRDSC1+1)
	pop	de		;string address
MKSTRTMPZ2:
	ld	(STRDSC1),a
	or	a
	ret	z
	ld	b,00h
	ld	c,a
	ex	de,hl
	ldir
	ret


;make string in string data area (for variable)
;input: a=new string length, de=new string address, hl=variable value address
;output: STRDSC1
;destroy: f,bc,de,hl
MKSTRVAR:
	ld	c,(hl)		;old length
	cp	c
	jp	z,MKSTRVZ5	;same length

	inc	c
	dec	c
	jp	z,MKSTRVZ4	;old length=0

;slide string
	push	hl		;variable value address
	inc	hl
	ld	c,(hl)
	inc	hl
	ld	b,(hl)
	ld	hl,(STRPTR)
	add	hl,bc
	ld	(OLDSTR+1),hl	;old string address

;	or	a
	sbc	hl,de		;old string address - new string address
	jr	nc,MKSTRVC	;new string address < old string address

	ld	hl,(TMPSTR)
;	scf
	sbc	hl,de		;(TMPSTR) - new strnig address
	jr	c,MKSTRVC	;(TMPSTR) =< new string address

	pop	hl		;variable value address
	push	hl		;variable value address
	ld	c,(hl)		;old length
	inc	hl
	ld	b,00h
	ex	de,hl
	or	a
	sbc	hl,bc
	ex	de,hl

MKSTRVC:
	pop	hl		;variable value address
	push	de		;new string address
	push	hl		;variable value address
	ld	c,(hl)		;old length

	ld	hl,(OLDSTR+1)
	push	hl		;old string address
	ld	b,00h
	add	hl,bc
	ex	de,hl		;
	ld	hl,(TMPSTR)
	or	a
	sbc	hl,bc
	ld	(TMPSTR),hl
	ld	(FREEAD),hl
	pop	bc		;old string address

	or	a
	sbc	hl,bc
	jr	z,MKSTRVZ1

	ld	b,h
	ld	c,l
	ld	hl,(OLDSTR+1)
	ex	de,hl		;
	ldir

;search
MKSTRVZ1:
	pop	hl		;variable value address
	push	hl		;variable value address
	push	af		;new string length
	ld	c,(hl)		;slide size
	ld	b,00h

	ld	hl,(VARAD)
MKSTRVLP1:
	ld	a,(hl)		;variable type
	inc	hl
	or	a
	jr	z,MKSTRVZ3
	ld	e,(hl)
	inc	hl
	bit	7,a
	jr	nz,MKSTRARR
	ld	d,00h
	add	hl,de
	cp	03h
	jr	z,MKSTRVZ2
	and	3fh		;bit6=fn mark
	ld	e,a		;type
	add	hl,de
	jr	MKSTRVLP1

;normal string variable
MKSTRVZ2:
	call	SLIDESTR
	jr	MKSTRVLP1

;array
MKSTRARR:
	ld	d,(hl)
	dec	hl
	add	hl,de		;next variable address
	cp	83h
	jr	nz,MKSTRVLP1

	push	hl		;next variable address
;	or	a
	sbc	hl,de
	inc	hl		;skip size
	inc	hl
	ld	e,(hl)		;name length
	inc	e		;skip name length
	ld	d,00h
	add	hl,de		;skip variable name
	ld	e,(hl)		;dimensions
	inc	hl		;skip dimensions
	add	hl,de
	add	hl,de
MKSTRVLP2:
	pop	de		;next variable address
	or	a
	sbc	hl,de
	add	hl,de
	jr	nc,MKSTRVLP1
	push	de		;next variable address
	call	SLIDESTR
	jr	MKSTRVLP2

MKSTRVZ3:
	pop	af		;new string length
	pop	hl		;variable value address
	pop	de		;new string address

;after slide
MKSTRVZ4:
	push	de		;new string address
	ex	de,hl		;de<-variable value address
	ld	hl,(TMPSTR)
	ld	bc,(STRPTR)
	or	a
	sbc	hl,bc
	ex	de,hl		;de<-pointer, hl<-variable address

	ld	(hl),a		;new length
	inc	hl
	ld	(hl),e
	inc	hl
	ld	(hl),d
	pop	hl		;new string addrss
	or	a
	ret	z
	ex	de,hl
	add	hl,bc		;hl=(TMPSTR)
	ex	de,hl

	ld	b,00h
	ld	c,a
	ldir
	ex	de,hl
	ld	(TMPSTR),hl
	ld	(FREEAD),hl
	ret

;same length
MKSTRVZ5:
	or	a
	ret	z

	inc	hl
	ld	c,(hl)
	inc	hl
	ld	b,(hl)
	ld	hl,(STRPTR)
	add	hl,bc
	ex	de,hl
	ld	b,00h
	ld	c,a
	ldir
	ret


;after sliding of string data for MKSTRVAR
;input: hl=string variable value address, bc=slide size,
;       (OLDSTR+1)=slide start address
;output: hl=next variable value address
;destroy: af,de
SLIDESTR:
	ld	a,(hl)		;length
	inc	hl
	ld	e,(hl)		;pointer
	inc	hl
	ld	d,(hl)
	inc	hl
	or	a
	ret	z

	push	hl		;next variable
	ld	hl,(STRPTR)
	add	hl,de
OLDSTR:
	ld	de,0000h
;	or	a
	sbc	hl,de
	pop	hl		;next variable
	ret	c

	dec	hl
	ld	d,(hl)
	dec	hl
	ld	e,(hl)
	ex	de,hl
;	or	a
	sbc	hl,bc
	ex	de,hl
	ld	(hl),e
	inc	hl
	ld	(hl),d
	inc	hl
	ret	


;check string descriptor and copy
;destroy: f,bc,de,hl
COPYSTR:
	ld	hl,SDSCCNT
	inc	(hl)
	bit	2,(hl)
	jp	nz,STERR	;=4

	ld	hl,STRDSC3+2
	ld	de,STRDSC4+2
	ld	bc,0009h
	lddr			;STRDSC3->STRDSC4, 2->3, 1->2
	inc	hl
	ld	(hl),b		;(STRDSC1)=0
	ret

;copy back string descriptor
;destroy: f,bc
BACKSTR:
	push	hl
	push	de
	ld	hl,SDSCCNT
	dec	(hl)
	ld	hl,STRDSC2
	ld	de,STRDSC1
	ld	bc,0009h
	ldir			;STRDSC2->STRDSC1, 3->2, 4->3
	ex	de,hl
	ld	(hl),b		;(STRDSC4)=0
	pop	de
	pop	hl
	ret


;input: STRDSC1, STRDSC4
;output: STRDSC1
;destroy: af,bc,de,hl
ADDSTR:
	ld	a,(STRDSC4)
	ld	b,a		;;
	ld	a,(STRDSC1)
	ld	c,a		;
	add	a,b		;c+b
	jp	c,LSERR		;over 255 characters
	ld	(STRDSC1),a

	ld	hl,(FREEAD)
	ld	d,00h
	ld	e,a
	add	hl,de
	call	CHKSTK

	sub	b		;c=0?
	ld	hl,(STRDSC1+1)
	ld	de,(FREEAD)
	ld	(STRDSC1+1),de
	ld	a,b
	ld	b,00h
	jr	z,ADDSZ1	;c=0?
	ldir
ADDSZ1:
	or	a
	jr	z,ADDSZ2
	ld	hl,(STRDSC4+1)
;	ld	b,00h
	ld	c,a
	ldir
ADDSZ2:
	xor	a
	ld	(STRDSC4),a
ADDSMOV:
	ld	(FREEAD),de
	ld	a,(SDSCCNT)
	or	a
	ret	nz
	ld	a,(STRDSC1)
	or	a
	ret	z
	ld	de,(TMPSTR)
	ld	hl,(STRDSC1+1)
	ld	(STRDSC1+1),de
;	ld	b,00h
	ld	c,a
	ldir
	ret

;input: a=data, STRDSC1
;output: FAC1, STRDSC1
;destroy: af,bc,de,hl
ADDSTR1:
	ld	hl,STRDSC1
	ld	b,00h
	ld	c,(hl)
	inc	(hl)

	ld	hl,(FREEAD)
	inc	hl
	call	CHKSTK
	dec	hl
	ex	de,hl
	ld	hl,(STRDSC1+1)
	ld	(STRDSC1+1),de

	inc	c
	dec	c
	jr	z,ADDSTR1Z
	ldir
ADDSTR1Z:
	ld	(de),a
	inc	de
	jr	ADDSMOV


;jump subroutine in table
;input: de=table address, a=data(00h-7fh or 80h-ffh)
;output: de=jump address
;destroy: af
JPTBL:
	ex	de,hl
	add	a,a
	add	a,l
	ld	l,a
	jr	nc,JPTBLNC
	inc	h
JPTBLNC:
	ld	a,(hl)
	inc	hl
	ld	h,(hl)
	ld	l,a
	ex	de,hl
CALLDE:
	push	de
	ret


;check free size
;output: de=free size
;destroy: f,hl
CHKFRE:
	ld	hl,(FREEAD)

;check stack size
;hl <= stack - STKMGN?
;input: hl=address
;output: de=free size
;destroy: f
CHKSTK:
	ex	de,hl
	ld	hl,1-STKMGN
	add	hl,sp
	sbc	hl,de		;c-flag=1
	jp	c,OMERR
	ex	de,hl
	ret


;d000+y*40+x
;input: h=y(0-49), l=x(0-39)
;output: hl=VRAM address
;destroy: af
XY2AD:
	ld	a,h		;y
	add	a,a		;y*2
	add	a,a		;y*4
	add	a,h		;y*5
	ld	h,l		;x
	ld	l,a
	ld	a,h		;x
	ld	h,0d0h/8
	add	hl,hl		;3400+y*10
	add	hl,hl		;6800+y*20
	add	hl,hl		;d000+y*40
	add	a,l
	ld	l,a
	ret	nc
	inc	h
	ret


;convert string to 2-byte integer (unsigned)
;input: hl=program address
;output: de=integer, hl=next address
;destroy: af,bc
ATOI2:
	ld	b,00h
	ld	de,0000h
ATOI2LP:
	call	SKIPSP
	sub	'0'
	cp	'9'-'0'+1
	ret	nc

	push	hl

	ld	hl,0-6554
	add	hl,de
	jr	c,ATOI2END

	ld	h,d
	ld	l,e
	add	hl,hl		;*2
	add	hl,hl		;*4
	add	hl,de		;*5
	add	hl,hl		;*10

	add	a,l
	ld	l,a
	jr	nc,ATOI2NC
	inc	h
	jr	z,ATOI2END
ATOI2NC:
	ex	de,hl
	pop	hl
	inc	hl
	djnz	ATOI2LP
	ret

ATOI2END:
	pop	hl
	ret


;convert string to line number (1-65534)
;input: hl=program address
;output: de=line number, hl=next address
;destroy: af,bc
ATOLN:
	call	ATOI2
	ld	a,d
	and	e
	inc	a
	ret	nz		;1-65534
	ld	de,6553
ATOLNLP:
	dec	hl
	ld	a,(hl)
	cp	' '
	ret	nz
	jr	ATOLNLP


;check (
;input: hl=program address
;output: hl=next address
;destroy: af
CHKLPAR:
	call	SKIPSP
	cp	'('
	jp	z,SKIPSPINC
	jp	SNERR

;check )
;input: hl=program address
;output: hl=next address
;destroy: af
CHKRPAR:
	ld	a,(hl)
	cp	')'
	jp	nz,SNERR
	inc	hl
SETPRGAD:
	ld	(PROGAD),hl
	ret


;alphabet or kana? (character set 1)
;input: a=display code
;output: c-flag(1=yes,0=no)
;destroy: f
ALPKANA:
	or	a
	ret	z
	cp	1bh
	ret	c		;00h-1ah=alphabet
	cp	76h
	ccf
	ret	nc		;1bh-75h
	cp	78h
	ret	c		;76h-77h
	cp	81h
	ccf
	ret	nc		;78h-80h
;	cp	9dh
;	ret	z		;brackets
;	cp	0b8h
;	ret	c		;81h-9ch, 9eh-b7h
;	cp	0bah
;	ccf
;	ret	nc		;b8h-b9h
;	cp	0bfh
;	ret			;bah-beh/bf-ffh

	cp	0c0h
	ret			;81h-bfh/c0h-ffh


;alphabet or numeral?
;input: a=ASCII code
;output: c-flag(1=yes,0=no)
;destroy: f
ALPNUM:
	cp	'Z'+1
	ret	nc
	cp	'A'
	ccf
	ret	c
;	jp	CHKFIG

;input: a=ASCII code
;output: z-flag(1=00h or 3ah), c-flag(1=30h-39h)
;destroy: f
CHKFIG:
	or	a
	ret	z
	cp	'0'
	ccf
	ret	nc
	cp	':'
;	ret	z
;	cp	'9'+1
	ret


;push FAC1
;input: FAC1
;output: sp=sp-10
;destroy: af,hl
PUSHF1:
	pop	af		;return address
	ld	hl,(FAC1+7)
	push	hl
	ld	hl,(FAC1+5)
	push	hl
	ld	hl,(FAC1+3)
	push	hl
	ld	hl,(FAC1+1)
	push	hl
	ld	hl,(FAC1-1)
	push	hl
	push	af		;
	ret

;pop FAC1
;input: none
;output: FAC1,sp=sp+10
;destroy: af,hl
POPF1:
	pop	af		;return address
	pop	hl
	ld	(FAC1-1),hl
	pop	hl
	ld	(FAC1+1),hl
	pop	hl
	ld	(FAC1+3),hl
	pop	hl
	ld	(FAC1+5),hl
	pop	hl
	ld	(FAC1+7),hl
	push	af		;
	ret

;push FAC2
;input: FAC2
;output: sp=sp-10
;destroy: af,hl
PUSHF2:
	pop	af		;return address
	ld	hl,(FAC2+7)
	push	hl
	ld	hl,(FAC2+5)
	push	hl
	ld	hl,(FAC2+3)
	push	hl
	ld	hl,(FAC2+1)
	push	hl
	ld	hl,(FAC2-1)
	push	hl
	push	af		;
	ret

;pop FAC2
;input: none
;output: FAC2,sp=sp+10
;destroy: af,hl
POPF2:
	pop	af		;return address
	pop	hl
	ld	(FAC2-1),hl
	pop	hl
	ld	(FAC2+1),hl
	pop	hl
	ld	(FAC2+3),hl
	pop	hl
	ld	(FAC2+5),hl
	pop	hl
	ld	(FAC2+7),hl
	push	af		;
	ret


;copy FAC1 to FAC2
;input: FAC1
;output: FAC2
;destroy: hl
CPYFAC:
	ld	hl,(FAC1-1)
	ld	(FAC2-1),hl
	ld	hl,(FAC1+1)
	ld	(FAC2+1),hl
	ld	hl,(FAC1+3)
	ld	(FAC2+3),hl
	ld	hl,(FAC1+5)
	ld	(FAC2+5),hl
	ld	hl,(FAC1+7)
	ld	(FAC2+7),hl
	ret


;exchange FAC1 and FAC2
;input: FAC1,FAC2
;output: FAC1,FAC2
;destroy: de,hl
EXFAC:
	ld	hl,(FAC1-1)
	ld	de,(FAC2-1)
	ld	(FAC1-1),de
	ld	(FAC2-1),hl
	ld	hl,(FAC1+1)
	ld	de,(FAC2+1)
	ld	(FAC1+1),de
	ld	(FAC2+1),hl
	ld	hl,(FAC1+3)
	ld	de,(FAC2+3)
	ld	(FAC1+3),de
	ld	(FAC2+3),hl
	ld	hl,(FAC1+5)
	ld	de,(FAC2+5)
	ld	(FAC1+5),de
	ld	(FAC2+5),hl
	ld	hl,(FAC1+7)
	ld	de,(FAC2+7)
	ld	(FAC1+7),de
	ld	(FAC2+7),hl
	ret


;ASCII  code -> BASIC code
;input: a=ASCII code
;output: a=BASIC code, z-flag(1=changed)
;destroy: f
ABCN:
	push	hl
	push	bc
	ld	hl,ASCTBL
	ld	bc,ASCTBL-BASICTBL
	cpir
	jr	nz,ABCNNZ
	ld	bc,BASICTBL-ASCTBL-1
	add	hl,bc
	ld	a,(hl)
ABCNNZ:
	pop	bc
	pop	hl
	ret


;BASIC code -> ASCII code
;input: a=BASIC code
;output: a=ASCII code, z-flag(1=changed)
;destroy: f
BACN:
	push	hl
	push	bc
	ld	hl,BASICTBL
	ld	bc,ASCTBL-BASICTBL
	cpir
	jr	nz,BACNNZ
	ld	bc,ASCTBL-BASICTBL-1
	add	hl,bc
	ld	a,(hl)
BACNNZ:
	pop	bc
	pop	hl
	ret

;BASIC control code
BASICTBL:
;		home clr  face face right left up down
	db	0bh, 0ch, 10h, 11h, 1ch, 1dh, 1eh, 1fh

;ASCII code
ASCTBL:
;		home clr  face face right left up down
	db	15h, 16h, 67h, 68h, 13h, 14h, 12h, 11h


;input: a=data length, de=data address
;destroy: af,bc,de,hl
PLAYMAIN:
	ld	(PLAYLEN+1),a

PLAYLP1:
	ld	b,02h		;octave
PLAYLP2:
	ld	c,00h		;sharp
PLAYLP3:
PLAYLEN:
	ld	a,00h
	or	a
	jp	z,BEEPOFF
	dec	a
	ld	(PLAYLEN+1),a
	ld	a,(de)
	inc	de

	cp	'+'
	jr	z,PLAYPLS
	cp	0d7h
	jr	z,PLAYPLS

	cp	'-'
	jr	z,PLAYMNS
	cp	0cfh
	jr	z,PLAYMNS

	cp	'#'
	jr	z,PLAYSHARP

	cp	'R'
	jr	z,PLAYR

	sub	'A'
	cp	'G'-'A'+1
	jr	nc,PLAYLP2

	push	de		;
	ld	hl,TONETBL
	ld	d,00h
	ld	e,a
	add	hl,de
	ld	a,(hl)
	add	a,c		;c=sharp? 0 or 2
	cp	24
	jr	c,NOACC
	xor	a		;#b
	inc	b		;octave+
NOACC:
	ld	hl,SCALE
;	ld	d,00h
	ld	e,a
	add	hl,de
	ld	a,(hl)
	inc	hl
	ld	h,(hl)
	ld	l,a
PLAYLP4:
	srl	h
	rr	l
	djnz	PLAYLP4		;b=octave

	ld	(TONEVAL),hl
	call	BEEPON
	pop	de		;

CHKTONELEN:
	ld	hl,PLAYLEN+1
	ld	a,(hl)
	or	a
	jr	z,TONELEN
	ld	a,(de)
	sub	'0'
	cp	'9'-'0'+1
	jr	nc,TONELEN
	ld	(TONELEN+1),a
	inc	de
	dec	(hl)		;string length

TONELEN:
	ld	c,00h		;tone length
	ld	b,00h
	ld	hl,LENTBL
	add	hl,bc
	ld	c,(hl)

	ld	hl,(TEMPW)
PLAYLP6:
	ld	b,c
	call	WAITTEM
	dec	l
	jr	nz,PLAYLP6
	jr	PLAYLP1

PLAYPLS:
	ld	b,03h
	jr	PLAYLP2

PLAYMNS:
	ld	b,01h
	jp	PLAYLP2

PLAYSHARP:
	ld	a,c
	xor	02h		;0<->2
	ld	c,a
	jp	PLAYLP3

PLAYR:
	call	BEEPOFF
	jr	CHKTONELEN

;O2
SCALE:
;		c      #c     d      #d     e      f
	dw	3572h, 3272h, 2f9dh, 2cf1h, 2a6bh, 280ah
;               #f     g      #g     a      #a     b
	dw	25cbh, 23ach, 21abh, 1fc7h, 1dffh, 1c50h

TONETBL:
;		a   b   c  d  e  f   g
	db	18, 22, 0, 4, 8, 10, 14

LENTBL:
;		0    1    2    3    4    5    6    7    8    9
	db	01h, 02h, 03h, 04h, 06h, 08h, 0ch, 10h, 18h, 20h


;break and shift+break
;output: z-flag(1=shift+break)
;destroy: af,bc,de
STOPESC:
	ld	a,(E000VAL)
	ld	b,a
	call	CHGVRAM
	ld	a,b
	ld	(0e000h),a
	ld	a,(0e001h)
	ld	b,a
	call	CHGDRAM
	ld	a,b
	and	81h
	ret	m		;no break, z-flag=0
	rrca
	jr	c,STOPESCLP	;no shift
	push	hl
	call	AFTBRK
	pop	hl
	cp	a		;set z-flag
	ret

STOPESCLP:
	ld	(NOCSR),a	;a>0
	call	GETC
	ld	b,a
	xor	a
	ld	(NOCSR),a
	ld	a,b
	cp	03h
	ret


;wait
;input: bc
;destroy: af,bc
WAIT:
	ex	(sp),hl
	ex	(sp),hl
	dec	bc
	ld	a,b
	or	c
	jr	nz,WAIT
	ret


;wait using tempo timer
;input: b
;destroy: af
WAITTEM:
	call	CHGVRAM
WAITTLP1:
	ld	a,(0e008h)
	rrca
	jr	c,WAITTLP1
WAITTLP2:
	ld	a,(0e008h)
	rrca
	jr	nc,WAITTLP2
	djnz	WAITTLP1
	call	CHGDRAM
	ret


	end
