/* alternative functions to cureses and editline         */
/*  by AKIKAWA, Hisashi 2015-2025                        */
/* This software is released under 2-clause BSD license. */

#include <stdio.h>
#include <stdarg.h>
#include <windows.h>
#define NO_OLDNAMES
#include <conio.h>
#include "win.h"

SCREEN *SP = NULL;
WINDOW *stdscr = NULL;

wchar_t **text, **text_old;
int **attr, **attr_old;
DWORD console_mode;

WINDOW *initscr()
{
  int x, y;
  HANDLE input_handle;
  CONSOLE_SCREEN_BUFFER_INFO info;

  if ((SP = malloc(sizeof(SCREEN))) == NULL) {
    return NULL;
  }
  SP->echo = 1;
  SP->autocr = 1;

  if ((stdscr = malloc(sizeof(WINDOW))) == NULL) {
    return NULL;
  }
  stdscr->winhandle = GetStdHandle(STD_OUTPUT_HANDLE);
  GetConsoleScreenBufferInfo(stdscr->winhandle, &info);
  stdscr->_curx = 0;
  stdscr->_cury = 0;
  stdscr->_maxx = info.srWindow.Right - info.srWindow.Left + 1;
  stdscr->_maxy = info.srWindow.Bottom - info.srWindow.Top + 1;
  stdscr->_attrs = A_NORMAL;
  stdscr->_top = 0;
  stdscr->_bottom = 24;
  stdscr->_scroll = 0;
  stdscr->_nodelay = 0;
  stdscr->_use_keypad = 0;

  input_handle = GetStdHandle(STD_INPUT_HANDLE);
  GetConsoleMode(input_handle, &console_mode);
  SetConsoleMode(input_handle,
		 ENABLE_WINDOW_INPUT
		 | ENABLE_QUICK_EDIT_MODE
		 | ENABLE_EXTENDED_FLAGS
		 | ENABLE_PROCESSED_INPUT);


  text = malloc(sizeof(wchar_t *) * stdscr->_maxy);
  text_old = malloc(sizeof(wchar_t *) * stdscr->_maxy);
  attr = malloc(sizeof(int *) * stdscr->_maxy);
  attr_old = malloc(sizeof(int *) * stdscr->_maxy);
  for (y = 0; y < stdscr->_maxy; y++) {
    text[y] = malloc(sizeof(wchar_t) * stdscr->_maxx);
    text_old[y] = malloc(sizeof(wchar_t) * stdscr->_maxx);
    attr[y] = malloc(sizeof(int) * stdscr->_maxx);
    attr_old[y] = malloc(sizeof(int) * stdscr->_maxx);
    for (x = 0; x < stdscr->_maxx; x++) {
      text[y][x] = ' ';
      text_old[y][x] = 0;
      attr[y][x] = A_NORMAL;
    }
  }

  refresh();
  return  stdscr;
}

int endwin()
{
  COORD pos;
  CONSOLE_SCREEN_BUFFER_INFO info;
  if (!stdscr) {
    return ERR;
  }
  GetConsoleScreenBufferInfo(stdscr->winhandle, &info);
  pos.X = 0;
  pos.Y = info.dwSize.Y - 1;
  SetConsoleCursorPosition(stdscr->winhandle, pos);
  curs_set(1);
  SetConsoleMode(GetStdHandle(STD_INPUT_HANDLE), console_mode);
  return OK;
}

int wresize(WINDOW *win, int lines, int columns)
{
  int x, y;
  int x_small, y_small;
  wchar_t **text_new;
  int **attr_new;

  if (!win || lines <= 0 || columns <= 0) {
    return ERR;
  }
  x_small = (win->_maxx < columns) ? win->_maxx : columns;
  y_small = (win->_maxy < lines  ) ? win->_maxy : lines;

  if ((text_new = malloc(sizeof(wchar_t *) * lines)) == NULL
      || (attr_new = malloc(sizeof(int *) * lines)) == NULL) {
    return ERR;
  }
  for (y = 0; y < lines; y++) {
    if ((text_new[y] = malloc(sizeof(wchar_t) * columns)) == NULL
	|| (attr_new[y] = malloc(sizeof(int    ) * columns)) == NULL) {
      return ERR;
    }
    for (x = 0; x < columns; x++) {
      text_new[y][x] = ' ';
      attr_new[y][x] = A_NORMAL;
    }
    if (y < y_small) {
      memcpy(text_new[y], text[y], sizeof(wchar_t) * x_small);
      memcpy(attr_new[y], attr[y], sizeof(int    ) * x_small);
    }
  }
  for (y = 0; y < win->_maxy; y++) {
    free(text[y]);
    free(attr[y]);
    free(text_old[y]);
    free(attr_old[y]);
  }
  free(text);
  free(attr);
  free(text_old);
  free(attr_old);
  text = text_new;
  attr = attr_new;

  if ((text_old = malloc(sizeof(wchar_t *) * lines)) == NULL
      || (attr_old = malloc(sizeof(int *    ) * lines)) == NULL) {
    return ERR;
  }
  for (y = 0; y < lines; y++) {
    if ((text_old[y] = malloc(sizeof(wchar_t) * columns)) == NULL
	|| (attr_old[y] = malloc(sizeof(int    ) * columns)) == NULL) {
      return ERR;
    }
    memset(text_old[y], 0, columns);
  }
  win->_maxx = columns;
  win->_maxy = lines;
  if (win->_curx >= win->_maxx) {
    win->_curx = win->_maxx - 1;
  }
  if (win->_cury >= win->_maxy) {
    win->_cury = win->_maxy - 1;
  }
  redrawwin(win);
  refresh();

  return OK;
}

int nonl()
{
  if (SP) {
    SP->autocr = 0;
    return OK;
  } else {
    return ERR;
  }
}

int noecho()
{
  if (SP) {
    SP->echo = 0;
    return OK;
  } else {
    return ERR;
  }
}

int raw()
{
  HANDLE handle;
  DWORD mode;

  handle = GetStdHandle(STD_INPUT_HANDLE);
  GetConsoleMode(handle, &mode);
  SetConsoleMode(handle, mode & ~ENABLE_PROCESSED_INPUT);
  return OK;
}

int noraw()
{
  HANDLE handle;
  DWORD mode;

  handle = GetStdHandle(STD_INPUT_HANDLE);
  GetConsoleMode(handle, &mode);
  SetConsoleMode(handle, mode | ENABLE_PROCESSED_INPUT);
  return OK;
}

int keypad(WINDOW *win, int bf)
{
  if (win) {
    win->_use_keypad = bf;
    return OK;
  } else {
    return ERR;
  }
}

int nodelay(WINDOW *win, int bf)
{
  if (win) {
    win->_nodelay = bf;
    return OK;
  } else {
    return ERR;
  }
}

int scrollok(WINDOW *win, int bf)
{
  if (win) {
    win->_scroll = bf;
    return OK;
  } else {
    return ERR;
  }
}

int set_escdelay(int size)
{
  return TRUE;
}

int curs_set(int visibility)
{
  CONSOLE_CURSOR_INFO info;
  int prev;

  GetConsoleCursorInfo(stdscr->winhandle, &info);
  prev = info.bVisible ? 1 : 0;

  switch (visibility) {
  case 0:
    info.bVisible = FALSE;
    break;
  case 1:
  case 2:
    info.bVisible = TRUE;
    break;
  default:
    return ERR;
  }
  SetConsoleCursorInfo(stdscr->winhandle, &info);
  return prev;
}

int setscrreg(int top, int bot)
{
  if (!stdscr || top < 0 || top > bot || bot >= stdscr->_maxy) {
    return ERR;
  }
  stdscr->_top = top;
  stdscr->_bottom = bot;
  return OK;
}

int getmaxy(WINDOW *win)
{
  CONSOLE_SCREEN_BUFFER_INFO info;
  int size_x, size_y;

  if (!win) {
    return ERR;
  }
  GetConsoleScreenBufferInfo(win->winhandle, &info);
  size_x = info.srWindow.Right - info.srWindow.Left + 1;
  size_y = info.srWindow.Bottom - info.srWindow.Top + 1;

  if (win->_maxx != size_x || win->_maxy != size_y) {
    wresize(win, size_y, size_x);
  }
  return win->_maxy;
}

int move(int y, int x)
{
  if (!stdscr || x < 0 || y < 0 || x >= stdscr->_maxx || y >= stdscr->_maxy) {
    return ERR;
  }
  stdscr->_curx = x;
  stdscr->_cury = y;
  return OK;
}

int addch(const chtype ch)
{
  int x, y;
  int c = ch & 0xffff;
  int attrs = ch & 0xffff0000;

  if (!stdscr) {
    return ERR;
  }
  if (c == '\n') {
    stdscr->_curx = 0;
    stdscr->_cury++;
  } else {
    if (c & 0xff80) {
      if (stdscr->_curx >= stdscr->_maxx - 1) {
	stdscr->_curx = 0;
	stdscr->_cury++;
      }
      text[stdscr->_cury][stdscr->_curx] = c;
      attr[stdscr->_cury][stdscr->_curx] = attrs | stdscr->_attrs;
      stdscr->_curx++;
      text[stdscr->_cury][stdscr->_curx] = ' ';
      attr[stdscr->_cury][stdscr->_curx] = attrs | stdscr->_attrs;
      stdscr->_curx++;
    } else {
      text[stdscr->_cury][stdscr->_curx] = c;
      attr[stdscr->_cury][stdscr->_curx] = attrs | stdscr->_attrs;
      stdscr->_curx++;
    }
  }
  if (stdscr->_curx >= stdscr->_maxx) {
    stdscr->_curx = 0;
    if (stdscr->_cury < stdscr->_maxy - 1) {
      stdscr->_cury++;
    }
  }

  if (stdscr->_cury > stdscr->_bottom) {
    stdscr->_cury = stdscr->_bottom;
    if (stdscr->_scroll) {
      for (y = stdscr->_top; y < stdscr->_bottom; y++) {
	memcpy(text[y], text[y + 1], sizeof(text[0][0]) * stdscr->_maxx);
	memcpy(attr[y], attr[y + 1], sizeof(attr[0][0]) * stdscr->_maxx);
      }
      for (x = 0; x < stdscr->_maxx; x++) {
	text[y][x] = ' ';
	attr[y][x] = A_NORMAL;
      }
      refresh();
    }
  }
  
  return OK;
}

int mvaddch(int y, int x, const chtype ch)
{
  if (move(y, x) == ERR) {
    return ERR;
  }
  return addch(ch);
}

int mvadd_wch(int y, int x, const cchar_t *wch)
{
  int result;
  if (wch == NULL) {
    return ERR;
  }
  attron(wch->attr);
  result = mvaddch(y, x, wch->chars[0]);
  attroff(wch->attr);
  return result;
}

int add_wch(const cchar_t *wch)
{
  int result;
  if (wch == NULL) {
    return ERR;
  }
  attron(wch->attr);
  result = addch(wch->chars[0]);
  attroff(wch->attr);
  return result;
}

int setcchar(cchar_t *wch, const wchar_t *wc, const int attrs, short color_pair, void *opts)
{
  int i;
  if (!wch || !wc) {
    return ERR;
  }
  wch->attr = attrs;
  for (i = 0; i < 5; i++) {
    wch->chars[i] = wc[i];
  }
  return OK;
}

int vwprintw(WINDOW *win, const char *fmt, va_list args)
{
  int i;
  int length;
  char printbuf[512];

  length = vsnprintf(printbuf, sizeof(printbuf), fmt, args);
  va_end(args);
  for (i = 0; i < length; i++) {
    addch(printbuf[i]);
  }
  return length;
}
	     
int printw(const char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  return vwprintw(stdscr, fmt, args);
}

int mvprintw(int y, int x, const char *fmt, ...)
{
  va_list args;

  if (move(y, x) == ERR) {
    return ERR;
  }
  va_start(args, fmt);
  return vwprintw(stdscr, fmt, args);
}

int clrtoeol()
{
  int x;

  if (!stdscr) {
    return ERR;
  }
  for (x = stdscr->_curx; x < stdscr->_maxx; x++) {
    text[stdscr->_cury][x] = ' ';
    attr[stdscr->_cury][x] = A_NORMAL;
  }
  return OK;
}

void refresh()
{
  int x, y;
  int length;
  COORD pos, size;
  CHAR_INFO ci[2];
  SMALL_RECT sr;

  if (!stdscr) {
    return;
  }
  for (y = 0; y < stdscr->_maxy; y++) {
    for (x = 0; x < stdscr->_maxx; x += length) {
      length = (text[y][x] & 0xff80) ? 2 : 1;
      if (text[y][x] != text_old[y][x] || attr[y][x] != attr_old[y][x] ||
	  x > 0 && (text_old[y][x - 1] & 0xff80) ){
	ci[0].Char.UnicodeChar = text[y][x];
	switch (attr[y][x]) {
	case A_REVERSE:
	  ci[0].Attributes = BACKGROUND_RED | BACKGROUND_GREEN | BACKGROUND_BLUE;
	  break;
	default:
	  ci[0].Attributes = FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_BLUE;
	}
	if (length == 2) {
	  ci[1].Char.UnicodeChar = 0;
	  ci[1].Attributes = ci[0].Attributes;
	}
	pos.X = 0;
	pos.Y = 0;
	size.X = length;
	size.Y = 1;
	sr.Top = y;
	sr.Bottom = y;
	sr.Left = x;
	sr.Right = x + length - 1;
	WriteConsoleOutputW(stdscr->winhandle, ci, size, pos, &sr);
      }
    }
    memcpy(text_old[y], text[y], sizeof(text[0][0]) * stdscr->_maxx);
    memcpy(attr_old[y], attr[y], sizeof(attr[0][0]) * stdscr->_maxx);
  }

  pos.X = stdscr->_curx;
  pos.Y = stdscr->_cury;
  SetConsoleCursorPosition(stdscr->winhandle, pos);
}

int redrawwin(WINDOW *win)
{
  int x, y;
  SMALL_RECT sr;

  if (!win) {
    return ERR;
  }

  for (y = 0; y < win->_maxy; y++) {
    for (x = 0; x < win->_maxx; x++) {
      text_old[y][x] = 0;
    }
  }

  getmaxy(win);
  sr.Top = 0;
  sr.Bottom = stdscr->_maxy - 1;
  sr.Left = 0;
  sr.Right = stdscr->_maxx -1;
  SetConsoleWindowInfo(win->winhandle, TRUE, &sr);

  return OK;
}

int get_wch(wint_t *wch)
{
  int i;
  const wchar_t winkeytable[][2] = {
    {0x3b, KEY_F(1)},
    {0x3c, KEY_F(2)},
    {0x3d, KEY_F(3)},
    {0x3e, KEY_F(4)},
    {0x3f, KEY_F(5)},
    {0x40, KEY_F(6)},
    {0x41, KEY_F(7)},
    {0x42, KEY_F(8)},
    {0x43, KEY_F(9)},
    {0x44, KEY_F(10)},
    {0x47, KEY_HOME},
    {0x48, KEY_UP},
    {0x49, KEY_PPAGE},
    {0x4b, KEY_LEFT},
    {0x4d, KEY_RIGHT},
    {0x4f, KEY_END},
    {0x50, KEY_DOWN},
    {0x51, KEY_NPAGE},
    {0x52, KEY_IC},
    {0x53, KEY_DC},
    {0x54, KEY_F(13)},
    {0x55, KEY_F(14)},
    {0x56, KEY_F(15)},
    {0x57, KEY_F(16)},
    {0x58, KEY_F(17)},
    {0x59, KEY_F(18)},
    {0x5a, KEY_F(19)},
    {0x5b, KEY_F(20)},
    {0x5c, KEY_F(21)},
    {0x5d, KEY_F(22)},
    {0x85, KEY_F(11)},
    {0x86, KEY_F(12)},
    {0x87, KEY_F(23)},
    {0x88, KEY_F(24)},
    {0x99, KEY_SPREVIOUS}
  };

  refresh();
  if (!wch) {
    return ERR;
  }
  if (!stdscr->_nodelay) {
    while (_kbhit() == 0) {
      ;
    }
  }
  if (_kbhit()) {
    *wch = _getwch();
    if ((*wch == 0 || *wch == 0xe0) && stdscr->_use_keypad) {
      *wch = _getwch();
      for (i = 0; i < sizeof(winkeytable) / sizeof(winkeytable[0]); i++) {
	if (*wch == winkeytable[i][0]) {
	  *wch = winkeytable[i][1];
	  if (GetKeyState(VK_SHIFT) < 0) {
	    if (*wch == KEY_PPAGE) {
	      *wch = KEY_SPREVIOUS;
	    } else if (*wch == KEY_HOME) {
	      *wch = KEY_SHOME;
	    }
	  }
	  return OK;
	}
      }
      *wch = 0;
    }
    if (SP->autocr && *wch == 0x0d) {
      *wch = 0x0a;
    }
    if (SP->echo) {
      addch(*wch);
    }
    return OK;
  } else {
    *wch = 0;
    return ERR;
  }
}

char *tilde_expand(char *fname)
{
  /* just copy */
  char *fname_expand;
  if (!fname) {
    return NULL;
  }
  if ((fname_expand = malloc(strlen(fname) + 1)) == NULL) {
    return NULL;
  }
  strcpy(fname_expand, fname);
  return fname_expand;
}
