/* PC-6001/mkII emulator on terminal                     */
/*  by AKIKAWA, Hisashi 2014-2025                        */
/* This software is released under 2-clause BSD license. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <math.h>
#include <time.h>
#include <wchar.h>
#include <locale.h>
#include "z80.h"

#ifdef __FreeBSD__
# include <edit/readline/readline.h>
#else
# include <editline/readline.h>
#endif

#if defined __CYGWIN__ || defined __MSYS__
# include <w32api/windows.h>
#endif

#ifdef __MINGW32__
# include "win.h"
  extern WINDOW *stdscr;
#else
# define _XOPEN_SOURCE_EXTENDED
# include <ncurses.h>
# include <termios.h>
#endif

#ifdef X11
# include <X11/XKBlib.h>
# include <X11/keysym.h>
Display *dispX11 = NULL;
#endif


#define DEBUG 0
#define NONE -1

/* hardware parameter */
#define CLOCK_4M 3993600	/* PC-6001/mkII */
//#define CLOCK_4M 4000000	/* PC-6601 */
//#define CLOCK_14M 14.318e6
#define CLOCK_14M (4.5e6 / 286 * 910)
#define KEY_TICK_FACTOR 1

/* 8255 status */
#define NOBF	0x80	/* PC7 = #output buffer full   = 8049 #INT */
#define INTE1	0x40	/* bit6= write interrupt enable            */
#define IBF	0x20	/* PC5 = input buffer full     = 8049 T0   */
#define INTE2	0x10	/* bit4= read interrupt enable             */
#define INTR	0x08	/* PC3 = interrupt request                 */
#define CGSW	0x04	/* PC2 = cgrom switch                      */
#define BSRQEN	0x02	/* PC1 = bus request enable                */
#define PRTSTRB	0x01	/* PC0 =printer strobe                     */

#define SETBIT(a) {port_c |= (a);}
#define RESBIT(a) {port_c &= ~(a);}
#define RISING(new, old) ((new) && (old) == 0)
#define FALLING(new, old) ((new) == 0 && (old))

/* external FDD parameter and status */
#define NUM_OF_EXTFDD_CMD 8
#define ATN_PC  0x80  /* PC -> FDD */
#define DAC_PC  0x40  /* PC -> FDD */
#define RFD_PC  0x20  /* PC -> FDD */
#define DAV_PC  0x10  /* PC -> FDD */
#define DAC_FDD 0x04  /* FDD -> PC */
#define RFD_FDD 0x02  /* FDD -> PC */
#define DAV_FDD 0x01  /* FDD -> PC */

double execframe(int start, double period, double time_horizontal);
byte readmem(word address, int *wait);
void writemem(word address, byte value, int *wait);
byte readport(word address);
void writeport(word address, byte value);
void init(int argc, char *argv[]);
void reset_p6(void);
int upd8049(int reset);
byte upd8049read(void);
void upd8049write(byte value);
void upd8049cmd(int cmd);
void upd8255(int reset, int adr);
void extfdd(int reset);
int interrupt(int intr8049);
void keyinput(int *inkeyp, int *gamekeyp);
int keyvector(int inkey, int *keycode);
void display(void);
char *getinput(int y, const char *prompt);
FILE *read_open(const char *fname);
int getp6t(int *data);
int disk_open(const char *fname, int drive);
byte *getd88(int drive, int track, int sector);
void chgvram(void);
void wait60hz(double tick);
void romread(byte *rom, int size, const char *fname);
int romread_nocheck(byte *rom, int size, const char *fname);
extern void monitor(void);

FILE *fp = NULL;
FILE *tape_read = NULL;
FILE *tape_write = NULL;
int normal_speed = 1;
int use_extram = 1;
int use_extrom = 0;	/* 16KB ROM */
int use_beluga = 0;	/* 128KB ROM + 64KB RAM */
int ndrive = 0;
int is_d88[2];
int mk2 = 0;
int ram64 = 0;
int turbo = 1;
int boost = 8;
int winkey = 0;
int x11key = 0;
long long count_cycles = 0;
int break_address = NONE;
int break_read = NONE;
int break_write = NONE;
int break_in = NONE;
int break_out = NONE;
int break_vector = NONE;
int break_memory = 0;
int break_io = 0;
int break_intr = 0;
int break_monitor = 0;
int count_break = 0;
int until_break= 1;
int step = 0;
int tape_counter;
wint_t keybuf[16384];
int ikbf, okbf;
char tapefile[PATH_MAX + 1] = "";
char diskfile[2][PATH_MAX + 1] = {"", ""};

z80 cpu;
byte *mem_r[8], *mem_win[8], *mem_wex[8], *beluga[8];
byte ram[0x10000];
byte rom[0x8000];
byte extrom[0x80000];
byte extram[0x20000];
byte cgrom[0x2000];
byte cgrom2[0x2000];
byte kanjirom[0x8000];
byte voicerom[0x4000];
byte *vkrom;
byte port[0x100];
byte psg[0x10];
byte psg_register;

int mode5;
int page;
int relay;
int timer;
byte *vram;
int count_timer;
double tick;
int vsync, hsync;

/* for 8251 */
int rxrdy = 1;
byte serial_data = 0xff;

/* for 8255 */
int nrd;	/* #read                        */
int nwr;	/* #write                       */
int nack;	/* #acknoledge = PC6 = 8049 #RD */
int nstb;	/* #strobe     = PC4 = 8049 #WR */
byte port_a;	/* port A: for 8049             */
byte port_b;	/* port B: for printer          */
byte port_c;	/* port C: status etc.          */
byte dbus_8255;	/* data bus                     */

/* for 8049 */
enum {
  IDLE,
  GAME,
  READ,
  WRITEOPEN,
  WRITESTART,
  WRITEDATA,
  INTERRUPT,
};
int mode_8049;
int count_query;
int count_tape;
int baudrate_read;
int baudrate_write;
int kana_mode;
int katakana_mode;

/* for P6T format */
int count_auto = NONE;
int is_p6t, p6t_block, p6t_blank, p6t_iauto, p6t_waittape, p6t_basic;
double p6t_blanktime = 0;
struct {
  byte version;
  byte nblock;
  byte autostart;
  byte mode;
  byte pages;
  word nauto;
  byte *autostring;
  word nextend;
  byte **data;
} p6t;

int ndrv = 0;
byte disk[2][2000000];

/* display character */
const wchar_t dispchar[]
= L"　月火水木金土日年円時分秒百千万"
  "π┻┳┫┣╋┃━┏┓┗┛×大中小"
  "　！″＃＄％＆′（）＊＋，‐．／"
  //  "　！”＃＄％＆’（）＊＋，‐．／"
  //  "　！”＃＄％＆’（）＊＋，─．／"
  //  "　！”＃＄％＆’（）＊＋，—．／"
  "０１２３４５６７８９：；＜＝＞？"
  "＠ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯ"
  "ＰＱＲＳＴＵＶＷＸＹＺ［￥］＾＿"
  "　ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏ"
  "ｐｑｒｓｔｕｖｗｘｙｚ｛｜｝〜　"
  "▲▼★◆○●をぁぃぅぇぉゃゅょっ"
  "　あいうえおかきくけこさしすせそ"
  "　。「」、・ヲァィゥェォャュョッ"
  "ーアイウエオカキクケコサシスセソ"
  "タチツテトナニヌネノハヒフヘホマ"
  "ミムメモヤユヨラリルレロワン゛゜"
  "たちつてとなにぬねのはひふへほま"
  "みむめもやゆよらりるれろわん　　";

int main(int argc, char *argv[])
{
  int scanline;
  double period;
  double time_horizontal;

  init(argc, argv);
  reset_p6();

  period = (mk2 ? 912. : 910.) / CLOCK_14M;
  tick = period * 262;

  srand(time(NULL));
  //  srand(299792458);
  scanline = rand() / (RAND_MAX / 262 + 1);
  time_horizontal = period * rand() / (RAND_MAX + 1.);

  time_horizontal = execframe(scanline, period, time_horizontal);
  while (1) {
    time_horizontal = execframe(0, period, time_horizontal);
  }
}

double execframe(int start, double period, double time_horizontal)
{
  int i;
  int busreq_cycles;

  for (i = start; i < 262; i++) {
    vsync = (i < 3) ? 0 : 1;
    if ((mode5 ? 30 : 38) <= i && i <= 229 && (port_c & BSRQEN)) {
      int busreq_count = mode5 ? 46 : (mk2 ? 38 : 37);
      busreq_cycles = ceil((busreq_count / (CLOCK_14M / 16) - time_horizontal) * CLOCK_4M) + 1;
      cpu.cycles += busreq_cycles;
      time_horizontal += busreq_cycles / (double)CLOCK_4M;
    }

    while (time_horizontal < period + 1./CLOCK_4M) {
      hsync = (time_horizontal < 64. / CLOCK_14M) ? 0 : 1;
      upd8255(0, 0);
      extfdd(0);

      if (interrupt(upd8049(0)) == 0) {
	z80_main(&cpu);
      }

      count_cycles += cpu.cycles;
      time_horizontal += cpu.cycles / (double)CLOCK_4M;

      if (cpu.mcycle == 0) {
	if (step == 1) {
	  monitor();
	} else if (cpu.pc == break_address || break_memory || break_io || break_intr || break_monitor) {
	  break_memory = 0;
	  break_io = 0;
	  break_intr = 0;
	  break_monitor = 0;
	  if (++count_break >= until_break) {
	    count_break = 0;
	    monitor();
	  }
	}
      }

    }
    time_horizontal -= period;
  }

  if (port_c & BSRQEN) {
    display();
  }

  if (normal_speed && (relay == 0 || turbo == 0)) {
    wait60hz(tick);
  }

  return time_horizontal;
}

/*
 * read memory
 */
byte readmem(word address, int *wait)
{
  int flg;
  int dummy;

  if (wait == NULL) wait = &dummy;

  *wait = cpu.m1 ? 1 : 0;

  if ((~port_c & CGSW) &&
      (!mk2 && (address & 0xe000) == 0x6000 ||
       mk2 && (port[0xf8] & 0x40) &&
       !(((address >> 13) ^ port[0xf8] & 0x07) & (~port[0xf8] >> 3)))) {
    *wait |= (!mk2 || (port[0xf8] & 0x80)) ? 1 : 0;
    if (mode5) {
      return cgrom2[address & 0x1fff];
    } else {
      return cgrom[address & 0x1fff];
    }
  }

  if (mk2) {
    flg = port[0xf0 + (address >> 15)] >> ((address >> 12) & 0x04) & 0x0f;
    if (flg != 0x0d && flg != 0x0e) {  /* ROM */
      *wait = 1;
    }
  } else {
    *wait |= (address < 0x8000) ? 1 : 0;
  }
  
  if (mem_r[address >> 13]) {
    return mem_r[address >> 13][address & 0x1fff];
  } else {
    return 0xff;
  }

}

/*
 * write memory
 */
void writemem(word address, byte value, int *wait)
{
  int i;
  int flg;
  int dummy;

  if (wait == NULL) wait = &dummy;
  *wait = 0;

  if (mk2) {
    flg = port[0xf2] >> ((address >> 14) * 2) & 0x03;
    if (flg == 0x01 || flg == 0x02) {
      *wait = cpu.m1 ? 1 : 0;
    } else {
      *wait = 1;
    }
  } else {
    *wait = (address < 0x8000) ? 1 : 0;
  }

  if (mem_win[address >> 13] == &ram[address & 0xe000]) {
    ram[address] = value;
  }
  for (i = 0; i < 0x10; i++) {
    if (mem_wex[address >> 13] == &extram[i * 0x2000]) {
      mem_wex[address >> 13][address & 0x1fff] = value;
      break;
    }
  }
}

/*
 * read I/O port
 */
byte readport(word address)
{
  address &= 0xff;

  switch (address) {
//case 0xc0:return 0;

  /* 8251 */
  case 0x80: case 0x82: case 0x84: case 0x86:
  case 0x88: case 0x8a: case 0x8c: case 0x8e:
    rxrdy = 1;
    return serial_data;

  case 0x81: case 0x83: case 0x85: case 0x87:
  case 0x89: case 0x8b: case 0x8d: case 0x8f:
    return
      0x80 |		/* DSR */
      0x04 |		/* TxEMPTY */
      (rxrdy << 1) |	/* #RxRDY */
      0x01;		/* TxRDY */

  
  /* 8255 port A = 8049 data */
  case 0x90: case 0x94: case 0x98: case 0x9c:
  /* 8255 port B = printer data */
  case 0x91: case 0x95: case 0x99: case 0x9d:
  /* 8255 port C (status etc.) */
  case 0x92: case 0x96: case 0x9a: case 0x9e:
  /* illegal condition */
  case 0x93: case 0x97: case 0x9b: case 0x9f:
    nrd = 0;
    upd8255(0, address & 0x03);
    nrd = 1;
    upd8255(0, address & 0x03);
    return dbus_8255;

  /* PSG */
  case 0xa2: case 0xa6: case 0xaa: case 0xae:
    if (psg_register == 0x0e && !(psg[0x07] & 0x40) ||
	psg_register == 0x0f && !(psg[0x07] & 0x80)) {
	/* input = joy stick port */
      if (mk2) {
	return (vsync ? 0x80 : 0) | (hsync ? 0x40 : 0) | 0x3f;
      } else {
	return 0xff;
      }
    } else {
      if (psg_register == 0x0e) {
	/* input = joy stick port? */
	if (mk2) {
	  return (vsync ? 0x80 : 0) | (hsync ? 0x40 : 0) | 0x3f;
	} else { 
	  return 0xff;
        }
      } else {
	return psg[psg_register];
      }
    }

  /* VRAM, relay, timer interrupt, etc. */
  case 0xb0: case 0xb1: case 0xb2: case 0xb3:
  case 0xb4: case 0xb5: case 0xb6: case 0xb7:
  case 0xb8: case 0xb9: case 0xba: case 0xbb:
  case 0xbc: case 0xbd: case 0xbe: case 0xbf:
    if (mk2) {
      return port[address];
    } else {
      writeport(0xb0, 0xff);
    }
    return 0xff;

  /* FDD */
  case 0xd0: case 0xd4: case 0xd8: case 0xdc:
    return (ndrive ? port[0xd0] : 0xff);
    break;

  case 0xd1: case 0xd5: case 0xd9: case 0xdd:
    return (ndrive ? port[0xd1] : 0xff);
    break;

  case 0xd2: case 0xd6: case 0xda: case 0xde:
    return (ndrive ? port[0xd2] : 0xff);
    break;

  case 0xd3: case 0xd7: case 0xdb: case 0xdf:
    return 0xff;
    break;

  /* 7752 */
  case 0xe0:
    return 0x40;	/* REQ=1 */

  /* Beluga cartridge */
  case 0x06:

  /* bank change */
  case 0xf0: case 0xf1: case 0xf2: case 0xf3:
  case 0xf4: case 0xf5: case 0xf6: case 0xf7:
  case 0xf8:
    return port[address];

  default:
    return 0xff;

  }
}

/*
 * write I/O port
 */
void writeport(word address, byte value)
{
  int i, j;
  int flg;
  FILE *serial;

  address &= 0xff;

  switch (address) {

  /* Beluga cartridge */
  case 0x06:
    if (use_beluga) {
      port[address] = value;
    }
    if (value == 0x66) {
      writeport(0xf0, port[0xf0]);
      writeport(0xf2, port[0xf2]);
    }
    break;

  case 0x30: case 0x31: case 0x32: case 0x33:
  case 0x34: case 0x35: case 0x36: case 0x37:
  case 0x38: case 0x39: case 0x3a: case 0x3b:
  case 0x3c: case 0x3d: case 0x3e: case 0x3f:
    if (use_beluga) {
      port[address & 0xf7] = value;
      port[address | 0x08] = value;
      switch(value & 0xc0) {
      case 0x00:  /* ROM */
	for (address = 0x30; address < 0x38; address++) {
	  if ((port[address] & 0xc0) == 0) {
	    port[address] = (value & 0x30) + (port[address] & 0x0f);
	    port[address | 0x08] = port[address];
	    beluga[address & 0x07] = &extrom[(port[address] & 0x3f) << 13];
	  }
	}
	break;
      case 0x40:  /* RAM */
	beluga[address & 0x07] = &extram[(value & 0x0f) << 13];
	break;
      case 0x80:  /* SCC, not implemented */
	beluga[address & 0x07] = NULL;
	break;
      case 0xc0:  /* not implemented */
	beluga[address & 0x07] = NULL;
	break;
      }
      writeport(0xf0, port[0xf0]);
      writeport(0xf1, port[0xf1]);
      writeport(0xf2, port[0xf2]);
      if (!ram64) {
	mem_r[2] = mem_wex[2] = beluga[2];
	mem_r[3] = mem_wex[3] = beluga[3];
      }
    }
    chgvram();
    break;

  case 0x70: case 0x71: case 0x72: case 0x73:
  case 0x74: case 0x75: case 0x76: case 0x77:
  case 0x78: case 0x79: case 0x7a: case 0x7b:
  case 0x7c: case 0x7d: case 0x7e: case 0x7f:
    if (use_beluga) {
      port[address] = value;
      writeport(0x32, (port[0x32] & 0x30) + (value & 0x0f));
      if (!ram64) {
	mem_r[2] = mem_wex[2] = beluga[2];
      }
    }
    break;

  /* 8251 */
  case 0x80:
    if ((serial = fopen("serial.txt", "ab")) != NULL) {
      fputc(value, serial);
      fclose(serial);
    }
    break;

  /* 8255 port A = 8049 data */
  case 0x90:  case 0x94:  case 0x98:  case 0x9c:
  /* 8255 port B = printer data */
  case 0x91:  case 0x95:  case 0x99:  case 0x9d:
  /* 8255 port C */
  case 0x92:  case 0x96:  case 0x9a:  case 0x9e:
  /* 8255 control register */
  case 0x93:  case 0x97:  case 0x9b:  case 0x9f:
    dbus_8255 = value;
    nwr = 0;
    upd8255(0, address & 0x03);
    nwr = 1;
    upd8255(0, address & 0x03);
    break;

  /* PSG */
  case 0xa0:  case 0xa4:  case 0xa8:  case 0xac:
    psg_register = value & 0xf;
    break;

  case 0xa1:  case 0xa5:  case 0xa9:  case 0xad:
    psg[psg_register] = value;
    break;

  case 0xb0:
    /* timer */
    if (!timer && !(value & 0x01)) {	/* off -> on */
      if (mk2) {
	count_timer = 0;
      } else {
	count_timer = 8192 / 2;
      }
    }
    timer = !(value & 0x01);

    /* change VRAM address */
    page = (value & 0x06) >> 1;		/* !!! 0-3 !!! */
    chgvram();

    /* tape relay */
    if (relay && !(value & 0x08)) {	/* on -> off */
      p6t_waittape = 0;
    }
    if ((value & 0x08) && !relay) {	/* off -> on */
      count_tape = 0;
    }
    if (!is_p6t && !relay && (value & 0x08)) {
      p6t_blank = 1;
      p6t_blanktime = 1;
      p6t_basic = 0;
    }
    relay = (value & 0x08) >> 3;

    break;

  case 0xb1: case 0xb3:
  case 0xb4: case 0xb5: case 0xb6: case 0xb7:
  case 0xb8: case 0xb9: case 0xba: case 0xbb:
  case 0xbc: case 0xbd: case 0xbe: case 0xbf:
    if (!mk2) {
      writeport(0xb0, value);
    }
    break;

  case 0xb2:
    if (mk2) {
      port[address] = (value & 0x0e) | 0xf1;
    } else {
      writeport(0xb0, value);
    }
    break;

  /* printer */
  case 0xc0:
    if (mk2) {
      port[address] &= ~0x07;
      port[address] |= (value & 0x07);
    }
    break;

  /* display mode */
  case 0xc1:
    if (mk2) {
      port[address] = value;
      mode5 = !(value & 0x02);
      chgvram();
    }
    break;

  /* kanji/voice rom control */
  case 0xc2:
    if (mk2) {
      port[address] = value;
      if (value & 0x01) {
	vkrom = (value & 0x02) ? &kanjirom[0x4000] : &kanjirom[0x0000];
      } else {
	vkrom = voicerom;
      }
      for (i = 0; i < 8; i++) {
	j = i % 2 * 0x2000;
	if (mem_r[i] == &kanjirom[j] ||
	    mem_r[i] == &kanjirom[j + 0x4000] ||
	    mem_r[i] == &voicerom[j]) {
	  mem_r[i] = &vkrom[j];
	}
      }
    }
    break;

  /* FDD */
  case 0xd0: case 0xd4: case 0xd8: case 0xdc:
    break;
  case 0xd1: case 0xd5: case 0xd9: case 0xdd:
    port[0xd1] = value;
    break;
  case 0xd2: case 0xd6: case 0xda: case 0xde:
    port[0xd2] = (value & 0xf0) | (port[0xd2] & 0x0f);
    break;
  case 0xd3: case 0xd7: case 0xdb: case 0xdf:
    port[0xd3] = value;
    if (value & 0x80) {			/* 8255 mode set */
      ;					/* not implemented */
    } else {				/* 8255 bit set/reset */
      if (value & 0x01) {
	port[0xd2] |= 1 << ((value & 0x0e) >> 1);
      } else {
	port[0xd2] &= ~(1 << ((value & 0x0e) >> 1));
      }
    }
    break;

  /* bank change */
  case 0xf0:
  case 0xf1:
    if (ram64) {
      port[address] = value;
      for (i = 0; i < 2; i++) {
	j = (address & 1) * 4 + i * 2;
	flg = (value >> (i * 4)) & 0x0f;

	if (flg == 0x1 || flg == 0x6 || flg == 0xa) {
	  mem_r[j] = &rom[(j * 0x2000) % 0x8000];
	} else if (flg == 0x2 || flg == 0x5 || flg == 0xc) {
	  mem_r[j] = vkrom;
	} else if (flg == 0x4 || flg == 0x7 || flg == 0xb) {
	  mem_r[j] = use_beluga ? beluga[j] : extrom;
	} else if (flg == 0x3 || flg == 0x8 || flg == 0x9) {
	  mem_r[j] = use_beluga ? beluga[j] : &extrom[0x2000];
	} else if (flg == 0xd) {
	  mem_r[j] = &ram[j * 0x2000];
	} else if (flg == 0xe) {
	  mem_r[j] = use_beluga ? beluga[j] : &extram[j * 0x2000];
	} else {
	  mem_r[j] = NULL;
	}

	j++;
	if (flg == 0x1 || flg == 0x5 || flg == 0x9) {
	  mem_r[j] = &rom[(j * 0x2000) % 0x8000];
	} else if (flg == 0x2 || flg == 0x6 || flg == 0xb) {
	  mem_r[j] = &vkrom[0x2000];
	} else if (flg == 0x4 || flg == 0x8 || flg == 0xc) {
	  mem_r[j] = use_beluga ? beluga[j] : extrom;
	} else if (flg == 0x3 || flg == 0x7 || flg == 0xa) {
	  mem_r[j] = use_beluga ? beluga[j] : &extrom[0x2000];
	} else if (flg == 0xd) {
	  mem_r[j] = &ram[j * 0x2000];
	} else if (flg == 0xe) {
	  mem_r[j] = use_beluga ? beluga[j] : &extram[j * 0x2000];
	} else {
	  mem_r[j] = NULL;
	}
      }

    } else if (use_beluga && address == 0xf0) {
      port[address] = value;
      mem_r[2] = mem_wex[2] = beluga[2];
      mem_r[3] = mem_wex[3] = beluga[3];
      mem_r[4] = mem_wex[4] = beluga[4];
      mem_r[5] = mem_wex[5] = beluga[5];
      if (port[0x06] ==0x66) {
	flg = value & 0x0f;
	if (flg == 0x1 || flg == 0x2 || flg == 0x5 || flg == 0x6) {
	  mem_r[0] = rom;
	  mem_r[1] = &rom[0x2000];
	} else {
	  mem_r[0] = beluga[0];
	  mem_r[1] = beluga[1];
	}
      }
    }

    break;

  case 0xf2:
    if (ram64) {
      port[address] = value;
      for (i = 0; i < 8; i++) {
	flg = (value >> (i & 6)) & 3;
	mem_win[i] = mem_wex[i] = NULL;
	if (flg & 1) {
	  mem_win[i] = &ram[i * 0x2000];
	}
	if (flg & 2) {
	  mem_wex[i] = use_beluga ? beluga[i] : &extram[i * 0x2000];
	}
      }

    } else if (use_beluga) {
      port[address] = value;
      if (port[0x06] == 0x66) {
	for (i = 0; i < 4; i++) {
	  flg = (value >> (i & 2)) & 3;
	  if (flg) {
	    mem_wex[i] = beluga[i];
	  } else {
	    mem_wex[i] = &rom[i * 0x2000];
	  }
	}
      }
    }
    break;

  case 0xf3: case 0xf4: case 0xf5: case 0xf6: case 0xf7:
  case 0xf8:
    if (mk2) {
      port[address] = value;
    }
    break;

  default:
    break;
  }
}


byte z80_read(word address)
{
  byte value;
  int wait;

  if (address == break_read) {
    break_memory = 1;
  }

  value = readmem(address, &wait);
  cpu.cycles += wait;

  return value;
}


void z80_write(word address, byte value)
{
  int wait;

  if (address == break_write) {
    break_memory = 1;
  }

  writemem(address, value, &wait);
  cpu.cycles += wait;
}


byte z80_in(word address)
{
  if ((address & 0xff) == break_in) {
    break_io = 1;
  }
  if ((address & 0xf0) == 0xa0) {
    cpu.cycles++;
  }
  return readport(address);
}


void z80_out(word address, byte value)
{
  if ((address & 0xff) == break_out) {
    break_io = 1;
  }
  if ((address & 0xf0) == 0xa0) {
    cpu.cycles++;
  }
  writeport(address, value);
}

/*
 * initialize emulator
 */
void init(int argc, char *argv[])
{
  int i;
  int rawmode;
  FILE *extrom_file;
  extern int rl_completion_append_character;

  if (DEBUG) {
    fp = fopen("log", "w");
  }

#ifdef __MINGW32__
  /* Windows key input */
  rawmode = 1;
  winkey = 1;
#else
  rawmode = 0;
#endif

  memset(extrom, 0xff, sizeof(extrom));

  /* analyze option */
  for (i = 1; i < argc; i++) {
    if (strcmp(argv[i], "-raw") == 0) {
      rawmode = 1;
    } else if (strcmp(argv[i], "-noraw") == 0) {
      rawmode = 0;
    } else if (strcmp(argv[i], "-16") == 0) {
      use_extram = 0;
    } else if (strcmp(argv[i], "-64") == 0) {
      ram64 = 1;
    } else if (strcmp(argv[i], "-mk2") == 0) {
      mk2 = 1;
      ram64 = 1;
    } else if (strcmp(argv[i], "-nowait") == 0) {
      normal_speed = 0;
    } else if (strcmp(argv[i], "-noturbo") == 0) {
      turbo = 0;
    } else if (strcmp(argv[i], "-boost") == 0) {
      if (++i >= argc) {
	break;
      }
      sscanf(argv[i], "%d", &boost);
      if (boost < 1) {
	boost = 1;
      }
    } else if (strcmp(argv[i], "-win") == 0) {
#ifdef WIN32
      winkey = 1;
#else
      printf("-win option is only for Windows\n");
      exit(EXIT_FAILURE);
#endif
    } else if (strcmp(argv[i], "-x11") == 0) {
#ifdef X11
      x11key = 1;
#else
      printf("-x11 option is only for X11\n");
      exit(EXIT_FAILURE);
#endif
    } else if (strcmp(argv[i], "-tape") == 0) {
      if (++i >= argc) {
	break;
      }
      strncpy(tapefile, argv[i], sizeof(tapefile) - 1);
      tapefile[sizeof(tapefile) - 1] = '\0';
      if ((tape_read = read_open(tapefile)) == NULL) {
	printf("cannot open %s\n", tapefile);
	exit(EXIT_FAILURE);
      }
      add_history(tapefile);
      if (is_p6t && p6t.autostart) {
	if (p6t.mode == 1 || p6t.mode == 3) {
	  use_extram = 0;
	} else {
	  use_extram = 1;
	}
      }
    } else if (strcmp(argv[i], "-disk") == 0) {
      if (++i >= argc) {
	break;
      }
      strncpy(diskfile[0], argv[i], sizeof(diskfile[0]) - 1);
      diskfile[0][sizeof(diskfile[0]) - 1] = '\0';
      if (!disk_open(diskfile[0], 0)) {
	printf("cannot open %s or illegal .d88\n", diskfile[0]);
	exit(EXIT_FAILURE);
      }
      ndrive = 1;
      if (++i >= argc) {
	break;
      }
      if (argv[i][0] == '-') {
	i--;
	continue;
      }
      strncpy(diskfile[1], argv[i], sizeof(diskfile[1]) - 1);
      diskfile[1][sizeof(diskfile[1]) - 1] = '\0';
      if (!disk_open(diskfile[1], 1)) {
	printf("cannot open %s or illegal .d88\n", diskfile[1]);
	exit(EXIT_FAILURE);
      }
      ndrive = 2;
    } else if (strcmp(argv[i], "-rom") == 0) {
      if (++i >= argc) {
	break;
      }
      //      memset(extrom, 0xff, sizeof(extrom));
      if ((extrom_file = fopen(argv[i], "rb")) == NULL) {
	printf("cannot open %s\n", argv[i]);
	exit(EXIT_FAILURE);
      }
      if (fread(extrom, 1, sizeof(extrom), extrom_file) <= 0x4000) {
	use_extrom = 1;
	memcpy(&extrom[0x4000], extrom, 0x4000);
	memcpy(&extrom[0x8000], extrom, 0x4000);
	memcpy(&extrom[0xc000], extrom, 0x4000);
      } else {
	use_beluga = 1;
      }
      fclose(extrom_file);
    } else {
      printf("options:\n"
	     " -16                   16KB RAM\n"
	     " -64                   64kB RAM\n"
	     " -mk2                  mkII mode\n"
	     " -nowait               start with full speed\n"
	     " -noturbo              no turbo tape\n"
	     " -boost <n>            tape speed=1200*n baud (default=8)\n"
	     " -x11                  use X11 API for game key input\n"
	     " -win                  use Windows API for game key input\n"
	     " -raw                  raw mode TTY\n"
	     " -noraw                not raw mode TTY\n"
	     " -tape <file>          set tape file\n"
	     " -rom <file>           set external rom file\n"
	     " -disk <file> [<file>] set disk file\n"
	     );
      exit(EXIT_FAILURE);
    }
  }

#ifdef X11
  /* X11 key input */
  if (x11key) {
    if ((dispX11 = XOpenDisplay(NULL)) == NULL) {
      printf("cannot open display\n");
      exit(EXIT_FAILURE);
    }
  }
#endif

  /* set RAM & ROM */
  if (use_beluga) {
    use_extram = 0;
  }
  memset(ram, 0x00, sizeof(ram));
  if (!use_extram && !ram64 && !use_beluga) {
    memset(&ram[0x8000], 0xff, 0x4000);
  }
  memset(rom, 0xff, sizeof(rom));
  memset(cgrom, 0, sizeof(cgrom));
  memset(extram, 0xff, sizeof(extram));
  if (!mk2) {
    for (i = 0x8000; i < sizeof(ram); i += 0x80) {
      memset(&ram[i], 0xff, 0x40);
      memset(&ram[i + 0x40], 0x00, 0x40);
    }
  }

  if (mk2) {
    romread(rom, sizeof(rom), "BASICROM.62");
    romread(voicerom, sizeof(voicerom), "VOICEROM.62");
    romread_nocheck(kanjirom, sizeof(kanjirom), "KANJIROM.62");
    romread_nocheck(cgrom, sizeof(cgrom), "CGROM60.62");
    if (!romread_nocheck(cgrom2, sizeof(cgrom2), "CGROM60m.62")) {
      romread_nocheck(cgrom2, sizeof(cgrom2), "CGROM62.62");
    }
  } else {
    romread(rom, 0x4000, "BASICROM.60");
    romread_nocheck(cgrom, sizeof(cgrom), "CGROM60.60");
    memcpy(cgrom + 0x1000, cgrom, 0x1000);
  }


  /* readline */
  using_history();
  rl_completion_append_character = '\0';

  /* ncurses */
  setlocale(LC_ALL, "");
  initscr();
  nonl();
  noecho();
  keypad(stdscr, TRUE);
  nodelay(stdscr, TRUE);
  curs_set(0);
  scrollok(stdscr, TRUE);
  set_escdelay(20);
  if (rawmode) {
    raw();
  }
}

/*
 * reset hardware
 */
void reset_p6()
{
  /* tape */
  tape_counter = 0;
  if (tape_read) {
    fseek(tape_read, 0, SEEK_SET);
  }
  if (is_p6t) {
    p6t_blank = 1;
    p6t_block = 0;
    p6t_iauto = -3;
    p6t_waittape = 0;
    if (p6t.autostart) {
      if (count_auto != NONE || p6t_iauto > p6t.nauto) {
	count_auto = NONE;
      } else {
	count_auto = 0;
      }
    }
  }

  /* reset I/O port */
  memset(port, 0xff, sizeof(port));
  memset(psg, 0, sizeof(psg));
  psg_register = 0;
  writeport(0xb0, 0);
  if (mk2) {
    writeport(0xc1, 0xff);
    writeport(0xc2, 0xff);
    writeport(0xf7, 0x06);
    writeport(0xf8, 0xc3);	/* CGROM: wait off, enable, 0x6000 */
  }

  /* timer */
  count_timer = rand() / (RAND_MAX / 8192 + 1);

  /* for Beluga cartridge */
  writeport(0x06, 0xff);
  writeport(0x30, 0x00);	/* 0000: ROM set#0,bank#0 */
  writeport(0x31, 0x00);	/* 2000: ROM set#0,bank#0 */
  writeport(0x32, 0x00);	/* 4000: ROM set#0,bank#0 */
  writeport(0x33, 0x43);	/* 6000: RAM bank#3 */
  writeport(0x34, 0x44);	/* 8000: RAM bank#4 */
  writeport(0x35, 0x45);	/* a000: RAM bank#5 */
  writeport(0x36, 0x00);	/* c000: ROM set#0,bank#0 */
  writeport(0x37, 0x00);	/* e000: ROM set#0,bank#0 */

  /* memory */
  mem_r[0] = &rom[0x0000];
  mem_r[1] = &rom[0x2000];
  mem_r[2] = &extrom[0x0000];
  mem_r[3] = &extrom[0x2000];
  mem_r[4] = use_beluga ? beluga[4] : (use_extram || ram64) ? &extram[0x8000] : NULL;
  mem_r[5] = use_beluga ? beluga[5] : (use_extram || ram64) ? &extram[0xa000] : NULL;
  mem_r[6] = &ram[0xc000];
  mem_r[7] = &ram[0xe000];

  mem_win[0] = NULL;
  mem_win[1] = NULL;
  mem_win[2] = NULL;
  mem_win[3] = NULL;
  mem_win[4] = NULL;
  mem_win[5] = NULL;
  mem_win[6] = &ram[0xc000];
  mem_win[7] = &ram[0xe000];

  mem_wex[0] = NULL;
  mem_wex[1] = NULL;
  mem_wex[2] = NULL;
  mem_wex[3] = NULL;
  mem_wex[4] = mem_r[4];
  mem_wex[5] = mem_r[5];
  mem_wex[6] = NULL;
  mem_wex[7] = NULL;

  /* for mk2 */
  writeport(0xf0, 0x71);	/* 0000-3fff:BASIC ROM, 4000-7fff:external ROM */
  writeport(0xf1, 0xdd);	/* 8000-ffff:internal RAM */
  writeport(0xf2, 0x55);	/* 0000-ffff:internal RAM for writing */

  page = 0;
  chgvram();

  /* reset Z80 */
  cpu.nmi = 0;
  cpu.busreq = 0;
  cpu.intr = 0;
  cpu.cycles = 0;
  z80_reset(&cpu);

  /* reset 8049 */
  upd8049(1);

  /* reset 8255 */
  nrd = 1;
  nwr = 1;
  nack = 1;
  nstb = 1;
  upd8255(1, 0);

  /* reset external FDD */
  extfdd(1);
}

/*
 * uPD8049 (sub CPU)
 *  return 1=interrupt, 0=no interrupt
 */
int upd8049(int reset)
{
  static int period;
  static int cmd;
  static int count_key;
  static int port90_next;
  static int inkey;
  static int gamekey;
  static int mode_next;
  static int rxrdy_old = 1;
  int data;

  /* reset */
  if (reset) {
    mode_8049 = IDLE;
    cmd = NONE;
    kana_mode = 0;
    katakana_mode = 0;
    baudrate_read = 1200;
    baudrate_write = 1200;
    count_tape = NONE;
    ikbf = 0;
    okbf = 0;
    inkey = ERR;

    /* reset keyinput */
    gamekey = 0xff;
    keyinput(&inkey, &gamekey);

    return 0;
  }

  if (relay && count_tape != NONE) count_tape += cpu.cycles;

  if ((port_c & NOBF) == 0 && cmd == NONE) {
    cmd = upd8049read();
  }

  if (period == 0) {
    period =  tick * CLOCK_4M * KEY_TICK_FACTOR;
  }
  count_key += cpu.cycles;
  if (count_key > period) {
    count_key %= period;
    keyinput(&inkey, &gamekey);
    if (inkey == 0x0d) {
      period =  tick * CLOCK_4M * KEY_TICK_FACTOR * 10;
    } else {
      period =  tick * CLOCK_4M * KEY_TICK_FACTOR;
    }
  }

  switch (mode_8049) {

  /* key input */
  case IDLE:
    if (cmd != NONE) {
      upd8049cmd(cmd);
      cmd = NONE;
    } else if (rxrdy == 0 && rxrdy_old == 0) {
      upd8049write(0x04);
      mode_8049 = INTERRUPT;
      mode_next = IDLE;
      port90_next = NONE;
      rxrdy = 1;
      return 1;
    } else if (inkey != ERR && (port_c & IBF) == 0) {
      upd8049write(keyvector(inkey, &port90_next));
      mode_8049 = INTERRUPT;
      mode_next = IDLE;
      inkey = ERR;
      return 1;
    }
    rxrdy_old = rxrdy;
    break;

  /* reply for game key query (in 8049 interrupt routine?) */
  case GAME:
    cmd = NONE;
    count_query += cpu.cycles;
    if (count_query > 10000 && (port_c & IBF) == 0) {
      upd8049write(0x16);
      mode_8049 = INTERRUPT;
      mode_next = IDLE;
      port90_next = gamekey;
      return 1;
    }
    break;

  /* CMT read */
  case READ:
    if (inkey == KEY_BREAK || inkey == KEY_END) {
      /* CMT read stop */
      upd8049write(0x10);
      mode_8049 = INTERRUPT;
      mode_next = IDLE;
      port90_next = NONE;
      inkey = ERR;
      return 1;
    } else if (cmd != NONE) {
      if (cmd != 0x06) {
	upd8049cmd(cmd);
      }
      cmd = NONE;
      inkey = ERR;
      break;
    }

    inkey = ERR;
    if (getp6t(&data)) {
      upd8049write(0x08);
      mode_8049 = INTERRUPT;
      mode_next = READ;
      port90_next = data;
      if (port90_next != EOF) {
	tape_counter++;
	mvprintw(mode5 ? 20 : 16, 73, "%6d", tape_counter);
      }
      return 1;
    }
    break;

  /* CMT write */
  case WRITEOPEN:
    if (cmd != NONE) {
      if (cmd != 0x06) {
	upd8049cmd(cmd);
	if (cmd != 0x38) {
	  cmd = NONE;
	}
      }
    }
    break;

  /* CMT write start bit (wait before getting data) */
  case WRITESTART:
    period = CLOCK_4M / baudrate_write;
    if (count_tape > period) {
      mode_8049 = WRITEDATA;
      cmd = NONE;
    }
    break;

  case WRITEDATA:
    period = CLOCK_4M * 12 / baudrate_write / boost;
    if (count_tape > period) {
      count_tape %= period;
      if (cmd != NONE) {
	mode_8049 = WRITEOPEN;
	if (tape_write) {
	  fputc(cmd, tape_write);
	  fflush(tape_write);
	}
	cmd = NONE;
      }
    }
    break;

  case INTERRUPT:
    if ((port_c & IBF)) {
      return 1;
    } else {
      if (port90_next != NONE) {
	upd8049write(port90_next);
      }
      mode_8049 = mode_next;
    }
    break;
  }

  if (mode_8049 == WRITEOPEN || mode_8049 == WRITESTART) {
    if (inkey == KEY_BREAK || inkey == KEY_END) {
      /* CMT write stop */
      upd8049write(0x0e);
      mode_8049 = INTERRUPT;
      mode_next = IDLE;
      port90_next = NONE;
      upd8049read();
      inkey = ERR;
      return 1;
    }
    inkey = ERR;
  }

  return 0;
}

/*
 * uPD8049 reads data from uPD8255
 */
byte upd8049read()
{
  nack = 0;
  upd8255(0, 0);
  nack = 1;
  upd8255(0, 0);
  return port_a;
}

/*
 * uPD8049 writes data to uPD8255
 */
void upd8049write(byte value)
{
  port_a = value;
  nstb = 0;
  upd8255(0, 0);
  nstb = 1;
  upd8255(0, 0);
}

/*
 * uPD8049 executes command
 */
void upd8049cmd(int cmd)
{
  switch (cmd) {

  case 0x06:		/* game key query */
    mode_8049 = GAME;
    count_query = 0;
    break;

  case 0x19:		/* open CMT for reading */
    mode_8049 = READ;
    //    count_tape = 0;
    break;

  case 0x1a:		/* close CMT for reading */
    mode_8049 = IDLE;
    count_tape = NONE;
    break;

  case 0x1d:		/* set baud rate for reading (600 baud) */
    baudrate_read = 600;
    break;

  case 0x1e:		/* set baud rate for reading (1200 baud) */
    baudrate_read = 1200;
    break;

  case 0x38:		/* write to CMT */
    if (mode_8049 == WRITEOPEN) {
      mode_8049 = WRITESTART;
      count_tape = 0;
    }
    break;

  case 0x39:		/* open CMT for writing */
    tape_write = fopen("savedata.p6", "ab");
    mode_8049 = WRITEOPEN;
    break;

  case 0x3a:		/* close CMT for writing */
    mode_8049 = IDLE;
    count_tape = NONE;
    if (tape_write) {
      fclose(tape_write);
      tape_write = NULL;
    }
    break;

  case 0x3d:		/* set baud rate for writing (600 baud) */
    baudrate_write = 600;
    break;

  case 0x3e:		/* set baud rate for writing (1200 baud) */
    baudrate_write = 1200;
    break;

  case 0x0c:		/* RS-232C ? */
  case 0x34:		/* ???? */
    break;
  }
}

/*
 * uPD8255 (PPI)
 *  implemented mode:
 *   group A (port A + port C upper) = mode 2
 *   group B (port B + port C lower) = mode 0
 *   port A = output
 *   port B = output
 *   port C upper 5bits = port A status
 *   port C lower 3bits = output
 */
void upd8255(int reset, int address)
{
  FILE *printer;
  static byte portc_old = 0;
  static int nrd_old = 1, nwr_old = 1;
  static int nack_old = 1, nstb_old = 1;
  static int print_data;
  static byte ibuffer, obuffer;

  if (reset) {
    ibuffer = 0xff;
    obuffer = 0xff;
    port_c = 0xff;
    return;
  }

  switch (address) {
  case 0:	/* port A (mode 2 output) */
    if (RISING(nrd, nrd_old)) {
      dbus_8255 = ibuffer;
      RESBIT(IBF);
    }
    if (RISING(nwr, nwr_old)) {
      obuffer = dbus_8255;
      RESBIT(NOBF);
    }
    if (FALLING(nack, nack_old)) {
      port_a = obuffer;
      SETBIT(NOBF);
    }
    if (FALLING(nstb, nstb_old)) {
      ibuffer = port_a;
      SETBIT(IBF);
    }
    /* interrupt(PC3): refered to uPD8255A data sheet */
    if (((port_c & IBF) && (port_c & INTE2) && nrd && nstb)
	|| ((port_c & NOBF) && (port_c & INTE1) && nwr && nack)) {
      SETBIT(INTR);
    } else {
      RESBIT(INTR);
    }
    break;

  case 1:	/* port B (mode 0 output) */
    if (RISING(nrd, nrd_old)) {
      dbus_8255 = port_b;
    }
    if (RISING(nwr, nwr_old)) {
      port_b = dbus_8255;
    }
    break;

  case 2:	/* port C (upper 5bits=port A status, lower 3bits=mode0 output) */
    if (RISING(nrd, nrd_old)) {
      dbus_8255 = port_c;
    }
    if (RISING(nwr, nwr_old)) {
      port_c = (port_c & 0xf8) | (dbus_8255 & 0x07);
    }
    break;

  case 3:	/* control register */
    if (RISING(nrd, nrd_old)) {
      dbus_8255	= port_c | ibuffer | IBF;	/* ? */
    }
    if (RISING(nwr, nwr_old)) {
      if (dbus_8255 & 0x80) {	/* 8255 mode set */
	/* not implemented */
	ibuffer = 0xff;
	obuffer = 0xff;
	port_c = portc_old = NOBF | INTE1 | INTE2 | INTR | 0x07;
      } else {			/* port C bit set/reset */
	if (dbus_8255 & 0x01) {
	  SETBIT(1 << ((dbus_8255 & 0x0e) >> 1));
	} else {
	  RESBIT(1 << ((dbus_8255 & 0x0e) >> 1));
	}
      }
      /* interrupt(PC3): refered to uPD8255A data sheet */
      if (((port_c & IBF) && (port_c & INTE2) && nrd && nstb)
	  || ((port_c & NOBF) && (port_c & INTE1) && nwr && nack)) {
	SETBIT(INTR);
      } else {
	RESBIT(INTR);
      }
    }
    break;
  }

  /* printer */
  if (RISING(port_c & PRTSTRB, portc_old & PRTSTRB)) {
    print_data = port_b;
  }
  if (FALLING(port_c & PRTSTRB, portc_old & PRTSTRB)) {
    if ((printer = fopen("printer.txt", "ab")) != NULL) {
      fputc((~print_data) & 0xff, printer);
      fclose(printer);
    }
  }

  portc_old = port_c;
  nrd_old = nrd;
  nwr_old = nwr;
}

/*
 * external FDD
 */
void extfdd(int reset)
{
  int i;
  static int idata;
  static int nrcv, nsend;
  static int count_extfdd;
  static int period;
  static int mode, nextmode;
  static int cmdstat, drvstat;
  static int sectors, drive, track, sector;
  static byte data[0x1000];
  static byte wrtbf[0x1000], readbf[0x1000];
  int wait_clock[NUM_OF_EXTFDD_CMD + 12] = {
    CLOCK_4M * 0.3,
    CLOCK_4M * 0.2,
    CLOCK_4M * 0.2,
    500,
    CLOCK_4M * 0.2,
    CLOCK_4M * 1.0,
    500,
    500,

    20,
    20,
    20,
    20,
    20,
    20,
    20,
    20,
    20,
    20,
    20,
    20,
  };
  enum {
    FDUINZ = 0, WRTDSK1, RDDSK1, TRNBUF1, COPY, FORMAT1, CMDSTAT, DRVSTAT,
    IDLE1, IDLE2, RCV1, RCV2, SEND1, SEND2, SEND3,
    WRTDSK2, WRTDSK3, RDDSK2, TRNBUF2, FORMAT2,
  };

  if (reset) {
    port[0xd2] = 0;
    mode = IDLE1;
    count_extfdd = 0;
    cmdstat = 0;
    drvstat = 0;
    return;
  }

  period = wait_clock[mode];
  count_extfdd += cpu.cycles;
  if (count_extfdd < period) return;
  count_extfdd = 0;

  switch (mode) {
  case IDLE1:
    if (port[0xd2] & ATN_PC) {
      mode = RCV1;
      nrcv = 1;
      idata = 0;
      nextmode = IDLE2;
    }
    break;
  case IDLE2:
    if (data[0] >= NUM_OF_EXTFDD_CMD) {
      mode = IDLE1;
    } else {
      mode = data[0];
    }
    break;

  case RCV1:
    port[0xd2] |= RFD_FDD;
    if (port[0xd2] & DAV_PC) {
      port[0xd2] &= ~RFD_FDD;
      if (idata < sizeof(data)) {
	data[idata] = port[0xd1];
      }
      port[0xd2] |= DAC_FDD;
      mode = RCV2;
    } else if (nextmode != IDLE2 && (port[0xd2] & ATN_PC)) {
      mode = IDLE1;
    }
    break;
  case RCV2:
    if ((port[0xd2] & DAV_PC) == 0) {
      port[0xd2] &= ~DAC_FDD;
      if (++idata >= nrcv) {
	mode = nextmode;
      } else {
	mode = RCV1;
      }
    } else if (port[0xd2] & ATN_PC) {
      mode = IDLE1;
    }
    break;

  case SEND1:
    if (port[0xd2] & RFD_PC) {
      if (idata < sizeof(data)) {
	port[0xd0] = data[idata];
      }
      port[0xd2] |= DAV_FDD;
      mode = SEND2;
    } else if (port[0xd2] & ATN_PC) {
      mode = IDLE1;
    }
    break;
  case SEND2:
    if (port[0xd2] & DAC_PC) {
      port[0xd2] &= ~DAV_FDD;
      mode = SEND3;
    } else if (port[0xd2] & ATN_PC) {
      mode = IDLE1;
    }
    break;
  case SEND3:
    if ((port[0xd2] & DAC_PC) == 0) {
      if (++idata >= nsend) {
	mode = nextmode;
      } else {
	mode = SEND1;
      }
    } else if (port[0xd2] & ATN_PC) {
      mode = IDLE1;
    }
    break;

  case FDUINZ:
    period = CLOCK_4M * 1.0;
    drvstat = 0;
    for (i = 0; i < ndrive; i++) {
      drvstat = (drvstat << 1 ) | 0x11;
    }
    count_extfdd = 0;
    mode = IDLE1;
    cmdstat = 0x80;	/* complete */
    break;

  case WRTDSK1:
    mode = RCV1;
    nrcv = 4;
    idata = 0;
    nextmode = WRTDSK2;
    break;

  case WRTDSK2:
    sectors = data[0];
    drive = data[1];
    track = data[2];
    sector = data[3];
    if (drive > ndrive || track > 40 || sector > 16) {
      mode = IDLE1;
      cmdstat = 0x80;  /* complete */
      if (is_d88[drive] && disk[drive][0x001a]) {
	cmdstat |= 0x01;  /* error (not writable) */
      }
    } else {
      mode = RCV1;
      idata = 0;
      nrcv = sectors * 0x100;
      nextmode = WRTDSK3;
      ;
    }
    break;

  case WRTDSK3:
    if (sector + sectors - 1 <= 16) {
      nrcv = sectors * 0x100;
    } else {
      nrcv = (17 - sector) * 0x100;
    }
    memcpy(wrtbf, data, nrcv);

    mode = IDLE1;
    cmdstat = 0x80;  /* complete */
    if (is_d88[drive]) {
      if (disk[drive][0x001a]) {
	cmdstat |= 0x01;  /* error (not writable) */
      } else {
	byte *p;
	for (i = 0; i < nrcv / 0x100; i++) {
	  if ((p = getd88(drive, track, sector + i))) {
	    memcpy(p, wrtbf + i * 0x100, 0x100);
	  }
	}
      }
    } else {
      memcpy(&disk[drive][256 * (16 * track + sector - 1)], wrtbf, nrcv);
    }
    if ((cmdstat & 0x01) == 0) {
      FILE *disk_write;
      int size;
      if (disk_write = fopen(diskfile[drive], "wb")) {
	size = is_d88[drive] ? (256 + 16) * 16 * 40 + 0x2b0 : 256 * 16 * 40;
	fwrite(disk[drive], 1, size, disk_write);
	fclose(disk_write);
      }
    }
    break;

  case RDDSK1:
    mode = RCV1;
    nrcv = 4;
    idata = 0;
    nextmode = RDDSK2;
    break;

  case RDDSK2:
    sectors = data[0];
    drive = data[1];
    track = data[2];
    sector = data[3];
    cmdstat = 0xc0;	/* complete and read buffer ok */
    if (drive >= ndrive || track > 40 || sector > 16) {
      mode = IDLE1;
      cmdstat = 0x81;	/* complete and error */
      break;
    } else {
      if (sector + sectors  - 1 <= 16) {
	nsend = sectors * 0x100;
      } else {
	nsend = (17 - sector) * 0x100;
      }
      if (is_d88[drive]) {
	byte *p;
	for (i = 0; i < nsend / 0x100; i++) {
	  if ((p = getd88(drive, track, sector + i))) {
	    memcpy(readbf + i * 0x100, p, 0x100);
	  } else {
	    mode = IDLE1;
	    cmdstat = 0x81;	/* complete and error */
	    break;
	  }
	}
      } else {
	memcpy(readbf, &disk[drive][256 * (16 * track + sector - 1)], nsend);
      }
      nsend = sectors * 0x100;
    }
    mode = IDLE1;
    break;

  case TRNBUF1:
    if (sector + sectors  - 1 <= 16) {
      nsend = sectors * 0x100;
    } else {
      nsend = (17 - sector) * 0x100;
    }
    memcpy(data, readbf, nsend);
    mode = SEND1;
    nsend = sectors * 0x100;
    idata = 0;
    nextmode = TRNBUF2;
    break;
  case TRNBUF2:
    cmdstat &= 0xbf;
    mode = IDLE1;
    break;

  case FORMAT1:
    mode = RCV1;
    nrcv = 1;
    idata = 0;
    nextmode = FORMAT2;
    break;
  case FORMAT2:
    drive = data[0];
    if (drive < ndrive) {
      cmdstat = 0xc0;	/* complete and read buffer ok */
    } else {
      cmdstat = 0x81;	/* complete and error */
    }
    mode = IDLE1;
    break;
    
  case CMDSTAT:
    mode = SEND1;
    nsend = 1;
    idata = 0;
    data[0] = cmdstat;
    nextmode = IDLE1;
    break;

  case DRVSTAT:
    mode = SEND1;
    nsend = 1;
    idata = 0;
    data[0] = drvstat;
    nextmode = IDLE1;
    break;

  default:
    break;
  }
}


/*
 * interrupt
 */
int interrupt(int intr8049)
{
  int period;
  int intack;
  static int intr_timer;
  static byte vector;

  /* 2ms timer */
  if (mk2) {
    period = (port[0xf6] + 1) * 2048;
  } else {
    period = 8192;
  }
  count_timer += cpu.cycles;
  cpu.cycles = 0;

  if (timer == 0 || (mk2 && (port[0xf3] & 0x04))) {
    intr_timer = 0;
  } else if (count_timer >= period) {
    intr_timer = 1;
  }
  count_timer %= period;

  /* interrupt? */
  if (mk2 && (port[0xf3] & 0x01)) {
    intr8049 = 0;
  }
  cpu.intr = intr8049 || intr_timer;
  if (cpu.intr == 0) {
    return 0;
  }

  /* CPU */
  intack = z80_intack(&cpu);
  if (intack) {
    if (intr8049) {
      vector = readport(0x90);
    } else {
      vector = mk2 ? port[0xf7] : 0x06;
    }
    z80_int(&cpu, vector);

    cpu.intr = 0;
    intr_timer = 0;
    if (vector == break_vector) {
      break_intr = 1;
    }
  }

  return intack;
}

/*
 * key input
 */
void keyinput(int *inkeyp, int *gamekeyp)
{
  int i;
  int y;
  wint_t c;
  char *fname, *fname_expand;
  static int framecount;
  static int period;

  const wchar_t dakuon[2][41] = {
    L"がぎぐげござじずぜぞだぢづでどばびぶべぼガギグゲゴザジズゼゾダヂヅデドバビブベボヴ",
    L"かきくけこさしすせそたちつてとはひふへほカキクケコサシスセソタチツテトハヒフヘホウ"
  };
  const wchar_t handakuon[2][10] = {
    L"ぱぴぷぺぽパピプペポ",
    L"はひふへほハヒフヘホ"
  };

  if (*gamekeyp == 0xff) {	/* reset */
    *gamekeyp = 0;
    period = mk2 ? 80 / KEY_TICK_FACTOR : 1;
    return;
  }

  if (count_auto != NONE && p6t_iauto < p6t.nauto &&
      mode_8049 == IDLE && !p6t_waittape) {
    count_auto++;
    if (count_auto >= period) {
      count_auto = 0;
      if (p6t_iauto == -3) {
	if (mk2) {
	  *inkeyp = '0' + p6t.mode;
	}
	p6t_iauto++;
	period = 40 / KEY_TICK_FACTOR;
      } else if (p6t_iauto == -2) {
	*inkeyp = '0' + p6t.pages;
	p6t_iauto++;
	period = 1;
      } else if (p6t_iauto == -1) {
	*inkeyp = 0x0d;
	p6t_iauto++;
	period = (p6t.mode == 5 ? p6t.pages * 10 + 60 : 30) / KEY_TICK_FACTOR;
      } else {
	*inkeyp = p6t.autostring[p6t_iauto++];
	if (*inkeyp == 0x0a) {
	  *inkeyp = 0x0d;
	  p6t_waittape = 1;
	}
	period = (*inkeyp == 0x0d) ? 40 / KEY_TICK_FACTOR : 1;
      }
      return;
    }
  }

  framecount++;
  if (framecount % 12 == 0) {
    *gamekeyp = 0;
  }
  while (get_wch(&c) != ERR) {
    if (c == KEY_F(23)) {			/* shift+F11: exit */
      endwin();
      exit(EXIT_SUCCESS);
    } else if (c == KEY_F(6)) {			/* F6: monitor mode */
      break_monitor = 1;
    } else if (c == KEY_F(10)) {		/* F10: no wait mode */
      normal_speed = !normal_speed;
    } else if (c == KEY_F(11)) {		/* F11: reset */
      reset_p6();
      *inkeyp = ERR;
    } else if (c == KEY_F(9)) {			/* F9: set tape file */
      y = getmaxy(stdscr) - 1;
      fname = getinput(y, "tape file: ");
      if (strlen(fname) > 0) {
	fname_expand = tilde_expand(fname);
	if (tape_read = read_open(fname_expand)) {
	  mvprintw(y, 0, "open %s", fname_expand);
	  tape_counter = 0;
	} else {
	  mvprintw(y, 0, "cannot open %s", fname_expand);
	}
	clrtoeol();
	free(fname_expand);
      }
      free(fname);
    } else if (c == KEY_F(21)) {		/* shift+F9: set extrom file */
      FILE *extrom_file;
      y = getmaxy(stdscr) - 1;
      fname = getinput(y, "external rom file: ");
      if (strlen(fname) > 0) {
	fname_expand = tilde_expand(fname);
	if (extrom_file = fopen(fname_expand, "rb")) {
	  mvprintw(y, 0, "open %s", fname_expand);
	  if (fread(extrom, 1, sizeof(extrom), extrom_file) <= 0x4000) {
	    use_extrom = 1;
	    memcpy(&extrom[0x4000], extrom, 0x4000);
	    memcpy(&extrom[0x8000], extrom, 0x4000);
	    memcpy(&extrom[0xc000], extrom, 0x4000);
	  } else {
	    use_beluga = 1;
	  }
	  fclose(extrom_file);
	  reset_p6();
	} else {
	  mvprintw(y, 0, "cannot open %s", fname_expand);
	}
	clrtoeol();
	free(fname_expand);
      }
      free(fname);
    } else if (c == KEY_F(12)) {		/* F12: set disk file */
      char prompt[30];
      y = getmaxy(stdscr) - 1;
      for (i = 0; i < ndrive; i++) {
	add_history(diskfile[i]);
	snprintf(prompt, sizeof(prompt), "disk file(drive %d): ", i + 1);
	fname = getinput(y, prompt);
	if (strlen(fname) > 0) {
	  fname_expand = tilde_expand(fname);
	  if (disk_open(fname_expand, i)) {
	    mvprintw(y, 0, "open %s", fname_expand);
	  } else {
	    mvprintw(y, 0, "cannot open %s", fname_expand);
	  }
	  clrtoeol();
	  free(fname_expand);
	}
	free(fname);
      }
    } else if ((ikbf + 1) % (sizeof(keybuf) / sizeof(keybuf[0])) != okbf) {
      for (i = 0; i < sizeof(dakuon[0]) / sizeof(dakuon[0][0]); i++) {
	if (c == dakuon[0][i]) {
	  keybuf[ikbf = (ikbf + 1) % (sizeof(keybuf) / sizeof(keybuf[0]))]
	    = dakuon[1][i];
	  c = L'゛';
	}
      }
      for (i = 0; i < sizeof(handakuon[0]) / sizeof(handakuon[0][0]); i++) {
	if (c == handakuon[0][i]) {
	  keybuf[ikbf = (ikbf + 1) % (sizeof(keybuf) / sizeof(keybuf[0]))]
	    = handakuon[1][i];
	  c = L'゜';
	}
      }
      keybuf[ikbf = (ikbf + 1) % (sizeof(keybuf) / sizeof(keybuf[0]))] = c;

      switch (c) {
      case ' ':			/* space */
	*gamekeyp |= 0x80;
	break;
      case KEY_LEFT:		/* left */
	*gamekeyp |= 0x20;
	break;
      case KEY_RIGHT:		/* right */
	*gamekeyp |= 0x10;
	break;
      case KEY_DOWN:		/* down */
	*gamekeyp |= 0x08;
	break;
      case KEY_UP:		/* up */
	*gamekeyp |= 0x04;
	break;
      case KEY_BREAK:		/* stop */
      case KEY_END:		/* stop */
	*gamekeyp |= 0x02;
	break;
      case 'z':
      case 'Z':
	if (! x11key && ! winkey) {
	  *gamekeyp |= 0x01;
	}
	break;
      }
    }
  }
  if (*inkeyp == ERR) {
    if (ikbf == okbf) {
      *inkeyp = ERR;
    } else {
      okbf = (okbf + 1) % (sizeof(keybuf) / sizeof(keybuf[0]));
      *inkeyp = keybuf[okbf];
    }
  }

#ifdef WIN32
  if (winkey) {
    *gamekeyp
      = ((GetKeyState(' ')      < 0) ? 0x80 : 0)
      | ((GetKeyState(VK_LEFT)  < 0) ? 0x20 : 0)
      | ((GetKeyState(VK_RIGHT) < 0) ? 0x10 : 0)
      | ((GetKeyState(VK_DOWN)  < 0) ? 0x08 : 0)
      | ((GetKeyState(VK_UP)    < 0) ? 0x04 : 0)
      | ((GetKeyState(VK_END)   < 0) ? 0x02 : 0)
      | ((GetKeyState(VK_SHIFT) < 0) ? 0x01 : 0);
  }
#endif

#ifdef X11
  if (x11key) {
    char keys[32];
    KeyCode code;

    *gamekeyp = 0;
    XQueryKeymap(dispX11, keys);

    code = XKeysymToKeycode(dispX11, XK_space);
    if (keys[code / 8] & (1 << (code % 8))) {
      *gamekeyp |= 0x80;
    }
    code = XKeysymToKeycode(dispX11, XK_Left);
    if (keys[code / 8] & (1 << (code % 8))) {
      *gamekeyp |= 0x20;
    }
    code = XKeysymToKeycode(dispX11, XK_Right);
    if (keys[code / 8] & (1 << (code % 8))) {
      *gamekeyp |= 0x10;
    }
    code = XKeysymToKeycode(dispX11, XK_Down);
    if (keys[code / 8] & (1 << (code % 8))) {
      *gamekeyp |= 0x08;
    }
    code = XKeysymToKeycode(dispX11, XK_Up);
    if (keys[code / 8] & (1 << (code % 8))) {
      *gamekeyp |= 0x04;
    }
    code = XKeysymToKeycode(dispX11, XK_End);
    if (keys[code / 8] & (1 << (code % 8))) {
      *gamekeyp |= 0x02;
    }
    code = XKeysymToKeycode(dispX11, XK_Break);
    if (keys[code / 8] & (1 << (code % 8))) {
      *gamekeyp |= 0x02;
    }
    code = XKeysymToKeycode(dispX11, XK_Shift_L);
    if (keys[code / 8] & (1 << (code % 8))) {
      *gamekeyp |= 0x01;
    }
    code = XKeysymToKeycode(dispX11, XK_Shift_R);
    if (keys[code / 8] & (1 << (code % 8))) {
      *gamekeyp |= 0x01;
    }
  }
#endif
}

/*
 * return interrupt vector and convert key code
 */
int keyvector(int inkey, int *keycode)
{
  int i;
  const int key_special[][3] = {
    {0x02, 0x00, KEY_F(7)},		/* KANA key */
    {0x02, 0x00, KEY_F(8)},		/* shift + PAGE key */
    {0x02, 0x00, KEY_SPREVIOUS},	/* shift + PAGE key */
    {0x02, 0x09, KEY_BTAB},		/* shift + tab key */
    {0x02, 0x0b, KEY_SHOME},		/* shift + HOME-CLR key */
    {0x02, 0x0c, KEY_HOME},		/* HOME-CLR key */
    {0x02, 0x12, KEY_IC},		/* INS key */
    {0x02, 0x1c, KEY_RIGHT},		/* cursor right */
    {0x02, 0x1c, KEY_SRIGHT},		/* shift + cursor right */
    {0x02, 0x1d, KEY_LEFT},		/* cursor left */
    {0x02, 0x1d, KEY_SLEFT},		/* shift + cursor left */
    {0x02, 0x1e, KEY_UP},		/* cursor up */
    {0x02, 0x1e, KEY_SR},		/* shift + cursor up */
    {0x02, 0x1f, KEY_DOWN},		/* cursor down */
    {0x02, 0x1f, KEY_SF},		/* shift + cursor down */
    {0x02, 0x7f, KEY_BACKSPACE},	/* DEL key */
    {0x02, 0x7f, KEY_DC},		/* DEL key*/
    {0x02, 0x7f, KEY_SDC},		/* shift + DEL key*/
    {0x02, 0xfe, KEY_PPAGE},		/* PAGE key */
    {0x14, 0xf0, KEY_F(1)},		/* F1 */
    {0x14, 0xf1, KEY_F(2)},		/* F2 */
    {0x14, 0xf2, KEY_F(3)},		/* F3 */
    {0x14, 0xf3, KEY_F(4)},		/* F4 */
    {0x14, 0xf4, KEY_F(5)},		/* F5 */
    {0x14, 0xf5, KEY_F(13)},		/* shift+F1 */
    {0x14, 0xf6, KEY_F(14)},		/* shift+F2 */
    {0x14, 0xf7, KEY_F(15)},		/* shift+F3 */
    {0x14, 0xf8, KEY_F(16)},		/* shift+F4 */
    {0x14, 0xf9, KEY_F(17)},		/* shift+F5 */
    {0x14, 0xfa, KEY_BREAK},		/* STOP key */
    {0x14, 0xfa, KEY_END},		/* STOP key */
  };
  const int key_special2[][3] = {
    {0x14, 0xfb, KEY_F(24)},		/* CAPS key: shift+F12 */
    {0x14, 0xfc, KEY_F(8)},		/* shift + PAGE key */
    {0x14, 0xfc, KEY_SPREVIOUS},	/* shift + PAGE key */
    {0x14, 0xfd, KEY_NPAGE},		/* MODE key */
    {0x14, 0xfe, KEY_F(7)},		/* KANA key */
  };
  const byte key_convert[][2] = {
    {0x00, 0x00}, {0x01, 0x00}, {0x02, 0x00}, {0x03, 0x00},
    {0x04, 0x00}, {0x05, 0x00}, {0x06, 0x00}, {0x07, 0x00},
    {0x08, 0x00}, {0x09, 0x00}, {0x0a, 0x00}, {0x0b, 0x00},
    {0x0c, 0x00}, {0x0d, 0x00}, {0x0e, 0x00}, {0x0f, 0x00},
    {0x10, 0x00}, {0x11, 0x00}, {0x12, 0x00}, {0x13, 0x00},
    {0x14, 0x00}, {0x15, 0x00}, {0x16, 0x00}, {0x17, 0x00},
    {0x18, 0x00}, {0x19, 0x00}, {0x1a, 0x00}, {0x1b, 0x00},
    {0x1c, 0x00}, {0x1d, 0x00}, {0x1e, 0x00}, {0x1f, 0x00},

    /*               !             "             # ぁ    */
    {0x20, 0x20}, {0x00, 0x00}, {0x00, 0x00}, {0x87, 0x00},
    /* $ ぅ          % ぇ          & ぉ          ' ゃ    */
    {0x89, 0x00}, {0x8a, 0x00}, {0x8b, 0x00}, {0x8c, 0x00},
    /* ( ゅ          ) ょ          *             +       */
    {0x8d, 0x00}, {0x8e, 0x00}, {0x00, 0x00}, {0x00, 0x00},
    /* , ね 小       - ほ ━       . る 大       / め ▲ */
    {0xe8, 0x1f}, {0xee, 0x17}, {0xf9, 0x1d}, {0xf2, 0x80},
    /* 0 わ 万       1 ぬ 日       2 ふ 月       3 あ 火 */
    {0xfc, 0x0f}, {0xe7, 0x07}, {0xec, 0x01}, {0x91, 0x02},
    /* 4 う 水       5 え 木       6 お 金       7 や 土 */
    {0x93, 0x03}, {0x94, 0x04}, {0x95, 0x05}, {0xf4, 0x06},
    /* 8 ゆ 百       9 よ 千       : け ▼       ; れ ★ */
    {0xf5, 0x0d}, {0xf6, 0x0e}, {0x99, 0x81}, {0xfa, 0x82},
    /* < 、          =             > 。          ? ・    */
    {0xa4, 0x00}, {0x00, 0x00}, {0xa1, 0x00}, {0xa5, 0x00},
    /* @ ゛          A             B             C       */
    {0xde, 0x00}, {0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00},
    /* D             E    ぃ       F             G       */
    {0x00, 0x00}, {0x88, 0x00}, {0x00, 0x00}, {0x00, 0x00},
    /* H             I             J             K       */
    {0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00},
    /* L             M             N             O       */
    {0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00},
    /* P             Q             R             S       */
    {0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00},
    /* T             U             V             W       */
    {0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00},
    /* X             Y             Z っ          [ ゜ ○ */
    {0x00, 0x00}, {0x00, 0x00}, {0x8f, 0x00}, {0xdf, 0x84},
    /* \ ー 円       ] む ●       ^ へ          _ ろ ◆ */
    {0xb0, 0x09}, {0xf1, 0x85}, {0xed, 0x00}, {0xfb, 0x83},
    /* `             a ち          b こ ┛       c そ ┗ */
    {0x00, 0x00}, {0xe1, 0x00}, {0x9a, 0x1b}, {0x9f, 0x1a},
    /* d し ┣       e い ┏       f は ╋       g き ┫ */
    {0x9c, 0x14}, {0x92, 0x18}, {0xea, 0x15}, {0x97, 0x13},
    /* h く 時       i に ┃       j ま          k の    */
    {0x98, 0x0a}, {0xe6, 0x16}, {0xef, 0x00}, {0xe9, 0x00},
    /* l り 中       m も 分       n み          o ら    */
    {0xf8, 0x1e}, {0xf3, 0x0b}, {0xf0, 0x00}, {0xf7, 0x00},
    /* p せ π       q た          r す ┳       s と 秒 */
    {0x9e, 0x10}, {0xe0, 0x00}, {0x9d, 0x12}, {0xe4, 0x0c},
    /* t か ┓       u な          v ひ ┻       w て    */
    {0x96, 0x19}, {0xe5, 0x00}, {0xeb, 0x11}, {0xe3, 0x00},
    /* x さ ×       y ん 年       z つ          { 「    */
    {0x9b, 0x1c}, {0xfd, 0x08}, {0xe2, 0x00}, {0xa2, 0x00},
    /* |             } 」          ~                     */
    {0x00, 0x00}, {0xa3, 0x00}, {0x00, 0x00}, {0x7f, 0x7f}
  };

  if (inkey == 0x1b) {
    if (ikbf == okbf) {
      inkey = ERR;
    } else {
      okbf = (okbf + 1) % (sizeof(keybuf) / sizeof(keybuf[0]));
      inkey = keybuf[okbf];
    }
    if (inkey == ERR) {
      *keycode = 0x1b;
      return 0x02;
    }

    if (inkey <= 0x7f) {
      *keycode = key_convert[inkey][1];
      if (*keycode < 0x20) {
	return 0x14;
      } else {
	return 0x02;
      }
    } else {
      *keycode = 0;
      return 0x14;
    }
  }

  if (inkey < 0x80) {
    if (kana_mode) {
      *keycode = key_convert[inkey][0];
      if (katakana_mode
	  && ((0x86 <= *keycode && *keycode <= 0x9f)
	      || (0xe0 <= *keycode && *keycode <= 0xfd))) {
	*keycode ^= 0x20;
      }
    } else {
      *keycode = inkey;
    }
    return 0x02;
  }

  if (mk2) {
    for (i = 0; i < sizeof(key_special2) / sizeof(key_special2[0]); i++) {
      if (inkey == key_special2[i][2]) {
	if (inkey == KEY_F(7)) {
	  kana_mode = !kana_mode;
	} else if (inkey == KEY_F(8) || inkey == KEY_SPREVIOUS) {
	  katakana_mode = !katakana_mode;
	}
	*keycode = key_special2[i][1];
	return key_special2[i][0];
      }
    }
  }

  for (i = 0; i < sizeof(key_special) / sizeof(key_special[0]); i++) {
    if (inkey == key_special[i][2]) {
      if (inkey == KEY_F(7)) {
	kana_mode = !kana_mode;
      } else if (inkey == KEY_F(8) || inkey == KEY_SPREVIOUS) {
	katakana_mode = !katakana_mode;
      }
      *keycode = key_special[i][1];
      return key_special[i][0];
    }
  }

  for (i = 0x20; i < 0x100; i++) {
    if (inkey == dispchar[i]) {
      *keycode = i;
      return 0x02;
    }
  }
  for (i = 0; i < 0x20; i++) {
    if (inkey == dispchar[i]) {
      *keycode = i;
      return 0x14;
    }
  }

  *keycode = 0;
  return 0x14;
}

/*
 * display
 */
void display()
{
  int i, j;
  int x, y;
  int fore, back;
  int sum;
  cchar_t wch;
  int graphic_mode;
  static int mode5_old;
  int color_used[16];
  int color1, color2;
  byte pattern[16][12];
  int weight[2];
  int ul, ur, dl, dr;
  int bitmap[160][40];
  char render[81] = {};
  byte table[] =
    " ._.____...___jj..______..______,,__,,,L,,____Ly,,___Lyy,,__Lyyy"
    "...jjjjj.._jjjjj.._jjjju.__jjxuo,,_jjuoy,>vjuyyy,jvuoyyy,cuoyyyy"
    "^.jjjjjj.jjjjjjj//_jjjjj/_jjjunn,,jjjzaa,jjjzyyy,)znayyg/zzjaygg"
    ".^_jjjj]./jjjj]1//jjj]11/>jju11q,jv]11aa,c]n1yyq,]z11yqgYzJwaqgg"
    "^^_jjj]1/jjjj]11/>jj]JJJ}jvu11dd,cJ212y2,zJeyy2ZYzJdydgO7JJddgOG"
    "^^jjj]11/<jj]111/jJ]JJJ3rcJJ1qdd,zJJay22YzJwyggg7JJddgOW7J2d4gG@"
    "^<jjj111/jj]1111/)JJ1J33,cJJdqq4,zJwa2gO7JJygggW7J2dggW@722dOG@@"
    "^jjj111yjj]j1113/JJJ1333YzJdaq4O,zJa2gOG7w22ggW87y2dOW@@72ZOG@@@"
    "''''''<j..___<jx..__{lxu.___lxuo,i_LxuLL,ivxuLyy,lvuLyyy,cLLyyyy"
    "^^^^^<jx.<<j<jxu//v>vxuu/ivxxuos,>vxuyyy}vvuyyyyrcuoyyyyrcoeyyyy"
    "^^^^jjj]///jjj]1//>j)un1/>vxunna/zvuzaaarzznayyg/zzeayggYzCwaggg"
    "^^^]1]]1/<<j1]11/>v]]111}vv]11qqrz]z1yyqY]zeyyqgYzJwyqggfzVyqggW"
    "^^<j1]11/<]]]111/)J]1JJ3rcJd1dddYzJ2ay22YzJwy2gZ7JJddgOG7J2d4OG@"
    "^<J]]111/<J]1111/JJJJJ337JJJdyd4YzJ2a2gO7JJydggW7JJd4gG@7J2dOG@@"
    "^)J]111y/]]]1113/JJJ1333YJJdaq4OzJJa2ggG7Jy2ggW07y2dgG@@72ZOG@@@"
    "^)]]11y3/]J11139YJJJ33357JJddqOG7z22ggG07y2mgWg@72ZOG@@@2ZZW@@@#"
    "''''i<i'!!!i<<iu!ii>i)un:iiicooo:iLLLLLw}iLLLewy:cLLewyg:cLewygg"
    "^^^^<<cu!<<<<<uo/>>))uoo:>vcuoow}ccooewarceoewygrcoewyggczewaggg"
    "^^^<j)nn^<<j)nn1/>))nnna})cnnqaarzzzaaagzzzeaaggYzewagggzzeagggg"
    "^^<]1]11^<)]]111>))]11qqrcnn1qqqYzzeaaqgYzewaqggzzVaqggWfzadggWg"
    "^<)]]11d<)]]111dr)Jn1333YcodddddYz22a222zz22dggG722d4gG@722d4G@@"
    "^)J]1133r)]]1133))JJ3333YzJdaqdOzz222g2G7w22ggW0722dgg@@72ZOG@@@"
    "^)]n11a3)]J11339YJJ3333377VddqqG7z22ggG07222ggg072ZOG@@@2ZZW@@@#"
    "^]J11a34YzJ3aq497JJ3343%77dd4qG87222ag0072Z2Gg@#2ZZW@@@#PZZ@@@#@"
    "'''''l'L!!!{llLL!i{llLLL:llLLLLL:lLLLLLallLLLLyp:LLLLypmYLLLapmm"
    "'^^<((uo!<<j(uos:{((uoCX}lcuossyrcLLswyprILpwyppYLCwypppYkpppppp"
    "^^^j^^CC^<)j]nCX}))nnCXarcCnCXaqYzLCaaggYzCmammmYCmmmmmmTmKmmmmm"
    "^^^]]11V}(]]11VVY(Vn1VV3YYVUVqUUYzVXaUUDYVVpmmWWfVVUUWW@7VmUWWW@"
    "^^]]11V3r(]]1V33YYJ34334Y7VX444O77V24ZOG7ZZpZZG@7ZZZOG@@7ZZOG@@@"
    "^(]313337YJ33339Y7J3333377V344OG77Z24OG07ZZmZW007ZZOG@@@7ZZW@@@#"
    "^YJ33339Y7J3339977V3335G77VU4OG87ZZUOG007pZZG@@@7ZZW@@@@ZZZ@@@#@"
    "Y7CX339977V3399G77V345%87VVUOG807ZZOG0007ZZWG@@#ZZZ@@@@@ZZZ@@#@@"
    "''''''''''<llILh:{llILhh:lLILhhKlLLLLLyblILLwypmILLwypm&LkLppg&W"
    "''<''Ihh'<((uhhh}((uhhhKrIIohhKbIILkwybbIkkwypp&Ikpppp&WkkKpp&W@"
    "^^^junCX}))]1CXKr)CnC%KSYICCKKq6YzCwaggDTkKmmmmWTKKmm&W@TKmm&%@@"
    "^^]]11VSf)]]11SSfYVSSSSSYfVXKq6OfkVKUDDWfVKWWWWWfKmWWW@@fmZWWW@@"
    "'']]1V3Sf]J113S9YfJS3SS5ffVOGOOGfkK2GOGG7ZZOOG@@7ZZOG@@@PZZW@@@@"
    "'IJ99999Yf999999f7V339AG77KOGOG87KZ0OG007ZZOG000PZZW@@@@PAW@@@#@"
    "YfJ99999ff999999f7K3S5G87KKOOG807bZOG000PZZW000#PZW@@@@@PW@@@#@@"
    "ffV9999Gf79999G97K3S5%807K9OG800PZDW000#PZW@@@##PW@@@@@@W@@@#@@@"
    "''''''Lh'{llLLhh:lLhLhhKllLLhhKb:LLLLyb&ILL&yp&&TkL&&&&&TkKb&&&&"
    "'<(xuhhh}<(uhhhKr(hhhhK%TIohhKb6Tkkbab&&Tkp&&&&&TkK&&&&&TKb&&&&&"
    "^^)TTT%KT)CnT%K%TTCTT%%%TTKKK%6%TTKKbg&%TTKm&&%&TKK&&%%@TKm&N%@#"
    "^)]S1SSST]]S1SSSYTVSSSS5TTVKN6NNTFKbNDNNFKm6&NNBFbE&NN@$FDDWN@$#"
    "^YJ55555YTJ55555TTV55555TFKdN5N8FK255NBBPZZ5N#B#PZZ@@@##P5E@@###"
    "YT999999T9999999T7K555N57KR9NN80PPZDN000PZAW#0##PAE00###PRR0####"
    "TT99995NT99995N9FK955%5BPP95N8BBPPDBBBB#P&EB####PERB####PRBB####"
    "TFK995NNF9995%NBPP95N%BBPPA88BB#PDEBB###PERB####PRBB####PBB#####"
    "'''''Lhh!lLLL%%%llLLh%%%:ILhhK%%ILLLab&&LkLpp&&%TkKp&&%%FKb&&%%%"
    "'(vuh%%%r(Ihh%%%rIhhh%%%IIhhK%%%Tkbbb&&%TkKp&&%&FKp&&%%@Fb&&&%%$"
    "^)Cn1%%%TTCT%%%%TTCF%%%%TTKKq%%%TFKbg%%%FKmm&%%%Fmm&%%%$FmE&%%$%"
    "^Y]11%S%YFVSS%%%FFVSS%%%FFKUN%NNFEEEDEEEFEEDN%$$FEEEN@$#PEEE@$#$"
    "YFJ55%55FFV55%5%FFK555%MFPROO%RBPPEOGRB$PZEW#B$#PEE@@@##PER@####"
    "PP99999%PF9999%MFPP95%MMPPRANR00PPEE000#PREM0###PER0####PRR0####"
    "FFK99%%MFP995%MMPP95N%%%PPRMR00#PEE00###PER0####PRR$####PR$#####"
    "FP995%%%PP95N%%%PP5MM%%%PRRMBB##PERB####PRR$####P@$#####P$######"
    "''''LhhK!lLLh%%%:ILhh%%%ILLhK%%%LkLbb&&%TkKp&&%$FKp&&%$$Fpm&&%$$"
    "'(Ioh%%%rIChh%%%YThha%%%TkKKb%%%TkKb&&%$Fpp&&%$$Fpp&&%$$F&&&$$$$"
    "^^CCa%%%TTCXa%%%TTKK%%%%TFK66%%%FKK6&%%$Fmm&%%$$FmE$$$$$P&E$$$$$"
    "YfVSS%S%ffVSS%%%fFKSS%%%FKK6N%$$FEEDE$$$F$$$$$$$P$E$$$$$$E$$$$$$"
    "ffV5455%fFK555%MFP555%%$PPROGR$$PZEE$$$$P$E$$$$$PE$@$$$$P$$$#$$$"
    "7F9999%MF9999%M0PP95N%0$PPR880$#PEE00$#$PER$$#$$PR$$#$$$P$$$#$$$"
    "FP995%M$PP99N%$$PP55M%$$PRR80$$$PER0$#$$P$$$#$$$$$$$#$$$$$$$#$$$"
    "PP95N%$$P999M%$$P5RM$$$$PRR0$$$$ER$$#$$$$$$$#$$$$$$$$$$$$$$$$$$$";

  const wchar_t semichar[]
    =L"　，、＿‘【／」’＼】Ｌ￣乁「■"
    "＿＿＿＿■■■■■■■■■】】】"
    "▲▲▼▼＃※○○→←↑↓／／＼＼"
    "０｜２３４５６７８９Ю‖仁∞∝≒"
    "ａｉｕｅｅｅｅａｕｏΒΔΛΣΦΩ"
    "αβγδεθλμνρσυφωφ∂"
    "ａΛｅａｅｏａｉｕｅεｅｏｙＹＩ"
    "ｎЗ∫∂ｎＲｃｒφшчΩａεｅｏ"
    "×÷±干≧≠≦＿）￣＿√∩∪∴∵"
    "⌒∪△▲★＋■□□×電＊※Ｓ§Φ"
    "♂♀凹凸◎◎◎◎◎◎◎◎◎◎◎◎"
    "廾卍凸＋〒乙◇文×Ｈ○占古↓↓士"
    "●●■‖≪×￣＜￣＼＿＿＝●‖‖"
    "●●■‖×≪≪／￣＿／＼＝‖●‖"
    "●●■‖×≫≫＼￣＿＼／＝‖‖‖"
    "●●■Ｖ≫×￣＞￣／＿＿＝●○■";

  if (mode5) {
    if (port[0xc1] & 0x04) {

      /* semi-graphic mode
	 (x,y) PC-6001 text coordinates
	 (2x,y) terminal coordinates
	 (4x,2y) AA framebuffer coordinates
	 0.5*1 P6 mode2 dots in framebuffer pixel
      */
      for (y = 0; y < 20; y++) {
	for (x = 0; x < 40; x++) {
	  i = x + y * 40;
	  ul = (vram[i + 0x400] & 0x08 ? vram[i] : (vram[i] >> 4)) & 0x07;
	  ur = (vram[i + 0x400] & 0x04 ? vram[i] : (vram[i] >> 4)) & 0x07;
	  dl = (vram[i + 0x400] & 0x02 ? vram[i] : (vram[i] >> 4)) & 0x07;
	  dr = (vram[i + 0x400] & 0x01 ? vram[i] : (vram[i] >> 4)) & 0x07;
	  render[x * 2] = table[(ul << 9) + (ul << 6) + (dl << 3) + dl];
	  render[x * 2 + 1] = table[(ur << 9) + (ur << 6) + (dr << 3) + dr];
	}
	mvprintw(y, 0, "%s", render);
      }

      /* text character */
      for (i = 0; i < 0x320; i++) {
	//	if (vram[i] & 0x80) continue;
	x = i % 40;
	y = i / 40;
	fore = ((vram[i] >> 2) & 1) + ((vram[i] >> 1) & 1) + ((vram[i] >> 0) & 1);
	back = ((vram[i] >> 6) & 1) + ((vram[i] >> 5) & 1) + ((vram[i] >> 4) & 1);
	if ((vram[i] & 7) == 0 && ((vram[i] >> 4) & 7) == 0) {
	  setcchar(&wch, &dispchar[0], A_NORMAL, 0, NULL);
	} else if ((vram[i] & 8) && (vram[i] & 7) == ((vram[i] >> 4) & 7)) {
	  setcchar(&wch, &dispchar[0], A_REVERSE, 0, NULL);
	} else if (fore > back) {
	  if (vram[i] & 0x80) {
	    setcchar(&wch, &semichar[vram[0x400 + i]], A_NORMAL, 0, NULL);
	  } else {
	    setcchar(&wch, &dispchar[vram[0x400 + i]], A_NORMAL, 0, NULL);
	  }
	} else {
	  if (vram[i] & 0x80) {
	    setcchar(&wch, &semichar[vram[0x400 + i]], A_REVERSE, 0, NULL);
	  } else {
	    setcchar(&wch, &dispchar[vram[0x400 + i]], A_REVERSE, 0, NULL);
	  }
	}
	mvadd_wch(y, x * 2, &wch);
      }
    } else {
      /* graphic mode */
      for (y = 0; y < 20; y++) {
	for (x = 0; x < 40; x++) {
	  for (i = 0; i < 4; i++) {
	    weight[0] = 0;
	    weight[1] = 0;
	    for (j = 0; j < 10; j++) {
	      color1 = ((vram[x + (y * 10 + j) * 40] >> (6 - i * 2)) & 0x03) +
		(((vram[x + (y * 10 + j) * 40 + 0x2000] >> (6 - i * 2)) & 0x03) << 2);
	      if (port[0xc1] & 0x08 && color1 == 0x08) {
		color1 = 0;
	      } else if (port[0xc0] & 0x04) {
		color1 &= 0x07;
	      }
	      weight[j / 5] += color1;
	    }
	    bitmap[x * 4 + i][y * 2 + 0] = (weight[0] + 11) / 12;
	    bitmap[x * 4 + i][y * 2 + 1] = (weight[1] + 11) / 12;
	  }
	  for (i = 0; i < 2; i++) {
	    ul = bitmap[x * 4 + i * 2    ][y * 2    ];
	    ur = bitmap[x * 4 + i * 2 + 1][y * 2    ];
	    dl = bitmap[x * 4 + i * 2    ][y * 2 + 1];
	    dr = bitmap[x * 4 + i * 2 + 1][y * 2 + 1];
	    render[x * 2 + i] = table[(ul << 9) + (ur << 6) + (dl << 3) + dr];
	  }
	}
	mvprintw(y, 0, "%s", render);

	if (port[0xc1] & 0x08) {
	  /* screen mode 3 text character */
	  for (x = 0; x < 20; x++) {
	    memset(color_used, 0, sizeof(color_used));
	    memset(pattern, 0, sizeof(pattern));

	    for (i = 0; i < 10; i++) {
	      for (j = 0; j < 4; j++) {
		color1 = ((vram[x * 2 + (y * 10 + i) * 40] >> (6 - j * 2)) & 0x03)
		  | ((vram[x * 2+ (y * 10 + i) * 40 + 0x2000] >> (6 - j * 2) << 2) & 0x0c);
		color2 = ((vram[x * 2 + (y * 10 + i) * 40 + 1] >> (6 - j * 2)) & 0x03)
		  | ((vram[x * 2 + (y * 10 + i) * 40 + 0x2001] >> (6 - j * 2) << 2) & 0x0c);
		pattern[color1][i] += 1 << (7 - j);
		pattern[color2][i] += 1 << (3 - j);
		color_used[color1] = 1;
		color_used[color2] = 1;
	      }
	    }
	    sum = 0;
	    color1 = -1;
	    color2 = -1;
	    for (i = 0; i < 16; i++) {
	      sum += color_used[i];
	      if (color_used[i]) {
		if (color1 == -1) {
		  color1 = i;
		} else {
		  color2 = i;
		}
	      }
	    }
	    if (sum != 2) continue;
	    for (i = 0; i < 0x100; i++) {
	      fore = color1;
	      back = color2;
	      for (j = 0; j < 10; j++) {
		if (pattern[fore][j] != cgrom2[i * 16 + j]) break;
	      }
	      if (j < 10) {
		fore = color2;
		back = color1;
		for (j = 0; j < 10; j++) {
		  if (pattern[fore][j] != cgrom2[i * 16 + j]) break;
		}
	      }
	      if (j < 10 || fore == 0x00 && back == 0x08 || fore == 0x08 && back == 0x00) continue;
	      mvaddch(y, x * 4, ' ');
	      if ((port[0xc0] & 0x04) == 0 && (port[0xc1] & 0x0c) == 0x08 &&
		  (fore == 0x00 || fore == 0x08 || back == 0x0f)) {
		setcchar(&wch, &dispchar[i], A_REVERSE, 0, NULL);
	      } else {
		setcchar(&wch, &dispchar[i], A_NORMAL, 0, NULL);
	      }
	      mvadd_wch(y, x * 4 + 1, &wch);
	      mvaddch(y, x * 4 + 3, ' ');
	    }
	  }

	} else {
	/* screen mode 4 text character */
	  for (x = 0; x < 40; x++) {
	    memset(color_used, 0, sizeof(color_used));
	    memset(pattern, 0, sizeof(pattern));
	    for (i = 0; i < 10; i++) {
	      for (j = 0; j < 8; j++) {
		color1 = ((vram[x + (y * 10 + i) * 40] >> (7 - j)) & 0x01)
		  | ((vram[x + (y * 10 + i) * 40 + 0x2000] >> (7 - j) << 1) & 0x02);
		pattern[color1][i] += 1 << (7 - j);
		color_used[color1] = 1;
	      }
	    }
	    sum = 0;
	    color1 = -1;
	    color2 = -1;
	    for (i = 0; i < 16; i++) {
	      sum += color_used[i];
	      if (color_used[i]) {
		if (color1 == -1) {
		  color1 = i;
		} else {
		  color2 = i;
		}
	      }
	    }
	    if (sum != 2) continue;
	    for (i = 0; i < 0x100; i++) {
	      for (j = 0; j < 10; j++) {
		if (pattern[color2][j] != cgrom2[i * 16 + j]) break;
	      }
	      if (j < 10) {
		for (j = 0; j < 10; j++) {
		  if (pattern[color1][j] != cgrom2[i * 16 + j]) break;
		}
	      }
	      if (j == 10) {
		setcchar(&wch, &dispchar[i], A_NORMAL, 0, NULL);
		mvadd_wch(y, x * 2, &wch);
		break;
	      }
	    }
	  }

	}

      }
    }
    mode5_old = 1;
    return;
  }	/* mode5 end */

  if (mode5_old) {
    for (y = 0; y < 23; y++) {
      mvprintw(y, 0, "%*s", 80, "");
    }
    mode5_old = 0;
  }

  /* mode 1-4 */
  graphic_mode = 0;
  for (i = 0; i < 0x200; i++) {
    if (vram[i] & 0x80) {
      graphic_mode = 1;
      break;
    }
  }

  if (graphic_mode) {
    /* graphic mode
      (x,y) PC-6001 text coordinates
      (2x,y) terminal coordinates
      (4x,2y) AA framebuffer coordinates
      2*6 P6 mode4 dots in framebuffer pixel
    */

    for (y = 0; y < 16; y++) {
      for (x = 0; x < 32; x++) {
	for (i = 0; i < 4; i++) {
	  weight[0] = 0;
	  weight[1] = 0;
	  for (j = 0; j < 12; j++) {
	    weight[j / 6] +=
	      ((vram[0x200 + x + (y * 12 + j) * 32] >> (6 - i * 2)) & 0x03)
	      * ((vram[x + ((y * 12 + j) % 16) * 32] & 0x02) ? 1 : 2);
	  }
	  bitmap[x * 4 + i][y * 2 + 0] =
	    (weight[0] == 0) ? 0 : (weight[0] + 6) / 6;
	  bitmap[x * 4 + i][y * 2 + 1] =
	    (weight[1] == 0) ? 0 : (weight[1] + 6) / 6;
	}
	for (i = 0; i < 2; i++) {
	  ul = bitmap[x * 4 + i * 2    ][y * 2    ];
	  ur = bitmap[x * 4 + i * 2 + 1][y * 2    ];
	  dl = bitmap[x * 4 + i * 2    ][y * 2 + 1];
	  dr = bitmap[x * 4 + i * 2 + 1][y * 2 + 1];
	  render[x * 2 + i] = table[(ul << 9) + (ur << 6) + (dl << 3) + dr];
	}
      }
      mvprintw(y, 0, "%s", render);

      /* screen mode 4 text character */
      for (x = 0; x < 32; x++) {
	for (i = 0; i < 0x100; i++) {
	  for (j = 0; j < 12; j++) {
	    if (vram[0x200 + y * 384 + j * 32 + x] != cgrom[i * 16 + j]) break;
	  }
	  if (j == 12) {
	    if (i) {
	      setcchar(&wch, &dispchar[i], A_NORMAL, 0, NULL);
	      mvadd_wch(y, x * 2, &wch);
	    }
	    break;
	  }
	  for (j = 0; j < 12; j++) {
	    if (vram[0x200 + y * 384 + j * 32 + x] != (~cgrom[i * 16 + j] & 0xff)) break;
	  }
	  if (j == 12) {
	    if (i) {
	      setcchar(&wch, &dispchar[i], A_REVERSE, 0, NULL);
	      mvadd_wch(y, x * 2, &wch);
	    }
	    break;
	  }
	}
      }

      /* screen mode 3 text character */
      for (x = 0; x < 16; x++) {
	memset(color_used, 0, sizeof(color_used));
	memset(pattern, 0, sizeof(pattern));

	for (i = 0; i < 12; i++) {
	  for (j = 0; j < 4; j++) {
	    color1 = (vram[0x200 + y * 384 + i * 32 + x * 2]
		      >> (6 - j * 2)) & 0x03;
	    color2 = (vram[0x200 + y * 384 + i * 32 + x * 2 + 1]
		      >> (6 - j * 2)) & 0x03;
	    pattern[color1][i] += 1 << (7 - j);
	    pattern[color2][i] += 1 << (3 - j);
	    color_used[color1] = 1;
	    color_used[color2] = 1;
	  }
	}
	if (color_used[0] + color_used[1] + color_used[2] + color_used[3] != 2) continue;
	color1 = color_used[0] ? 0 : color_used[1] ? 1 : 2;
	color2 = color_used[3] ? 3 : color_used[2] ? 2 : 1;
	for (i = 0; i < 0x100; i++) {
	  for (j = 0; j < 12; j++) {
	    if (pattern[color2][j] != cgrom[i * 16 + j]) break;
	  }
	  if (j < 12) {
	    for (j = 0; j < 12; j++) {
	      if (pattern[color1][j] != cgrom[i * 16 + j]) break;
	    }
	  }
	  if (j == 12) {
	    mvaddch(y, x * 4, ' ');
	    setcchar(&wch, &dispchar[i], A_NORMAL, 0, NULL);
	    mvadd_wch(y, x * 4 + 1, &wch);
	    mvaddch(y, x * 4 + 3, ' ');
	    break;
	  }
	}
      }
    }

  } else {
    /* semi-graphic mode
      (x,y) PC-6001 text coordinates
      (2x,y) terminal coordinates
      (4x,2y) AA framebuffer coordinates
      0.5*1.5 P6 mode2 dots in framebuffer pixel
    */
    for (y = 0; y < 16; y++) {
      for (x = 0; x < 32; x++) {
	for (i = 0; i < 2; i++) {
	  weight[0] = ((vram[0x200 + x + y * 32] >> (4 - i)) & 2)
	    + ((vram[0x200 + x + y * 32] >> (3 - i)) & 1);
	  weight[1] = ((vram[0x200 + x + y * 32] >> (3 - i)) & 1)
	    + ((vram[0x200 + x + y * 32] << i) & 2);
	  ul = (weight[0] == 0) ? 0 :
	    (weight[0] * (8 - ((vram[x + y * 32] & 0x02) << 1) - (((vram[0x200 + x + y * 32] & 0xc0) >> 6))) + 6) / 7;
	  ur = ul;
	  dl = (weight[1] == 0) ? 0 :
	    (weight[1] * (8 - ((vram[x + y * 32] & 0x02) << 1) - (((vram[0x200 + x + y * 32] & 0xc0) >> 6))) + 6) / 7;
	  dr = dl;
	  render[x * 2 + i] = table[(ul << 9) + (ur << 6) + (dl << 3) + dr];
	}
      }
      mvprintw(y, 0, "%s", render);
    }

    /* screen mode 1 and 2 text character */
    for (i = 0; i < 0x200; i++) {
      if (vram[i] & 0x40) continue;
      x = i & 0x1f;
      y = (i & 0x1e0) >> 5;
      if (vram[i] & 0x01) {
	setcchar(&wch, &dispchar[vram[0x200 + i]], A_REVERSE, 0, NULL);
      } else {
	setcchar(&wch, &dispchar[vram[0x200 + i]], A_NORMAL, 0, NULL);
      }
      mvadd_wch(y, x * 2, &wch);
    }
  }
}

/*
 * print prompt and get input by readline/editline
 */
char *getinput(int y, const char *prompt)
{
  int i, j;
  int maxy;
  char *input;
  static char *input_prev = NULL;

#ifndef __MINGW32__
  struct termios term;
  tcgetattr(0, &term);
  term.c_lflag |= ECHO;
  tcsetattr(0, TCSANOW, &term);
#endif

  maxy = getmaxy(stdscr);
  setscrreg(19, maxy - 1);
  if (y >= maxy) {
    y = maxy - 1;
  }
  move(y, 0);
  clrtoeol();
  curs_set(1);
  refresh();

  do {
    input = readline(prompt);
    redrawwin(stdscr);
    refresh();
  } while (input == NULL);

#ifndef __MINGW32__
  term.c_lflag &= ~ECHO;
  tcsetattr(0, TCSANOW, &term);
#endif

  curs_set(0);

  maxy = getmaxy(stdscr);
  setscrreg(19, maxy - 1);

  mvprintw(y, 0, "%s%s", prompt, input);
  i = strspn(input, " ");
  for (j = strlen(input) - 1; j > i; j--) {
    if (input[j] != ' ') {
      break;
    }
  }
  memmove(input, input + i, j - i + 1);
  input[j -i + 1] = '\0';
  if (strlen(input) > 0 && (input_prev == NULL || strcmp(input, input_prev))) {
    add_history(input);
    if (input_prev) {
      free(input_prev);
    }
    input_prev = malloc(strlen(input) + 1);
    strcpy(input_prev, input);
  }

  return input;
}

/*
 * open tape file for reading
 */
FILE *read_open(const char *fname)
{
  int i;
  int c;
  FILE *new;

  if (fname == NULL || (new = fopen(fname, "rb")) == NULL) {
    return NULL;
  }

  if (tape_read) {
    fclose(tape_read);
    tape_read = NULL;
  }

  if (is_p6t) {
    free(p6t.autostring);
    if (p6t.nblock) {
      for (i = 0; i < p6t.nblock; i++) {
	free(p6t.data[i]);
      }
      free(p6t.data);
    }
  }

  is_p6t = 0;
  if (strcasecmp(&fname[strlen(fname) - 4], ".p6t")) {
    return new;
  }

  fseek(new, -4, SEEK_END);
  fread(&i, 4, 1, new);
  fseek(new, i, SEEK_SET);
  c = fgetc(new);
  if (c == 'P' && fgetc(new) == '6') {
    p6t.version = fgetc(new);
    p6t.nblock = fgetc(new);
    p6t.autostart = fgetc(new);
    p6t.mode = fgetc(new);
    p6t.pages = fgetc(new);
    fread(&p6t.nauto, 2, 1, new);
    p6t.autostring = malloc(p6t.nauto + 1);
    for (i = 0; i < p6t.nauto; i++) {
      p6t.autostring[i] = fgetc(new);
    }
    p6t.autostring[i] = '\0';
    fread(&p6t.nextend, 2, 1, new);
    for (i = 0; i < p6t.nextend; i++) {
      fgetc(new);
    }
    if (p6t.nblock) {
      p6t.data = malloc(sizeof(p6t.data[0]) * p6t.nblock);
      for (i = 0; i < p6t.nblock; i++) {
	p6t.data[i] = malloc(33);
	fread(p6t.data[i], 1, 33, new);
      }
    }
    fseek(new, 4, SEEK_CUR);
    if (i == p6t.nblock && fgetc(new) == EOF) {
      is_p6t = 1;
      p6t_block = 0;
      p6t_blank = 1;
    }
  }

  fseek(new, 0, SEEK_SET);
  tape_counter = 0;
  return new;
}

/*
 * get data from .p6t file
 */
int getp6t(int *data)
{
  static int idata;
  static int icheck;
  static int is_headerfooter;
  static int checkdata = -1;
  int period;

  if (!relay || !tape_read) {
    return 0;
  }

  //  count_tape += cpu.cycles;

  if (p6t_blank) {
    if (is_p6t) {
      if (p6t_block >= p6t.nblock) {
	count_tape = 0;
	return 0;
      }
      period = CLOCK_4M / 1000
	* (p6t.data[p6t_block][21] + (p6t.data[p6t_block][22] << 8)
	   + p6t.data[p6t_block][23] + (p6t.data[p6t_block][24] << 8));
    } else {
      period = CLOCK_4M * p6t_blanktime;
    }
  } else {
    /* start bit(1) + data(8) + stop bit(3) = 12 */
    if (is_p6t && p6t_block < p6t.nblock) {
      baudrate_read = p6t.data[p6t_block][19] + (p6t.data[p6t_block][20] << 8);
    }
    period = CLOCK_4M * 12 / baudrate_read;
    if (boost > 1) {
      if (ram[0xfa19]) {
	count_tape = 0;
      } else {
	period /= boost;
      }
    }
  }

  if (count_tape <= period || cpu.mcycle > 0) {
    return 0;
  }

  if (p6t_blank) {
    p6t_blank = 0;
    count_tape = 0;
    idata = 0;
  }

  if (is_p6t) {
    idata++;
    if (idata > p6t.data[p6t_block][29] + (p6t.data[p6t_block][30] << 8)
	+ (p6t.data[p6t_block][31] << 16) + (p6t.data[p6t_block][32] << 24)) {
      count_tape = 0;
      p6t_blank = 1;
      p6t_block++;
      if (p6t_block < p6t.nblock) {
	fseek(tape_read,
	      p6t.data[p6t_block][25]
	      + (p6t.data[p6t_block][26] << 8)
	      + (p6t.data[p6t_block][27] << 16)
	      + (p6t.data[p6t_block][28] << 24),
	      SEEK_SET);
      }
      return 0;
    }
  }

  *data = fgetc(tape_read);
  count_tape %= period;

  /* insert blank */
  if (!is_p6t) {
    if (is_headerfooter) {
      if (checkdata == 0xd3 && ++icheck == 16 ||	/* BASIC */
	  checkdata == 0xaf && ++icheck == 11 ||	/* I/O monitor */
	  checkdata == '0' && ++icheck == 17) {		/* intel HEX */
	p6t_blank = 1;
	is_headerfooter = 0;
	icheck = 0;
      }
    } else {
      if (*data != 0xaf && *data != 0xd3 && *data != 0x00 && *data != '0') {
	checkdata = -1;
      } else {
	if (*data != checkdata) {
	  icheck = 0;
	  checkdata = *data;
	}
	switch (*data) {
	case 0xaf:	/* I/O monitoer header */
	  if (++icheck == 5) {
	    is_headerfooter = 1;
	    p6t_blanktime = 3.5;
	  }
	  break;
	case 0xd3:	/* BASIC header */
	  if (++icheck == 10) {
	    is_headerfooter = 1;
	    p6t_blanktime = 3.5;
	    p6t_basic = 1;
	  }
	  break;

	case 0x00:	/* BASIC footer */
	  if (p6t_basic && ++icheck == 10) {
	    p6t_blank = 1;
	    p6t_blanktime = 0.05;
	    p6t_basic = 0;
	  }
	  break;
	case '0':	/* intel HEX footer */
	  if (++icheck == 16) {
	    is_headerfooter = 1;
	    p6t_blanktime = 0.05;
	  }
	  break;
	}
      }
    }
  }

  return 1;
}

/*
 * open disk file for reading
 */
int disk_open(const char *fname, int drive)
{
  FILE *disk_read;

  if (fname == NULL || (disk_read = fopen(fname, "rb")) == NULL) {
    return 0;
  }
  strncpy(diskfile[drive], fname, sizeof(diskfile[drive]) - 1);
  diskfile[drive][sizeof(diskfile[drive]) - 1] = '\0';
  memset(disk[drive], 0xff, sizeof(disk[drive]));
  if (strncasecmp(fname + strlen(fname) - 4, ".d88", 4) == 0) {
    is_d88[drive] = 1;
    /*
    if (disk[drive][0x001b] != 0) {
      return 0;
    }
    */
  } else {
    is_d88[drive] = 0;
  }
  fread(disk[drive], 1, sizeof(disk[drive]), disk_read);
  fclose(disk_read);
  return 1;
}

/*
 * get data pointer from .d88 file
 */
byte *getd88(int drive, int track, int sector)
{
  int i;
  int adr;

  adr = disk[drive][0x0020 + track * 8]
    + (disk[drive][0x0020 + track * 8 + 1] << 8)
    + (disk[drive][0x0020 + track * 8 + 2] << 16)
    + (disk[drive][0x0020 + track * 8 + 3] << 24);
  if (adr == 0 || adr + 16 > sizeof(disk[drive])) {
    return NULL;
  }

  for (i = 0; i < disk[drive][adr + 4] + (disk[drive][adr + 4] << 8); i++) {
    if (adr + 16 + 256 > sizeof(disk[drive])) {
      return NULL;
    }
    if (disk[drive][adr] == track &&
	disk[drive][adr + 1] == 0 &&
	disk[drive][adr + 2] == sector &&
	disk[drive][adr + 3] == 1) {
      return &disk[drive][adr + 16];
    }
    //    adr += disk[drive][adr + 3] * (disk[drive][adr + 14] + (disk[drive][adr + 15] << 8));
    adr += 16 + (128 << disk[drive][adr + 3]);
  }
  return NULL;
}

/*
 * synchronize for 60Hz
 */
void wait60hz(double tick)
{
  struct timespec now, wait;
  static struct timespec old = {0, 0};
  long long tdiff;

  wait.tv_sec = 0;
  wait.tv_nsec = tick / 10 * 1e9;

  while (1) {
    clock_gettime(CLOCK_MONOTONIC, &now);
    tdiff = now.tv_nsec - old.tv_nsec + (now.tv_sec - old.tv_sec) * 1e9;
    if (tdiff > tick * 1e9) {
      break;
    }
    nanosleep(&wait, NULL);
  }
  old.tv_sec = now.tv_sec;
  old.tv_nsec = now.tv_nsec - tdiff % (long)(tick * 1e9);
  if (old.tv_nsec < 0) {
    old.tv_sec--;
    old.tv_nsec += 1e9;
  }
}

/*
 * change VRAM
 */
void chgvram()
{
  int bank;
  word vramtable[] = {0xc000, 0xe000, 0x8000, 0xa000};

  /* !!! page=0-3 !!! */
  if (mode5) {
    vram = &ram[(vramtable[page] << 1) & 0xffff];
  } else if (ram64 || page < 2) {
    vram = &ram[vramtable[page]];
  } else if (!use_beluga) {
    vram = &extram[vramtable[page]];
  } else {
    /* Beluga cartridge */
    bank = port[0x32 + page];	/* 0x34 or 0x35 */
    switch (bank & 0xc0) {
    case 0x00:	/* external ROM */
      vram = &extrom[(bank << 13) & 0x7e000];
      break;
    case 0x40:	/* external RAM */
      vram = &extram[(bank << 13) & 0x1e000];
      break;
    case 0x80:	/* SCC */
      break;
    case 0xc0:	/* not implemented */
      break;
    }
  }
}

/*
 * read ROM file with check
 */
void romread(byte *rom, int size, const char *fname)
{
  if (romread_nocheck(rom, size, fname)) {
    return;
  }

  printf("cannot open %s\n", fname);
  exit(EXIT_FAILURE);
}

/*
 * read ROM file without check
 */
int romread_nocheck(byte *rom, int size, const char *fname)
{
  int i;
  char *fname_case;
  FILE *rom_file;

  rom_file = fopen(fname, "rb");
  if (rom_file) {
    fread(rom, 1, size, rom_file);
    fclose(rom_file);
    return 1;
  }

  fname_case = malloc(strlen(fname) + 1);
  for (i = 0; i < strlen(fname); i++) {
    fname_case[i] = toupper(fname[i]);
  }
  fname_case[i] = '\0';
  rom_file = fopen(fname_case, "rb");
  if (rom_file) {
    fread(rom, 1, size, rom_file);
    fclose(rom_file);
    return 1;
  }

  for (i = 0; i < strlen(fname); i++) {
    fname_case[i] = tolower(fname[i]);
  }
  rom_file = fopen(fname_case, "rb");
  if (rom_file) {
    fread(rom, 1, size, rom_file);
    fclose(rom_file);
    return 1;
  }

  fname_case[0] = toupper(fname[0]);
  rom_file = fopen(fname_case, "rb");
  if (rom_file) {
    fread(rom, 1, size, rom_file);
    fclose(rom_file);
    return 1;
  }

  return 0;
}

/*
 * override function for ambiguous width characters
 */
int wcwidth(wchar_t c)
{
  int i;
  /*
  if (wcschr(dispchar, c)) {
    return 2;
  } else {
    return 1;
  }
  */
  int retval = 1;
  for (i = 0; i < 0x100; i++) {
    if (c == dispchar[i]) {
      retval = 2;
      break;
    }
  }
  return retval;
}
