/* PC-6001/mkII emulator on terminal, monitor mode       */
/*  by AKIKAWA, Hisashi 2015-2025                        */
/* This software is released under 2-clause BSD license. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#ifdef __FreeBSD__
# include <edit/readline/readline.h>
#else
# include <editline/readline.h>
#endif
#include "z80.h"

#if defined __CYGWIN__ || defined __MSYS__
# include <w32api/windows.h>
#endif

#ifdef __MINGW32__
# include "win.h"
  extern WINDOW *stdscr;
#else
# define _XOPEN_SOURCE_EXTENDED
# ifdef __FreeBSD__
#  include <ncurses.h>
# else
#  include <ncursesw/ncurses.h>
# endif
# include <termios.h>
#endif

/* 8255 status */
#define NOBF	0x80	/* PC7 = #output buffer full   = 8049 #INT */
#define INTE1	0x40	/* PC6 = write interrupt enable            */
#define IBF	0x20	/* PC5 = input buffer full     = 8049 T0   */
#define INTE2	0x10	/* PC4 = read interrupt enable             */
#define INTR	0x08	/* PC3 = interrupt request                 */
#define CGSW	0x04	/* cgrom switch                            */
#define BSRQEN	0x02	/* bus request enable                      */
#define PRTSTRB	0x01	/* printer strobe                          */

#define NONE -1

void monitor(void);
static void monitor_refresh(void);
static void assembler(int y, char **input);
static void mon_help(void);
static void mon_quit(void);
static void mon_go(void);
static void mon_step(void);
static void mon_break(void);
static void break_clear(void);
static void mon_dump(void);
static void mon_dram(void);
static void dump_common(char *arg1, int readram);
static void mon_write(void);
static void mon_disasm(void);
static void mon_bank(void);
static void mon_load(void);
static void mon_save(void);
static void mon_in(void);
static void mon_out(void);
static void mon_reg(void);
static void mon_stack(void);
static void mon_di(void);
static void mon_ei(void);
static void mon_serial(void);
static void mon_char(void);
static void write_memory(int address, int value);
static void prtbank(byte **bank, int i);
static int upperhex(int value, int mask);

extern void display(void);
extern void reset_p6(void);
extern char *getinput(int y, const char *prompt);
extern byte readmem(word address, int *wait);
extern void writemem(word address, byte value, int *wait);
extern byte readport(word address);
extern void writeport(word address, byte value);

/*
 * monitor mode main
 */
void monitor()
{
  int i;
  char *input;
  char *command;
  static int y = 19;

  extern long long count_cycles;
  extern int step;
  extern int mode5;

  struct {
    char *command;
    void (*func)(void);
  } mon[] = {
    {"help",	mon_help},	{"?",	mon_help},
    {"quit",	mon_quit},
    {"reset",	reset_p6},
    {"go",	mon_go},	{"g",	mon_go},
    {"step",	mon_step},	{"s",	mon_step},
    {"break",	mon_break},	{"b",	mon_break},
    {"bank",	mon_bank},
    {"dump",	mon_dump},	{"d",	mon_dump},
    {"dram",	mon_dram},	{"dr",	mon_dram},
    {"write",	mon_write},
    {"in",	mon_in},
    {"out",	mon_out},
    {"disasm",	mon_disasm},	{"dis",	mon_disasm},
    {"stack",	mon_stack},
    {"di",	mon_di},
    {"ei",	mon_ei},
    {"register",mon_reg},	{"reg",	mon_reg},
    {"load",	mon_load},
    {"save",	mon_save},
    {"serial",  mon_serial},
    {"char",	mon_char},	{"c", mon_char},

  };

  if (mode5 && y < 23) y = 23;
  
  step = 0;
  do {
    monitor_refresh();
    input = getinput(y, ">");
    printw("\n");

    if (strchr(input, ':')) {
      assembler(y, &input);
    } else {
      command = strtok(input, " \t");
      if (command) {
	for (i = 0; i < sizeof(mon) / sizeof(mon[0]); i++) {
	  if (strcasecmp(command, mon[i].command) == 0) {
	    mon[i].func();
	    break;
	  }
	}
      }
    }

    getyx(stdscr, y, i);	/* i=dummy */
    free(input);
  } while (step == 0);

  count_cycles = 0;

  monitor_refresh();
}

/*
 * refresh monitor display
 */
void monitor_refresh()
{
  int i;
  int x, y;
  int bytes;
  char disasm[80];

  extern long long count_cycles;
  extern z80 cpu;
  extern int mode5;
  extern int tape_counter;

  getyx(stdscr, y, x);
  display();

  bytes = z80_dasm(disasm, cpu.pc);
  for (i = 0; i <  bytes; i++) {
    readmem(cpu.pc + i, NULL);
  }
  
  move(mode5 ? 20 : 16, 0);
  for (i = 0; i < 8; i++) {
    addch((cpu.f & (0x80 >> i)) ? "SZYHXPNC"[i] : '-');
  }
  printw(" IFF:%02X   I  :%02X   R  :%02X    %-13s  Cycles:%lld\n",
	 cpu.iff1, cpu.i, cpu.r, disasm, count_cycles);
  mvprintw(mode5 ? 20 : 16, 73, "%6d", tape_counter);
  mvprintw(mode5 ? 21 : 17, 0,
	   "AF :%02X%02X BC :%02X%02X DE :%02X%02X "
	   "HL :%02X%02X  IX :%02X%02X IY :%02X%02X\n",
	   cpu.a, cpu.f, cpu.b, cpu.c, cpu.d, cpu.e,
	   cpu.h, cpu.l, cpu.ixh, cpu.ixl, cpu.iyh, cpu.iyl);
  mvprintw(mode5 ? 22 : 18, 0,
	   "AF':%02X%02X BC':%02X%02X DE':%02X%02X "
	   "HL':%02X%02X  PC :%04X SP :%04X\n",
	   cpu.alt_a, cpu.alt_f, cpu.alt_b, cpu.alt_c, cpu.alt_d, cpu.alt_e,
	   cpu.alt_h, cpu.alt_l, cpu.pc, cpu.sp);
  move(y, x);
  redrawwin(stdscr);
  refresh();
}

/*
 * assembler
 */
void assembler(int y, char **input)
{
  int i, j;
  int address_start;
  int bytes;
  int data[16];
  byte asmble[4];
  char *code;
  char prompt[10];

  do {
    monitor_refresh();
    move(y, 27);
    if (strchr(*input, ':')) {
      sscanf(*input, "%x:", &address_start);
      memmove(*input, strchr(*input, ':') + 1, strlen(strchr(*input, ':')));
    }
    code = *input + strspn(*input, " ");
    if (strcasecmp(code, "n") == 0) {
      address_start = (address_start + 1) & 0xffff;
    } else if (strcasecmp(code, "p") == 0) {
      address_start = (address_start - 1) & 0xffff;
    } else if (sscanf(code, "db %x%n", data, &j) == 1
	       || sscanf(code, "DB %x%n", data, &j) == 1) {
      for (i = j; ; i += j) {
	write_memory(address_start, upperhex(data[0], 0xff));
	address_start = (address_start + 1) & 0xffff;
	printw("%02X ", data[0]);
	if (sscanf(code + i, " ,%x%n", data, &j) != 1
	    && sscanf(code + i, "h ,%x%n", data, &j) != 1
	    && sscanf(code + i, "H ,%x%n", data, &j) != 1) {
	  break;
	}
      }
    } else {
      bytes = z80_asm(code, asmble, address_start);
      for (i = 0; i < bytes; i++) {
	write_memory(address_start, asmble[i]);
	printw("%02X ", asmble[i]);
	address_start = (address_start + 1) & 0xffff;
      }
    }
    printw("\n");

    address_start = upperhex(address_start, 0xffff);
    getyx(stdscr, y, i);			/* i=dummy */
    sprintf(prompt, "%04X: ", address_start);
    free(*input);
    *input = getinput(y, prompt);

  } while (sscanf(*input, "%*s") != EOF && strcasecmp(*input, "quit"));
  printw("\n");
}

/*
 * help command
 */
void mon_help()
{
  printw("command:\n");
  printw(" dump, dram, char, write, disasm, bank, in, out, load, save,\n"
	 " serial, stack, reg, di, ei, break, step, go, reset, quit, help\n");
}

/*
 * quit command
 */
void mon_quit()
{
  endwin();
  exit(EXIT_SUCCESS);
}

/*
 * go command
 */
void mon_go()
{
  extern int step;
  step = NONE;
}

/*
 * step command
 */
void mon_step()
{
  extern int step;
  step = 1;
}

/*
 * break command
 */
void mon_break()
{
  int address;
  char *arg1, *arg2, *arg3;

  extern int break_address;
  extern int break_read;
  extern int break_write;
  extern int break_in;
  extern int break_out;
  extern int break_vector;
  extern int until_break;

  arg1 = strtok(NULL, " \t");
  arg2 = strtok(NULL, " \t");
  arg3 = strtok(NULL, " \t");

  if (arg1 && strcasecmp(arg1, "clear") == 0) {
    printw("clear break point\n");
    break_clear();
  } else if (arg1 && strcasecmp(arg1, "read") == 0
	     && arg2 && (sscanf(arg2, "0x%x", &address)  == 1
			 || sscanf(arg2, "%x", &address)  == 1)) {
    break_clear();
    break_read = upperhex(address, 0xffff);
    if (arg3) {
      sscanf(arg3, "%d", &until_break);
    }
  } else if (arg1 && strcasecmp(arg1, "write") == 0
	     && arg2 && (sscanf(arg2, "0x%x", &address)  == 1
			 || sscanf(arg2, "%x", &address)  == 1)) {
    break_clear();
    break_write = upperhex(address, 0xffff);
    if (arg3) {
      sscanf(arg3, "%d", &until_break);
    }
  } else if (arg1 && strcasecmp(arg1, "in") == 0
	     && arg2 && (sscanf(arg2, "0x%x", &address)  == 1
			 || sscanf(arg2, "%x", &address)  == 1)) {
    break_clear();
    break_in = upperhex(address, 0xff);
    if (arg3) {
      sscanf(arg3, "%d", &until_break);
    }
  } else if (arg1 && strcasecmp(arg1, "out") == 0
	     && arg2 && (sscanf(arg2, "0x%x", &address)  == 1
			 || sscanf(arg2, "%x", &address)  == 1)) {
    break_clear();
    break_out = upperhex(address, 0xff);
    if (arg3) {
      sscanf(arg3, "%d", &until_break);
    }
  } else if (arg1 && strcasecmp(arg1, "intr") == 0
	     && arg2 && (sscanf(arg2, "0x%x", &address)  == 1
			 || sscanf(arg2, "%x", &address)  == 1)) {
    break_clear();
    break_vector = upperhex(address, 0xff);
    if (arg3) {
      sscanf(arg3, "%d", &until_break);
    }
  } else if (arg1 && (sscanf(arg1, "0x%x", &address) == 1
		      || sscanf(arg1, "%x", &address) == 1)) {
    break_clear();
    break_address = upperhex(address, 0xffff);
    if (arg2) {
      sscanf(arg2, "%d", &until_break);
    }
  } else {
    printw("break: set break point\n");
    printw("usage: b[reak] <address> [<count>]\n");
    printw("       b[reak] read <address> [<count>]\n");
    printw("       b[reak] write <address> [<count>]\n");
    printw("       b[reak] intr <vector> [<count>]\n");
    printw("       b[reak] clear\n");
  }
  if (until_break <= 0) {
    until_break = 1;
  }
  if (break_address != NONE) {
    printw("break point = address 0x%04X (%d counts)\n",
	   break_address, until_break);
  } else if (break_read != NONE) {
    printw("break point = read from memory 0x%04X (%d counts)\n",
	   break_read, until_break);
  } else if (break_write != NONE) {
    printw("break point = write to memory 0x%04X (%d counts)\n",
	   break_write, until_break);
  } else if (break_in != NONE) {
    printw("break point = input from port 0x%02X (%d counts)\n",
	   break_in, until_break);
  } else if (break_out != NONE) {
    printw("break point = output to port 0x%02X (%d counts)\n",
	   break_out, until_break);
  } else if (break_vector != NONE) {
    printw("break point = interrupt vector 0x%02X (%d counts)\n",
	   break_vector, until_break);
  } else {
    printw("no break point\n");
  }
}

/*
 * clear break point
 */
void break_clear(void)
{
  extern int break_address;
  extern int break_read;
  extern int break_write;
  extern int break_in;
  extern int break_out;
  extern int break_vector;
  extern int count_break;
  extern int until_break;

  break_address = NONE;
  break_read = NONE;
  break_write = NONE;
  break_in = NONE;
  break_out = NONE;
  break_vector = NONE;
  count_break = 0;
  until_break = 1;
}

/*
 * dump command
 */
void mon_dump()
{
  char *arg1;

  arg1 = strtok(NULL, " \t");

  if (arg1) {
    dump_common(arg1, 0);
  } else {
    printw("dump: dump memory\n");
    printw("usage: d[ump] <start address> [<end address>]\n");
  }
}

/*
 * dump RAM command
 */
void mon_dram()
{
  char *arg1;

  arg1 = strtok(NULL, " \t");

  if (arg1) {
    dump_common(arg1, 1);
  } else {
    printw("dram: dump RAM\n");
    printw("usage: dr[am] <start address> [<end address>]\n");
  }
}

static void dump_common(char *arg1, int readram)
{
  int i, j;
  int address_start, address_end;
  int data[16];
  char *arg2;
  extern int break_memory;
  extern int use_extram;
  extern int ram64;
  extern byte ram[];
  extern byte extram[];

  arg2 = strtok(NULL, " \t");

  if (sscanf(arg1, "0x%x", &address_start) != 1) {
    sscanf(arg1, "%x", &address_start);
  }
  address_start = upperhex(address_start, 0xffff);
  if (arg2) {
    if (sscanf(arg2, "0x%x", &address_end) != 1) {
      sscanf(arg2, "%x", &address_end);
    }
    address_end = upperhex(address_end, 0xffff);
  } else {
    address_end = address_start + 0x0f;
  }
  printw("      +0 +1 +2 +3 +4 +5 +6 +7 +8 +9 +A +B +C +D +E +F\n");
  for (i = address_start & 0xfff0; i <= address_end; i += 0x10) {
    printw("%04X:", i & 0xffff);
    for (j = i; j < i + 0x10; j++) {
      if (j >= address_start && j <= address_end) {
	if (readram) {
	  if (use_extram && !ram64 && 0x8000 <= j && j < 0xc000) {
	    data[j & 0xf] = extram[j & 0xffff];
	  } else {
	    data[j & 0xf] = ram[j & 0xffff];
	  }
	} else {
	  data[j & 0xf] = readmem(j, NULL);
	}
	printw(" %02X", data[j & 0xf]);
      } else {
	data[j & 0xf] = ' ';
	printw("   ");
      }
      break_memory = 0;
    }
    printw("  ");
    for (j = i; j < i + 0x10; j++) {
      if (0x20 <= data[j & 0xf] && data[j & 0xf] <= 0x7e) {
	printw("%c", data[j & 0xf]);
      } else {
	printw(".");
      }
    }
    printw("\n");
  }
}

/*
 * write command
 */
void mon_write()
{
  int address;
  int value;
  char *arg1, *arg2;

  arg1 = strtok(NULL, " \t");
  arg2 = strtok(NULL, " \t");

  if (arg2
      && (sscanf(arg1, "0x%x", &address) == 1
	  || sscanf(arg1, "%x", &address) == 1)
      && (sscanf(arg2, "0x%x", &value) == 1
	  || sscanf(arg2, "%x", &value) == 1)
      && 0 <= address && address <= 0xffff
      && 0 <= value && value <= 0xff) {
    write_memory(address, value);
    printw("memory (0x%04x) <= 0x%02x\n", address, value);

    while (arg2 = strtok(NULL, " \t")) {
      if ((sscanf(arg2, "0x%x", &value) == 0
	   && sscanf(arg2, "%x", &value) == 0)
	  || value < 0 || 0xff < value) break;
      write_memory(++address, value);
      printw("memory (0x%04x) <= 0x%02x\n", address, value);
    }
  } else {
    printw("write: write data to RAM\n");
    printw("usage: write <address> <value> [<value>...]\n");
  }
}

/*
 * disasm command
 */
void mon_disasm()
{
  int i, j;
  int bytes;
  static int address_start, address_end;
  char *arg1, *arg2;
  char disasm[80];
  extern int break_memory;
  extern z80 cpu;

  arg1 = strtok(NULL, " \t");
  arg2 = strtok(NULL, " \t");

  if (arg1 && sscanf(arg1, "0x%x", &address_start) != 1) {
    sscanf(arg1, "%x", &address_start);
  }
  address_start = upperhex(address_start, 0xffff);
  address_end = address_start + 0x07;
  if (arg2 && sscanf(arg2, "0x%x", &address_end) != 1) {
    sscanf(arg2, "%x", &address_end) ;
  }

  for (i = address_start;
       (address_start <= address_end && i <= address_end)
	 || (address_start > address_end && i < address_end + 0x10000);) {
    bytes = z80_dasm(disasm, i & 0xffff);
    printw("%04X: %s", i & 0xffff, disasm);
    for (j = 0; j < 16 - strlen(disasm); j++){
      addch(' ');
    }
    for (j = 0; j < bytes; j++) {
      printw(" %02X", readmem(i, NULL));
      i++;
    }
    break_memory = 0;
    printw("\n");
  }
  address_start = i;
}

/*
 * load command
 */
void mon_load()
{
  int i;
  int c;
  int address;
  char *arg1, *arg2;
  char *fname;
  FILE *binary_file;

  arg1 = strtok(NULL, " \t");
  arg2 = strtok(NULL, " \t");

  if (arg2
      && (sscanf(arg2, "0x%x", &address) == 1
	  || sscanf(arg2, "%x", &address) == 1)
      && 0 <= address && address <= 0xffff) {
    fname = tilde_expand(arg1);
    if ((binary_file = fopen(fname, "rb")) == NULL) {
      printw("cannot open %s\n", fname);
    } else {
      for (i = address;
	   (c = fgetc(binary_file)) != EOF;
	   i = (i + 1) & 0xffff) {
	write_memory(i, c);
      }
      fclose(binary_file);
      printw("loaded %s to 0x%04x-0x%04x\n", fname, address, (i - 1) & 0xffff);
    }
    free(fname);
  } else {
    printw("load: load file to memory\n");
    printw("usage: load <filename> <address>\n");
  }
}

/*
 * save command
 */
void mon_save()
{
  int i;
  int address_start, address_end;
  char *arg1, *arg2, *arg3;
  char *fname;
  FILE *binary_file;

  arg1 = strtok(NULL, " \t");
  arg2 = strtok(NULL, " \t");
  arg3 = strtok(NULL, " \t");

  if (arg3
      && (sscanf(arg2, "0x%x", &address_start) == 1
	  || sscanf(arg2, "%x", &address_start) == 1)
      && (sscanf(arg3, "0x%x", &address_end) == 1
	  || sscanf(arg3, "%x", &address_end) == 1)
      && 0 <= address_start && address_start <= 0xffff
      && 0 <= address_end && address_end <= 0xffff
      && address_start <= address_end) {
    fname = tilde_expand(arg1);
    if ((binary_file = fopen(fname, "wb")) == NULL) {
      printw("cannot open %s\n", fname);
    } else {
      for (i = 0; i < ((address_end - address_start) & 0xffff) + 1; i++) {
	fputc(readmem((address_start + i) & 0xffff, NULL), binary_file);
      }
      fclose(binary_file);
      printw("saved 0x%04x-0x%04x into %s\n",
	     address_start, address_end, fname);
    }
    free(fname);
  } else {
    printw("save: save memory to file\n");
    printw("usage: save <filename> <start-address> <end-address>\n");
  }
}

/*
 * in command
 */
void mon_in()
{
  int address;
  char *arg1;

  arg1 = strtok(NULL, " \t");

  if (arg1
      && (sscanf(arg1, "0x%x", &address) == 1
	  || sscanf(arg1, "%x", &address) == 1)
      && 0 <= address && address <= 0xff) {
    printw("port (0x%02x) = 0x%02x\n", address, readport(address));
  } else {
    printw("in: print I/O port data\n");
    printw("usage: in <port-address>\n");
  }
}

/*
 * out command
 */
void mon_out()
{
  int address;
  int value;
  char *arg1, *arg2;

  arg1 = strtok(NULL, " \t");
  arg2 = strtok(NULL, " \t");

  if (arg2
      && (sscanf(arg1, "0x%x", &address) == 1
	  || sscanf(arg1, "%x", &address) == 1)
      && (sscanf(arg2, "0x%x", &value) == 1
	  || sscanf(arg2, "%x", &value) == 1)
      && 0 <= address && address <= 0xff
      && 0 <= value && value <= 0xff) {
    writeport(address, value);
    printw("port (0x%02x) <= 0x%02x\n", address, value);
  } else {
    printw("out: output I/O port data\n");
    printw("usage: out <port-address> <value>\n");
  }
}

/*
 * reg command
 */
void mon_reg()
{
  int i;
  int value;
  int is_reg8 = 0, is_reg16 = 0;
  char *arg1, *arg2;
  extern z80 cpu;

  enum {
    REG_A = 1, REG_F, REG_B, REG_C,
    REG_D, REG_E, REG_H, REG_L,
    REG_ALT_A, REG_ALT_F, REG_ALT_B, REG_ALT_C,
    REG_ALT_D, REG_ALT_E, REG_ALT_H, REG_ALT_L,
    REG_I, REG_R, REG_IFF
  };
  enum {
    REG_AF = 1, REG_BC, REG_DE, REG_HL,
    REG_ALT_AF, REG_ALT_BC, REG_ALT_DE, REG_ALT_HL,
    REG_IX, REG_IY, REG_PC, REG_SP
  };

  struct reglist {
    int num;
    char name[4];
  };
  struct reglist reg8[] = {
    {REG_A, "A"}, {REG_F, "F"},
    {REG_B, "B"}, {REG_C, "C"},
    {REG_D, "D"}, {REG_E, "E"},
    {REG_H, "H"}, {REG_L, "L"},
    {REG_ALT_A, "A'"}, {REG_ALT_F, "F'"},
    {REG_ALT_B, "B'"}, {REG_ALT_C, "C'"},
    {REG_ALT_D, "D'"}, {REG_ALT_E, "E'"},
    {REG_ALT_H, "H'"}, {REG_ALT_L, "L'"},
    {REG_I, "I"}, {REG_R, "R"}, {REG_IFF, "IFF"}
  };
  struct reglist reg16[] = {
    {REG_AF, "AF"}, {REG_BC, "BC"},
    {REG_DE, "DE"}, {REG_HL, "HL"},
    {REG_ALT_AF, "AF'"}, {REG_ALT_BC, "BC'"},
    {REG_ALT_DE, "DE'"}, {REG_ALT_HL, "HL'"},
    {REG_IX, "IX"}, {REG_IY, "IY"},
    {REG_PC, "PC"}, {REG_SP, "SP"}
  };

  arg1 = strtok(NULL, " \t");
  arg2 = strtok(NULL, " \t");

  if (arg1) {
    for (i = 0; i < strlen(arg1); i++) {
      arg1[i] = toupper((int)arg1[i]);
    }
  }
  
  if (arg2
      && (sscanf(arg2, "0x%x", &value) == 1
	  || sscanf(arg2, "%x", &value) == 1)) {

    for (i = 0; i < sizeof(reg8) / sizeof(reg8[0]); i++) {
      if (strcmp(arg1, reg8[i].name) == 0) {
	value = upperhex(value, 0xff);
	is_reg8 = reg8[i].num;
	break;
      }
    }
    switch (is_reg8) {
    case REG_A: cpu.a = value; break;
    case REG_F: cpu.f = value; break;
    case REG_B: cpu.b = value; break;
    case REG_C: cpu.c = value; break;
    case REG_D: cpu.d = value; break;
    case REG_E: cpu.e = value; break;
    case REG_H: cpu.h = value; break;
    case REG_L: cpu.l = value; break;
    case REG_ALT_A: cpu.alt_a = value; break;
    case REG_ALT_F: cpu.alt_f = value; break;
    case REG_ALT_B: cpu.alt_b = value; break;
    case REG_ALT_C: cpu.alt_c = value; break;
    case REG_ALT_D: cpu.alt_d = value; break;
    case REG_ALT_E: cpu.alt_e = value; break;
    case REG_ALT_H: cpu.alt_h = value; break;
    case REG_ALT_L: cpu.alt_l = value; break;
    case REG_I: cpu.i = value; break;
    case REG_R: cpu.r = value; break;
    case REG_IFF: value = value ? 1 : 0; cpu.iff1 = value; break;
    default: break;
    }

    for (i = 0; i < sizeof(reg16) / sizeof(reg16[0]); i++) {
      if (strcmp(arg1, reg16[i].name) == 0) {
	value = upperhex(value, 0xffff);
	is_reg16 = reg16[i].num;
	break;
      }
    }
    switch (is_reg16) {
    case REG_AF: cpu.a = value >> 8; cpu.f = value & 0xff; break;
    case REG_BC: cpu.b = value >> 8; cpu.c = value & 0xff; break;
    case REG_DE: cpu.d = value >> 8; cpu.e = value & 0xff; break;
    case REG_HL: cpu.h = value >> 8; cpu.l = value & 0xff; break;
    case REG_ALT_AF: cpu.alt_a = value >> 8; cpu.alt_f = value & 0xff; break;
    case REG_ALT_BC: cpu.alt_b = value >> 8; cpu.alt_c = value & 0xff; break;
    case REG_ALT_DE: cpu.alt_d = value >> 8; cpu.alt_e = value & 0xff; break;
    case REG_ALT_HL: cpu.alt_h = value >> 8; cpu.alt_l = value & 0xff; break;
    case REG_IX: cpu.ixh = value >> 8; cpu.ixl = value & 0xff; break;
    case REG_IY: cpu.iyh = value >> 8; cpu.iyl = value & 0xff; break;
    case REG_PC: cpu.pc = value; break;
    case REG_SP: cpu.sp = value; break;
    default: break;
    }
  }

  if (is_reg8) {
    printw("register %s <= 0x%02x\n", arg1, value);
  } else if (is_reg16) {
    printw("register %s <= 0x%04x\n", arg1, value);
  } else {
    printw("reg: set Z80 register\n");
    printw("usage: reg[ister] <register-name> <value>\n");
  }
}

/*
 * stack command
 */
void mon_stack()
{
  int i;
  extern z80 cpu;
  for (i = 0; i < 16; i += 2) {
    printw("+%02X:%04X  ", i,
	   readmem(cpu.sp + i, NULL)  + (readmem(cpu.sp + i + 1, NULL) << 8));
  }
  printw("\n");
}

/*
 * di command
 */
void mon_di()
{
  extern z80 cpu;
  cpu.iff1 = 0;
  cpu.iff2 = 0;
  printw("disable interrupt\n");
}

/*
 * ei command
 */
void mon_ei()
{
  extern z80 cpu;
  cpu.iff1 = 1;
  cpu.iff2 = 1;
  printw("enable interrupt\n");
}

/*
 * bank command
 */
void mon_bank()
{
  extern int mk2;
  extern byte *mem_r[], *mem_win[], *mem_wex[];
  extern byte port[];
  extern byte port_c;
  int i;

  for (i = 0; i < 8; i++) {
    printw("  %04X:", i * 0x2000);
    if ((~port_c & CGSW) &&
	(!mk2 && i == 3 ||
	 mk2 && (port[0xf8] & 0x40) &&
	 !((i ^ port[0xf8] & 0x07) & (~port[0xf8] >> 3)))) {
      printw("CGROM   ");
    } else {
      prtbank(mem_r, i);
    }
    printw("/ ");
    prtbank(mem_win, i);
    printw("| ");
    prtbank(mem_wex, i);
    if (i % 2) {
      printw("\n");
    }
  }
}

/*
 * serial command
 */
void mon_serial()
{
  extern int rxrdy;
  extern byte serial_data;
  int value;
  char *arg1;

  arg1 = strtok(NULL, " \t");
  if (arg1
      && (sscanf(arg1, "0x%x", &value) == 1
	  || sscanf(arg1, "%x", &value) == 1)
      && 0 <= value && value <= 0xff) {
    serial_data = value;
    rxrdy = 0;
    printw("receive data: 0x%02x\n", serial_data);
  } else {
    printw("serial: receive data by RS-232C\n");
    printw("usage: serial <data>\n");
  }
}

/*
 * char command
 */
void mon_char()
{
  int i, j;
  int address_start, address_end;
  int data[16];
  char *arg1, *arg2;
  cchar_t wch;
  extern wchar_t dispchar[];

  arg1 = strtok(NULL, " \t");
  if (!arg1) {
    printw("char: character dump\n");
    printw("usage: c[har] <start address> [<end address>]\n");
    return;
  }

  arg2 = strtok(NULL, " \t");
  if (sscanf(arg1, "0x%x", &address_start) != 1) {
    sscanf(arg1, "%x", &address_start);
  }
  address_start = upperhex(address_start, 0xffff);
  if (arg2) {
    if (sscanf(arg2, "0x%x", &address_end) != 1) {
      sscanf(arg2, "%x", &address_end);
    }
    address_end = upperhex(address_end, 0xffff);
  } else {
    address_end = address_start + 0x0f;
  }
  printw("      +0+1+2+3+4+5+6+7+8+9+A+B+C+D+E+F\n");
  for (i = address_start & 0xfff0; i <= address_end; i += 0x10) {
    printw("%04X: ", i & 0xffff);
    for (j = i; j < i + 0x10; j++) {
      data[j & 0xf] = readmem(j, NULL);
      if (j >= address_start && j <= address_end) {
	printw("%02X", data[j & 0xf]);
      } else {
	printw("  ");
      }
    }
    printw("\n      ");
    for (j = i; j < i + 0x10; j++) {
      if (j >= address_start && j <= address_end) {
	setcchar(&wch, &dispchar[data[j & 0xf]], A_NORMAL, 0, NULL);
	add_wch(&wch);
      } else {
	printw("  ");
      }
    }
    printw("\n");
  }
}

/*
 * write to memory (current bank or internal RAM)
 */
void write_memory(int address, int value)
{
  extern int ram64;
  extern byte ram[];

  address &= 0xffff;
  value &= 0xff;

  writemem(address, value, NULL);
  if (ram64 && readmem(address, NULL) != value) {
    ram[address] = value;
  }
}

/*
 * print bank
 */
static void prtbank(byte **bank, int i)
{
  extern int use_beluga;
  extern byte rom[], ram[], extrom[], extram[], voicerom[], kanjirom[];
  extern byte port[];

  if (bank[i] == &rom[i * 0x2000 % 0x8000]) {
    printw("BASICROM");
  } else if (bank[i] == &voicerom[i * 0x2000 % 0x4000]) {
    printw("VOICEROM");
  } else if (bank[i] == &kanjirom[i * 0x2000 % 0x4000] ||
	     bank[i] == &kanjirom[i * 0x2000 % 0x4000 + 0x4000]) {
    printw("KANJIROM");
  } else if (bank[i] == &ram[i * 0x2000]) {
    printw("INTRAM  ");
  } else if (use_beluga) {
    if (bank[i] == &extrom[port[0x30 + i] * 0x2000]) {
      printw("EXTROM%02X", port[0x30 + i] & 0x1f);
    } else if (bank[i] == &extram[(port[0x30 + i] & 0x0f) * 0x2000]) {
      printw("EXTRAM%d ", port[0x30 + i] & 0x0f);
    } else {
      printw("XXX     ");
    }
  } else {
    if (bank[i] == extrom) {
      printw("EXTROM0 ");
    } else if (bank[i] == &extrom[0x2000]) {
      printw("EXTROM1 ");
    } else if (bank[i] == &extram[i * 0x2000]) {
      printw("EXTRAM  ");
    } else {
      printw("XXX     ");
    }
  }
}

/*
 * get the upper hex digits
 */

int upperhex(int value, int mask)
{
  if (value < 0) {
    value = -value;
    while (value > mask) {
      value >>= 4;
    }
    value = -value;
  }
  while (value > mask) {
    value >>= 4;
  }

  return value;
}
